define('confluence/cjc/common/jira-issues-helper', [
    'jquery',
    'ajs',
    'underscore'
],
function(
    $,
    AJS,
    _
) {
    'use strict';

    var JiraIssueCache = {
        put : function(key, value) {
            sessionStorage.setItem(key, JSON.stringify(value));
        },

        get : function(key) {
            var value = sessionStorage.getItem(key);
            if (value) {
                try {
                    var obj = JSON.parse(value);
                } catch (e) {
                    return null;
                }
                return obj;
            }
            return null;
        },

        containKey : function(key) {
            return sessionStorage.getItem(key) != null;
        }
    };

    var JIRA_SCHEMA_EPIC_KEY = "com.pyxis.greenhopper.jira:gh-epic-link";
    var JIRA_GREENHOPPER_EPIC_PROPS_REST_URL = "/rest/greenhopper/1.0/api/epicproperties";
    var JIRA_EPIC_ISSUE_TYPE_CACHED_KEY_PREFIX = "JIRA_EPIC_ISSUE_TYPE_";
    var CREATE_JIRA_ISSUE_FIELD_CACHED_KEY_PREFIX = "CREATE_JIRA_ISSUE_FIELD_";

    // XHR object, hold ajax get fields request to call abort when needed
    var xhrField, xhrEpicIssueType;

    /**
     * Discovering FIELD metadata (server>project>issuetype>(fields))
     * @param paramObj: request param object with serverId, projectId, issueTypeId params
     * @param onSuccessCallback: callback function receive list fields json object param
     * @param onErrorCallback: callback function receive error message param
     */
    var discoverIssueTypeField = function(paramObj, onErrorCallback) {
        // If xhrProject is existed, this mean previous ajax hadn't completed yet, call abort() to cancel it.
        if (xhrField && xhrField.readyState !== 4 && xhrField.abort) {
            xhrField.abort();
        }
        if (!paramObj.serverId || !paramObj.projectId || !paramObj.issueTypeId) {
            AJS.logError(AJS.format('confluence-jira-content:discoverIssueTypeField - Error with parameters: serverId={0}, projectId={1}, issueTypeId={2}',
                    paramObj.serverId, paramObj.projectId, paramObj.issueTypeId));
            // call error when get field params are unexpected.
            onErrorCallback && onErrorCallback(AJS.I18n.getText("createjiracontent.dialog.create.issue.discover.fields.error.request.param"));
            return;
        }

        xhrField = AJS.$.ajax({
            type : 'GET',
            timeout: 0,
            url : Confluence.getContextPath() + '/rest/jira-integration/1.0/servers/' + paramObj.serverId
            + '/projects/' + paramObj.projectId + '/issue-types/' + paramObj.issueTypeId + '/fields-meta'
        }).then(function(issueFieldData) {
            if (!issueFieldData) {
                AJS.logError("confluence-jira-content:discoverIssueTypeField - Data discovering error! Unexpected data return.");
                // call error when received unexpected data.
                onErrorCallback && onErrorCallback(AJS.I18n.getText("createjiracontent.dialog.create.issue.discover.fields.error.data.return"));
                return;
            }
            return issueFieldData;
        });

        return xhrField;
    };

    /**
     *
     */
    var resolveEpicIssueType = function(serverId, onSuccess, onError) {
        if (!serverId) {
            AJS.logError(AJS.format("confluence-jira-content:resolveEpicIssueType - Error with parameters: serverId={0}", serverId));
            return;
        }
        var epicIssueTypeCacheKey = JIRA_EPIC_ISSUE_TYPE_CACHED_KEY_PREFIX + serverId;
        if (JiraIssueCache.containKey(epicIssueTypeCacheKey)) {
            onSuccess(JiraIssueCache.get(epicIssueTypeCacheKey));
            return;
        }

        // If xhrEpicIssueType is existed, this mean previous ajax hadn't completed yet, call abort() to cancel it.
        if (xhrEpicIssueType && xhrEpicIssueType.readyState !== 4 && xhrEpicIssueType.abort) {
            xhrEpicIssueType.abort();
        }

        xhrEpicIssueType = window.AppLinks.makeRequest({
            appId : serverId,
            type : "GET",
            url : JIRA_GREENHOPPER_EPIC_PROPS_REST_URL,
            dataType : "json",
            success : function(epicProperties) {
                if (!epicProperties || !epicProperties.epicTypeId) {
                    AJS.logError("confluence-jira-content:resolveEpicIssueType - Data discovering error! Unexpected data return.");
                    return;
                }
                var epicIssueTypeId = epicProperties.epicTypeId;
                onSuccess(epicIssueTypeId);
                JiraIssueCache.put(epicIssueTypeCacheKey, epicIssueTypeId); // Cached storing
            },
            error : function(xhr) {
                onError(xhr.status);
                AJS.logError("confluence-jira-content:resolveEpicIssueType - Error with status=" + xhr.status);
            }
        });
    };

    /**
     * Get create issue fields depend on project and issue type
     *
     * @param paramObj: request param object with serverId, projectId, issueTypeId params
     * @param onSuccessCallback: callback function receive list fields json object param
     * @param onErrorCallback: callback function receive error message param
     */
    var getCreateIssueMeta = function(paramObj, onSuccessCallback, onErrorCallback) {
        var fieldDataKey = buildFieldDataKey(paramObj);
        // get field data in cached
        if (JiraIssueCache.containKey(fieldDataKey)) {
            processCacheData(fieldDataKey, onSuccessCallback, onErrorCallback);
            return;
        }

        discoverIssueTypeField(paramObj, onErrorCallback).done(function(fields) {
            if (fields) {
                var createMeta = AJS.$.extend({}, detectEpicFieldExistence(fields), extractAdditionalRequiredFields(fields, paramObj.defaultRequiredFields));
                // cache create meta data
                JiraIssueCache.put(fieldDataKey, createMeta);
                onSuccessCallback(createMeta);
            }
        }).fail(function(xhr) {
            AJS.logError("confluence-jira-content:discoverIssueTypeField - Error with status=" + xhr.status);
            onErrorCallback && onErrorCallback(AJS.I18n.getText("createjiracontent.dialog.create.issue.discover.fields.error.request.response", xhr.status));
        });
    };

    var processCacheData = function(cacheKey, onSuccessCallback, onErrorCallback) {
        var cacheData = JiraIssueCache.get(cacheKey);
        // has some problem with cache when cache data is null, process error for it
        if(cacheData) {
            onSuccessCallback(cacheData);
        }
        else {
            onErrorCallback
            && onErrorCallback(AJS.I18n
                    .getText("createjiracontent.dialog.create.issue.discover.fields.error.data.return"));
        }
    };

    /**
     * Detect Epic customField
     *
     * @param fields : create meta fields get from Jira
     */
    const detectEpicFieldExistence = function(fields) {
        const epicField = fields.filter(function(field) {
            return field.schema.custom === JIRA_SCHEMA_EPIC_KEY;
        });

        return !_.isEmpty(epicField) ? {epicField: epicField[0].fieldId} : {};
    };

    /**
     * Detect additional required fields needed for issue creation
     *
     * @param fields : create meta fields get from Jira
     * @param defaultRequiredFields : default required fields that we already acknowledged
     */
    const extractAdditionalRequiredFields = function(fields, defaultRequiredFields) {
        let descriptionFieldStatus = Confluence.CreateJiraContent.JiraIssue.FieldStatus.HIDDEN;
        let otherRequiredFields = {};

        fields.forEach(function (value) {
            const key = value.fieldId || value.key; // JIRA Cloud removed fieldId
            if (key === "description") {
                descriptionFieldStatus = (value.required === true ?
                    Confluence.CreateJiraContent.JiraIssue.FieldStatus.REQUIRED:
                    Confluence.CreateJiraContent.JiraIssue.FieldStatus.VISIBLE);
            } else if (value.required === true  && !value.hasDefaultValue && !_.contains(defaultRequiredFields, key)) {
                otherRequiredFields[key] = value;
            }
        });

        return {descriptionFieldStatus: descriptionFieldStatus, requiredFields: otherRequiredFields};
    }

    var getFieldName = function(fieldKey, paramObj) {
        var fieldDataKey = buildFieldDataKey(paramObj);
        // get field data in cached
        if (JiraIssueCache.containKey(fieldDataKey)) {
            var fields = JiraIssueCache.get(fieldDataKey).requiredFields;
            return fields[fieldKey] && fields[fieldKey].name;
        }

        return "";
    };

    var buildFieldDataKey = function(paramObj) {
        return CREATE_JIRA_ISSUE_FIELD_CACHED_KEY_PREFIX + paramObj.serverId + "_" + paramObj.projectId + "_" + paramObj.issueTypeId;
    };

    return {
        resolveEpicIssueType: resolveEpicIssueType,
        getCreateIssueMeta: getCreateIssueMeta,
        getFieldName: getFieldName
    };
});

require('confluence/cjc/module-exporter')
    .exportModuleAsGlobal('confluence/cjc/common/jira-issues-helper', 'Confluence.CreateJiraContent.JiraIssue');
