define('confluence/cjc/create-issue-form/jira-epic-content-view', [
    'ajs',
    'jquery',
    'backbone',
    'confluence/cjc/common/jira-issues-helper'
],
function(
    AJS,
    $,
    Backbone,
    jiraIssueHelper
) {
    'use strict';
    var DEFAULT_OPTION_VALUE = '-1';

    /**
     * This class is migrated from a part of `.../js/jira-form-helper.js`.
     * We want to kill `.../js/jira-form-helper.js` and first step to split it up as many modules as possible.
     */
    var JiraEpicContentView = Backbone.View.extend({
        template: window.Confluence.CreateJiraContent,

        initialize: function(options) {
            // get all necessary input options
            this.formObjectView = options.formObjectView;

            // initialize variables
            this.$formWrapper = this.formObjectView.$el;
            this.$projectSelect = this.$formWrapper.find('#jira-projects');
            this.$issueTypeSelect = this.$formWrapper.find('#jira-issue-types');

            this.epicSummaryText = '';
            this.epicIssueTypeId = null;

            // XHR object, hold this object to call abort function when needed
            this.xhrEpicPage = null;
            this.xhrProject = null;
        },

        /**
         * Detect if there is any Epic issue on the page with match with server selection
         * If have 1 ->display on Create-Jira-Issue dialog
         * If 0 or more than 2: ignore it
         */
        _epicLinkJiraIssuePageHandling: function(serverId) {
            var self = this;

            // If xhrEpicPage is existed, this mean previous ajax hadn't completed yet, call abort() to cancel it.
            if (this.xhrEpicPage && this.xhrEpicPage.readyState !== 4 && this.xhrEpicPage.abort) {
                this.xhrEpicPage.abort();
            }

            this.xhrEpicPage = $.ajax({
                type: 'GET',
                timeout: 0,
                contentType: 'application/json',
                url: AJS.Confluence.getContextPath() + '/rest/createjiracontent/1.0/find-epic-issue',
                data: {
                    pageId: AJS.params.pageId,
                    serverId: serverId,
                    epicIssueTypeId: self.epicIssueTypeId
                },
                success: function(epicData) {
                    if (!epicData || !epicData.epicKey) {
                        self.reset();
                        return;
                    }
                    var epicIssueHtml = self.template.renderEpicContent({
                        epicKey: epicData.epicKey,
                        epicHtmlPlaceHolder: epicData.epicHtmlPlaceHolder
                    });
                    self.$el.html(epicIssueHtml);

                    // get summary data before it removed out dom.
                    self.epicSummaryText = self.$el.find('.summary').text();

                    // remove summary, status out of epic render
                    var epicEl = self.$el.find('a:first-child').clone();
                    self.$el.find('.jira-issue').html(epicEl);

                    // add tooltip
                    self.$('.jira-issue').attr('title', self.epicSummaryText).tooltip({gravity: 'sw'});
                    self.formObjectView.refreshDialog();
                },
                error: function(xhr) {
                    self.$el.empty();
                    AJS.logError('confluence-jira-content:epicLinkJiraIssuePageHandling - Error when detect epic on page with status=' + xhr.status);
                }
            });
        },


        /**
         * Epic link can be configured to not display on screen, in this case, remove link suggestion.
         */
        handleEpicContent: function(epicField) {
            var self = this;

            $.when(this.xhrProject, this.xhrEpicPage).done(function() {
                var showEpic = self.epicIssueTypeId
                        && !self.formObjectView.extraRequiredFieldsView.hasUnsupportedFields // do not display epic content when there are unsupported fields
                        && ((self.$projectSelect.val() === DEFAULT_OPTION_VALUE) // project has not chosen yet
                        || (epicField && self.epicIssueTypeId !== self.$issueTypeSelect.val())); // had epic field and does not choose the Epic issue type.

                self.$el.toggle(showEpic ? true : false);
            });
        },

        /**
         * Try to resolve epicIssueType and detect epicLink on page
         */
        detectEpicIssueOnPage: function(serverId) {
            var onSuccess = function(issueTypeId) {
                this.epicIssueTypeId = issueTypeId;
                this._epicLinkJiraIssuePageHandling(serverId);
            }.bind(this);

            var onError = function() {
                this.epicIssueTypeId = (serverId === this.formObjectView.serverSelectView.getCurrentServer().id)
                                        ? null
                                        : this.epicIssueTypeId;
            }.bind(this);

            jiraIssueHelper.resolveEpicIssueType(serverId, onSuccess, onError);
        },

        reset: function() {
            this.$el.empty();
        },

        isEpicChecked: function() {
            var $epicCheckbox = this.$('#epic-link > .checkbox');
            return ($epicCheckbox.is(':visible') && $epicCheckbox.is(':checked'));
        },

        getEpicCheckBoxValue: function() {
            var $epicCheckbox = this.$('#epic-link > .checkbox');
            return $epicCheckbox.val();
        }
    });

    return JiraEpicContentView;
});
