define('confluence/cjc/create-issue-form/project-issue-type-select-view', [
    'ajs',
    'jquery',
    'backbone',
    'underscore',
    'confluence/cjc/common/select2-helper',
    'confluence/cjc/jira-issue-data-helper'
],
function(
    AJS,
    $,
    Backbone,
    _,
    select2Helper,
    jiraIssueDataHelper
) {
    'use strict';

    var DEFAULT_OPTION_PROJECT_TEXT = AJS.I18n.getText('createjiracontent.dialog.form.select.default.project.text');
    var DEFAULT_OPTION_ISSUETYPE_TEXT = AJS.I18n.getText('createjiracontent.dialog.form.select.default.issuetype.text');

    /**
     * This class is migrated from a part of `.../js/jira-form-helper.js`.
     * We want to kill `.../js/jira-form-helper.js` and first step is to split it up as many modules as possible.
     */
    const ProjectIssueTypeSelectView = Backbone.View.extend({
        template: window.Confluence.CreateJiraContent,

        events: {
            'change #jira-projects': '_loadIssueTypes',
            'change #jira-issue-types': '_onIssueTypeSelectChanged'
        },

        initialize: function(options) {
            // get all necessary input options
            this.formObjectView = options.formObjectView;
            this.getStoredDialogValues = options.getStoredDialogValues;
            this.setStoredDialogValues = options.setStoredDialogValues;

            // initialize variables
            this.$formWrapper = this.formObjectView.$el;
            this.$projectSelect = this.$formWrapper.find('#jira-projects');
            this.$serverSelect = this.$formWrapper.find('#jira-servers');
            this.$issueTypeSelect = this.$formWrapper.find('#jira-issue-types');

            // Get default issue type names from i18n for set default display of issue type select box
            this.defaultIssueTypeNames = AJS.I18n.getText('createjiracontent.dialog.form.issuetype.default.text').toLowerCase();
            this.defaultIssueTypeNames = this.defaultIssueTypeNames.split(',');
        },

        render: function() {
            select2Helper.createSelect2WithTooltip(this.$projectSelect,
                {
                    containerCssClass: 'select-container select-project-container',
                    dropdownCssClass: 'projects-dropdown',
                    formatSelection: this._projectSelectFormat.bind(this),
                    formatResult: this._projectSelectFormat.bind(this)
                }
            );

            select2Helper.createSelect2WithTooltip(this.$issueTypeSelect,
                {
                    containerCssClass: 'select-container select-issuetype-container',
                    dropdownCssClass: 'issue-types-dropdown',
                    minimumResultsForSearch: -1,
                    formatSelection: this._issueTypeSelectFormat.bind(this),
                    formatResult: this._issueTypeSelectFormat.bind(this)
                }
            );

            select2Helper.clearAndDisable(this.$issueTypeSelect, DEFAULT_OPTION_ISSUETYPE_TEXT);
        },

        /**
         * Handle issue type select2 change
         * @param e
         * @private
         */
        _onIssueTypeSelectChanged: function(/* e */) {
            this.changeIssuesTypeIconForPreviewPanel();
            this.formObjectView.checkCreateIssueFields();
        },

        _projectSelectFormat: function(result) {
            var projectAvatarUrl;

            if (!result) {
                return '';
            }

            if (result.id >= 0) {
                projectAvatarUrl = jiraIssueDataHelper.getProjectIconUrl(this.getServerId(), result.id);
            }

            return this.template.selectOptionProject({
                optionValue: result.text,
                imageUrl: projectAvatarUrl,
                lazyLoadImage: true
            });
        },

        _issueTypeSelectFormat: function(result) {
            var issueTypeImageUrl;

            if (!result) {
                return '';
            }

            if (result.id >= 0) {
                issueTypeImageUrl = jiraIssueDataHelper.getIssueTypeIconUrl(
                        this.getServerId(),
                        this.getProjectId(),
                        result.id);
            }

            return this.template.selectOptionWithImage({
                optionValue: result.text,
                imageUrl: issueTypeImageUrl
            });
        },

        fillProjectSelectBox: function(projects) {
            select2Helper.clearAndAddDefaultOption(this.$projectSelect, DEFAULT_OPTION_PROJECT_TEXT);

            var selectHTML = [];
            $.each(projects, function(index, project) {
                var optionContent = aui.form.optionOrOptgroup({
                    value: project.id,
                    text: project.name
                });
                selectHTML.push(optionContent);
            });

            this.$projectSelect.append(selectHTML.join(''));
            select2Helper.toggleEnable(this.$projectSelect, true);
        },

        loadProjects: function() {
            var self = this;
            var currentServer = self.formObjectView.serverSelectView.getCurrentServer();
            var serverSupported = this.formObjectView.serverSelectView.verifySupportedServer(currentServer);

            if (serverSupported) {
                select2Helper.showLoadingAndDisable(self.$projectSelect);

                var dfd = jiraIssueDataHelper.loadProjects(currentServer.id);
                dfd.done(function(projects) {
                    if (projects.errors && projects.errors.length) {
                        var err = projects.errors[0];
                        self.trigger('projects.loaded.error', err);

                        select2Helper.displayErrorLoading(self.$projectSelect);
                        select2Helper.toggleEnable(self.$projectSelect, true);

                    } else {
                        Backbone.trigger('conf.jira.content.projects.loaded');
                        self.fillProjectSelectBox(projects);

                        // Sometime two separate servers have the same projects id, we have to make sure they match both server and issueType before show Summary
                        var storedDialogValues = self.getStoredDialogValues();
                        if (storedDialogValues && storedDialogValues.serverId === currentServer.id) {
                            // If storedDialogValues is defined, bind it into Select
                            var isSetSuccess = select2Helper.setSelectFieldValue(self.$projectSelect, storedDialogValues.projectId);

                            // If project binding successful, IssueType should be loaded
                            if (isSetSuccess) {
                                self._loadIssueTypes();
                            } else {
                                self.setStoredDialogValues(false);
                            }
                        } else {
                            // Hide and reset IssueType Field
                            select2Helper.clearAndDisable(self.$issueTypeSelect, DEFAULT_OPTION_ISSUETYPE_TEXT);
                        }
                    }
                })
                .fail(function(xhr) {
                    self.trigger('projects.loaded.error', xhr);

                    select2Helper.displayErrorLoading(self.$projectSelect);
                    select2Helper.toggleEnable(self.$projectSelect, true);
                });

                return dfd;
            }
        },

        /**
         * Check issue type name equals with default issue type name
         */
        _isMatchIssueTypeDefaultName: function(issueTypeName) {
            for ( var i = 0; i < this.defaultIssueTypeNames.length; i++) {
                if (this.defaultIssueTypeNames[i] === issueTypeName.toLowerCase()) {
                    return true;
                }
            }
            return false;
        },

        _loadIssueTypes: function() {
            var projectId = this.getProjectId();

            if (projectId && projectId !== select2Helper.DEFAULT_OPTION_VALUE) {
                this._fillIssueTypeSelectBox();
            } else {
                select2Helper.clearAndDisable(this.$issueTypeSelect, DEFAULT_OPTION_ISSUETYPE_TEXT);
            }
        },

        getIssueTypes: function() {
            return jiraIssueDataHelper.getIssueTypes(
                this.getServerId(),
                this.getProjectId()
            );
        },

        changeIssuesTypeIconForPreviewPanel: function() {
            const issueTypeId = this.getIssueTypeId();

            if (issueTypeId !== select2Helper.DEFAULT_OPTION_VALUE) {
                this.getIssueTypes()
                    .done(function () {
                        const issueTypes = jiraIssueDataHelper.getProject(
                            this.getServerId(),
                            this.getProjectId()
                        ).issueTypes;
                        const currentIssueType = _.find(issueTypes, function (issueType) {
                            return issueType.id === issueTypeId;
                        });

                        if (currentIssueType) {
                            const issueIconUrl = currentIssueType.iconUrl;
                            this.$formWrapper.find('.issue-container img').attr('src', issueIconUrl);
                        }
                    }.bind(this));
            }
        },

        _fillIssueTypeSelectBox: function() {
            let dfd = this.getIssueTypes()
                .done(function () {
                    const issueTypes = jiraIssueDataHelper.getProject(
                        this.getServerId(),
                        this.getProjectId()
                    ).issueTypes;

                    // show default select when data is empty
                    if (_.isEmpty(issueTypes)) {
                        select2Helper.clearAndDisable(this.$issueTypeSelect, DEFAULT_OPTION_ISSUETYPE_TEXT);
                    } else {
                        var selectedValue = null;
                        var selectHtml = [];

                        $.each(issueTypes, function (index, issueType) {
                            // store issueType id for make default value of issuetype select box
                            if (!selectedValue && this._isMatchIssueTypeDefaultName(issueType.name)) {
                                selectedValue = issueType.id;
                            }

                            var optionContent = aui.form.optionOrOptgroup({
                                value: issueType.id,
                                text: issueType.name
                            });

                            selectHtml.push(optionContent);
                        }.bind(this));

                        this.$issueTypeSelect.html(selectHtml.join(''));
                        this.$issueTypeSelect.auiSelect2('val',
                            selectedValue ? selectedValue : select2Helper.DEFAULT_OPTION_VALUE);

                        const currentIssueType = _.find(issueTypes, function (issueType) {
                            return issueType.id === this.getIssueTypeId();
                        }.bind(this));
                        currentIssueType && this.$formWrapper.find('.issue-container img').attr('src', currentIssueType.iconUrl);

                        select2Helper.toggleEnable(this.$issueTypeSelect, true);

                        // If storedDialogValues is defined, bind it into Select
                        if (this.getStoredDialogValues()) {
                            const isSetSuccess = select2Helper.setSelectFieldValue(this.$issueTypeSelect, this.getStoredDialogValues().issueTypeId);
                            if (!isSetSuccess) {
                                this.setStoredDialogValues(false);
                            }
                        }

                        const projectValue = this.getProjectId();
                        this.trigger('project.changed', projectValue);

                        this.formObjectView.checkCreateIssueFields();
                    }
                }.bind(this));
            return dfd;
        },

        getIssueTypeId: function() {
            return this.$issueTypeSelect.val();
        },

        getProjectId: function() {
            return this.$projectSelect.val();
        },

        getServerId: function() {
            return this.$serverSelect.val();
        },

        closeAllSelect2Dropdown: function() {
            this.$projectSelect.auiSelect2('close');
            this.$issueTypeSelect.auiSelect2('close');
        },

        updateAfterServerChanged: function() {
            var self = this;

            $.when(self.loadProjects()).done(function() {
                // We only clear and disable Issue Type if storedDialogValues is not defined. If not it will clear all storevalue binding
                if (!self.getStoredDialogValues()) {
                    select2Helper.clearAndDisable(self.$issueTypeSelect, DEFAULT_OPTION_ISSUETYPE_TEXT);
                    self.changeIssuesTypeIconForPreviewPanel();
                }

                // trigger the event for projectSelect change.
                self.$projectSelect.change();
            });
        }
    });

    return ProjectIssueTypeSelectView;
});
