define('confluence/cjc/create-issue-form/server-select-view', [
    'ajs',
    'jquery',
    'backbone',
    'confluence/cjc/error/error-type',
    'underscore',
    'confluence/cjc/common/select2-helper',
    'confluence/cjc/jira-issue-data-helper'
],
function(
    AJS,
    $,
    Backbone,
    ErrorType,
    _,
    select2Helper,
    jiraIssueDataHelper
) {
    'use strict';

    /**
     * This class is migrated from a part of `.../js/jira-form-helper.js`.
     * We want to kill `.../js/jira-form-helper.js` and first step to split it up as many modules as possible.
     */
    const ServerSelectView = Backbone.View.extend({
        template: window.Confluence.CreateJiraContent,

        initialize: function(options) {
            // get all necessary input options
            this.formObjectView = options.formObjectView;

            // initialize variables
            this.$formWrapper = this.formObjectView.$el;
            this.$serverSelect = this.$formWrapper.find('#jira-servers');
            this.$serverSelect.change(this._onServerSelectChanged.bind(this));

            // all els should be hidden if there is an error with Jira server
            this.$hiddenElementsWhenInvalid = this.$formWrapper
                                        .find('#jira-content-create-issue-form')
                                        .children().not('.dialog-header, .servers, .messages');

            Backbone.on('conf.jira.content.projects.loaded', function(projects) {
                this.jiraServerStatus(200, projects);
            }.bind(this));
        },

        render: function() {
            // begin load servers information
            this._loadServers().then(
                this._handleServerLoadingSuccess.bind(this),
                this._handleServerLoadingError.bind(this)
            );
        },

        _onServerSelectChanged: function() {
            // clear the error message if have
            this.formObjectView.errorMessageView.removeMessage(ErrorType.ERROR_JIRA_SERVER_ID);
            this.formObjectView.errorMessageView.removeMessage(ErrorType.ERROR_FORM_ID);

            var currentServer = this.getCurrentServer();
            var serverSupported = this.verifySupportedServer(currentServer);
            if (serverSupported) {
                this.$hiddenElementsWhenInvalid.removeClass('hidden');
                this.trigger('server.select.changed');
            }
        },

        _handleServerLoadingSuccess: function() {
            this._removeDialogLoading();

            var serverId = this.getServerId();
            // do not load project when server is empty
            if (serverId) {
                // display create isue dialog content when finished loading server data
                this.$formWrapper
                        .find('#jira-servers, .dialog-content, .buttons-group, #text-suggestion')
                        .removeClass('hidden');

                this.trigger('server.loaded.success');
            }
        },

        _handleServerLoadingError: function() {
            this._removeDialogLoading();
            this.showJiraServerWarning(AJS.I18n.getText('createjiracontent.dialog.form.get.jiraserver.error'));
        },

        showJiraServerWarning: function(message) {
            this.formObjectView.errorMessageView.addMessage(ErrorType.ERROR_JIRA_SERVER_ID, message);
        },

        _removeDialogLoading: function() {
            this.$formWrapper.find('#create-issue-loading').remove();
        },

        getCurrentServer: function() {
            return jiraIssueDataHelper.getServerById(this.$serverSelect.val());
        },

        /**
         * Clear project cached base on serverId
         * @private
         */
        _clearCacheInCurrentServer: function() {
            jiraIssueDataHelper.clearProjects(this.$serverSelect.val());
        },

        // OAuthentication handling
        _handleUnauthorizedServer: function() {
            this.$hiddenElementsWhenInvalid.addClass('hidden');
            var currentServer = this.getCurrentServer();

            if (currentServer.authUrl) {
                this._createOauthForm(function onAuthorizedSuccessCallback() {
                    this.$hiddenElementsWhenInvalid.removeClass('hidden');
                    this._clearCacheInCurrentServer();
                    this.trigger('authoriszed.success');
                }.bind(this));
            } else {
                // Jira authenticated configuration was changed when server list are loaded
                this.showJiraServerWarning(AJS.I18n.getText('createjiracontent.dialog.form.jiraserver.authurl.error'));
            }
        },

        _createOauthForm: function(successCallback) {
            var currentServer = this.getCurrentServer();
            var oauthCallbacks = {
                onSuccess: function() {
                    currentServer.authUrl = null;
                    successCallback();
                },
                onFailure: function() {}
            };

            this.showJiraServerWarning(this.template.getOAuthMessage({
                confluenceApplicationName: 'Confluence'
            }));

            $('.' + ErrorType.ERROR_JIRA_SERVER_ID)
                .find('.oauth-init').click(function(e) {
                    window.AppLinks.authenticateRemoteCredentials(
                        currentServer.authUrl,
                        oauthCallbacks.onSuccess,
                        oauthCallbacks.onFailure);
                    e.preventDefault();
            });
        },

        jiraServerStatus: function(statusCode, data) {
            // application link existed, but do not have any configuration further
            if (statusCode === 200 && data && data.length === 0) {
                this.$hiddenElementsWhenInvalid.addClass('hidden');

                this._clearCacheInCurrentServer();
                this.showJiraServerWarning(
                    AJS.format(AJS.I18n.getText('createjiracontent.dialog.form.jiraserver.project.empty'),
                        AJS.escapeHtml(this.getCurrentServer().name))
                );

            } else {
                switch (statusCode) {
                    case 401: // Unauthorized
                        this._handleUnauthorizedServer();
                        break;
                    case 404: // not found
                    case 500: // applink error
                    case 504: // proxy timeout
                        this.$hiddenElementsWhenInvalid.addClass('hidden');
                        this.showJiraServerWarning(AJS.I18n.getText('createjiracontent.dialog.form.jiraserver.connect.error'));
                        break;
                    default: // normal
                        this.$hiddenElementsWhenInvalid.removeClass('hidden');
                }
            }

            this.formObjectView.refreshDialog();
        },

        // Check whether selected Jira Server is supported version
        verifySupportedServer: function(server) {
            if (!server.supportedVersion) {
                this.$hiddenElementsWhenInvalid.addClass('hidden');
                this.showJiraServerWarning(AJS.format(AJS.I18n.getText('createjiracontent.messages.server.notsupported'), server.url));
            } else {
                this.$hiddenElementsWhenInvalid.removeClass('hidden');
            }

            this.formObjectView.refreshDialog();

            return server.supportedVersion;
        },

        _fillAndDisplayServers: function(servers) {
            var selectHTML = [];
            $.each(servers, function(index, server) {
                var optionContent = aui.form.optionOrOptgroup({
                    value: server.id,
                    text: server.name
                });
                selectHTML.push(optionContent);
            });
            this.$serverSelect.append(selectHTML.join(''));

            if (servers.length === 1) {
                this.$serverSelect.closest('.servers').addClass('hidden');
            } else {
                select2Helper.createSelect2WithTooltip(this.$serverSelect,
                {
                    containerCssClass: 'select-container select-server-container',
                    dropdownCssClass: 'server-dropdown',
                    minimumResultsForSearch: -1
                });
            }

            this.trigger('fill.server.data.to.select', this.$serverSelect);
        },

        _loadServers: function() {
            var promise = jiraIssueDataHelper.loadServers();
            promise.done(function(servers) {
                if (servers.length) {
                    this._fillAndDisplayServers(servers);
                } else {
                    this.showJiraServerWarning(AJS.I18n.getText('createjiracontent.dialog.form.get.jiraserver.error.empty'));
                }
            }.bind(this));

            return promise;
        },

        getServerId: function() {
            return this.$serverSelect.val();
        },

        closeAllSelect2Dropdown: function() {
            this.$serverSelect.auiSelect2('close');
        }
    });

    return ServerSelectView;
});
