/**
 * A quick reload handler for page level comments
 *
 * @module confluence-quick-reload/handlers/quick-reload-comments
 *
 * @tainted Confluence.CommentDisplayManager
 * @tainted Confluence.Likes
 * @tainted QuickReload.Templates
 *
 */
define('confluence-quick-reload/handlers/quick-reload-comments', [
    'ajs',
    'underscore',
    'jquery',
    'confluence/flag',
    'confluence/api/event',
    'confluence-quick-reload/utils/quick-reload-count',
    'confluence/legacy'
],
function(
    AJS,
    _,
    $,
    Flag,
    event,
    QuickReloadCount,
    Confluence
) {
    'use strict';

    var pageCommentFlag;
    var idsToIgnore = [];

    function _processResultsForDisplay(results) {
        var _results = _.clone(results); // make a copy so we don't modify the original data
        _results.reverse(); // reverse to make put the most recent commenter at the beginning of list

        // only keep the most recent comment by each author
        _results = _.uniq(_results, function(commentAndUser) {
            return commentAndUser.commenter.userName;
        });

        // only keep the part of the commentAndUser object necessary for rendering
        return _.map(_results, function(commentAndUser) {
            return commentAndUser.commenter;
        });
    }

    /**
     * Render the page level comments on the page
     * @param comments comments to render
     * @private
     */
    function _displayPageComments(comments) {
        if (Confluence && Confluence.CommentDisplayManager) {
            Confluence.CommentDisplayManager.clearFocus();
            _.map(comments, function(commentAndUser) {
                var update = Confluence.CommentDisplayManager.getCommentNode(_getCommentId(commentAndUser)) !== null;
                var comment = Confluence.CommentDisplayManager.addOrUpdateComment(commentAndUser.commenter, commentAndUser.comment, true, true);
                // TODO Ideally likes are handled by standard display manager
                if (Confluence.Likes && !update) {
                    Confluence.Likes.appendAction($(comment));
                    Confluence.Likes.updateComment($(comment), {});
                }
            });
            // update comment section title with new comment count
            Confluence.CommentDisplayManager._updateCommentSectionTitle();
            event.trigger('ic-jim-async-supported');
        }
    }

    /**
     * Shorthand for getting a comment id from a commentAndUser object
     * @param commentAndUser
     * @returns Number comment id
     * @private
     */
    function _getCommentId(commentAndUser) {
        return commentAndUser.comment.id;
    }

    return {
        results: [],

        property: 'comments',

        /**
         * Register an ID with this function to ignore it the next time we get an update for this. Call this when
         * you don't want the user to see something - e.g. maybe you're handing the displaying of comments yourself.
         * @param id comment id to ignore once
         */
        ignoreOnce: function(id) {
            idsToIgnore.push(id);
        },

        /**
         * Filters out inline comments and existing comments from newResults
         * @param currentResults array of currentResults
         * @param newResults array of newResults
         * @returns {*}
         */
        filterNewResults: function(currentResults, newResults) {
            if (newResults.length === 0) {
                return newResults;
            }

            var currentsIds = _.map(currentResults, _getCommentId);

            return _.filter(newResults, function(commentAndUser) {
                if (commentAndUser.comment.isInlineComment) {
                    return false;
                }
                var commentId = _getCommentId(commentAndUser);
                if (_.contains(currentsIds, commentId)) {
                    return false;
                }
                // If something is matched in the ignore-once list, then remove it
                var ignoreIdx = $.inArray(commentId, idsToIgnore);
                if (ignoreIdx >= 0) {
                    idsToIgnore.splice(ignoreIdx, 1);
                    return false;
                }

                return true;
            });
        },

        render: function(newResults) {
            QuickReloadCount.setCount(QuickReloadCount.getCount() + newResults.length);

            var flagDefaults = {
                close: 'manual',
                type: 'info',
                extraClasses: 'qr-flag',
                fifo: true,
                stack: 'quickreload'
            };

            var title = (this.results.length > 1) ? AJS.I18n.getText('quick.reload.newpagecomments', this.results.length)
                : AJS.I18n.getText('quick.reload.newpagecomment');

            var commentUsers = _processResultsForDisplay(this.results);
            var lastCommentText = $('<div></div>').html(this.results[this.results.length - 1].comment.html).text();
            var flagBody = QuickReload.Templates.pageComment({
                commentUsers: commentUsers,
                noticeText: lastCommentText
            });
            var self = this;
            if (pageCommentFlag === undefined || pageCommentFlag.getAttribute('aria-hidden') === 'true') {
                pageCommentFlag = new Flag($.extend({}, {
                    body: flagBody,
                    title: title
                }, flagDefaults));

                var onShow = function() {
                    _displayPageComments(self.results);
                    pageCommentFlag.close();
                };
                $(pageCommentFlag).on('click', '.qr-notice-show', _.bind(onShow, this));

                var onClose = function() {
                    QuickReloadCount.setCount(QuickReloadCount.getCount() - self.results.length);
                    self.results = [];
                };
                $(pageCommentFlag).on('aui-flag-close', _.bind(onClose, this));
            } else {
                $(pageCommentFlag).find('.qr-container').replaceWith(flagBody);
                $(pageCommentFlag).find('.title').text(title);
            }
        }
    };
});
