/**
 * @module confluence-quick-reload/handlers/quick-reload-page
 */
define('confluence-quick-reload/handlers/quick-reload-page', [
    'underscore',
    'jquery',
    'ajs',
    'confluence/flag',
    'confluence-quick-reload/utils/quick-reload-count'
],
/**
 * A QuickReload handler for updating the page content.
 * Obviously for multiple updates only the most recent change will be shown.
 *
 * @tainted QuickComment.Templates
 * @exports confluence-quick-reload/handlers/quick-reload-page
 */
function(
    _,
    $,
    AJS,
    Flag,
    QuickReloadCount
) {
    'use strict';

    var pageEditFlag;

    function _processResultsForDisplay(results) {
        var _results = _.clone(results); // make a copy so we don't modify the original data
        _results.reverse(); // reverse to make put the most recent editor at the beginning of list

        // only keep the most recent edit by each author
        _results = _.uniq(_results, function(pageEdit) {
            return pageEdit.editor.userName;
        });

        // only keep the part of the pageEdit object necessary for rendering
        return _.map(_results, function(pageEdit) {
            return pageEdit.editor;
        });
    }

    return {
        results: [],
        property: 'page',

        filterNewResults: function(current, newResults) {
            // REST resource sets result equal to empty string when null, instead of empty list. Sanitizing here.
            if (newResults.length > 0 && typeof newResults[0] !== 'string') {
                return newResults;
            }
            return [];
        },

        render: function(newResults) {
            QuickReloadCount.setCount(QuickReloadCount.getCount() + newResults.length);

            var flagDefaults = {
                close: 'manual',
                type: 'info',
                extraClasses: 'qr-flag',
                fifo: true,
                stack: 'quickreload'
            };
            var title = (this.results.length > 1) ? AJS.I18n.getText('quick.reload.edits', this.results.length)
                : AJS.I18n.getText('quick.reload.edit');

            var editors = _processResultsForDisplay(this.results);
            var flagBody = QuickReload.Templates.pageEdit({
                pageEditors: editors
            });
            var self = this;
            if (pageEditFlag === undefined || pageEditFlag.getAttribute('aria-hidden') === 'true') {
                pageEditFlag = new Flag($.extend({}, {
                    body: flagBody,
                    title: title
                }, flagDefaults));

                $(pageEditFlag).on('click', '.qr-notice-show', function() {
                    $(this).prop('disabled', true).prepend('<span class="aui-icon aui-icon-wait"></span>&nbsp;');
                    window.location.reload();
                });

                var onClose = function() {
                    QuickReloadCount.setCount(QuickReloadCount.getCount() - self.results.length);
                    self.results = [];
                };
                $(pageEditFlag).on('aui-flag-close', _.bind(onClose, this));
            } else {
                $(pageEditFlag).find('.qr-container').replaceWith(flagBody);
                $(pageEditFlag).find('.title').text(title);
            }
        }
    };
});
