/**
 * @module confluence-quick-reload/handlers/quick-reload-read-only-mode
 */
define('confluence-quick-reload/handlers/quick-reload-read-only-mode', [
    'ajs',
    'jquery',
    'confluence/message-controller',
    'confluence/meta',
    'confluence/api/event'
],
/**
 * A quick reload handler for read-only mode
 *
 * @exports confluence-quick-reload/handlers/quick-reload-read-only-mode
 */
function(
    AJS,
    $,
    MessageController,
    Meta,
    event
) {
    'use strict';

    return {
        results: [],

        property: 'readOnlyMode',

        filterNewResults: function(current, newResults) {
            if (newResults.length > 0 && typeof newResults[0] !== 'string') {
                return newResults;
            }
            return [];
        },
        render: function(newResults) {
            var readOnlyModeResult = newResults[0];
            var currentAccessMode = Meta.get('access-mode');
            var newAccessMode = readOnlyModeResult.isEnabled ? 'READ_ONLY' : 'READ_WRITE';
            var successMessage = Meta.get('render-mode') === 'READ_ONLY' ? AJS.I18n.getText('read.write.mode.success.banner.with.refresh.link') : AJS.I18n.getText('read.write.mode.default.success.banner.message');

            // If the success banner is being shown and the access mode does not changed to READ_ONLY, we won't need to do anything else
            if (MessageController.hasSuccess(MessageController.Location.BANNER) && !readOnlyModeResult.isEnabled) {
                return;
            }

            Meta.set('access-mode', newAccessMode);
            if (newAccessMode !== currentAccessMode) {
                event.trigger('qr:access-mode', { accessMode: newAccessMode });
            }
            if (this.shouldDisplayError(readOnlyModeResult)) {
                // If the warning banner is not present, show a new one
                if (!MessageController.hasErrors(MessageController.Location.BANNER)) {
                    this.showError(readOnlyModeResult.bannerMessage);
                }
            } else if (currentAccessMode === 'READ_ONLY' && newAccessMode === 'READ_WRITE') {
                // The warning banner must be replaced with the success banner OR removed
                // when Read-only mode was enabled then turned off
                this.showSuccess(successMessage);
            } else {
                this.clearAll();
            }
        },
        clearAll: function() {
            MessageController.clearErrors(MessageController.Location.ALL);
            MessageController.clearSuccess(MessageController.Location.ALL);
        },
        showError: function(bannerMessage) {
            MessageController.clearSuccess(MessageController.Location.ALL);
            MessageController.showError(bannerMessage, MessageController.Location.BANNER);
        },
        showSuccess: function(successMessage) {
            MessageController.clearErrors(MessageController.Location.ALL);
            MessageController.showSuccess(successMessage, MessageController.Location.BANNER, {
                closeable: true
            });
        },
        shouldDisplayError: function(readOnlyModeResult) {
            return readOnlyModeResult.isBannerMessageOn || readOnlyModeResult.isEnabled;
        }
    };
});
