/**
 * @module confluence-quick-reload/main/quick-reload-manager
 * Quick reload library for informing users of new comments, edits and inline comments when editing a page
 * @tainted Meta
 * @exports confluence-quick-reload/main/quick-reload-manager
 */
define('confluence-quick-reload/main/quick-reload-manager', [
    'underscore',
    'jquery',
    'ajs',
    'confluence/api/constants',
    'confluence-quick-reload/utils/quick-reload-timer',
    'confluence/meta',
    'confluence/api/logger'
],
function(
    _,
    $,
    AJS,
    Constants,
    SmartTimer,
    Meta,
    Logger
) {
    'use strict';

    /**
     * @constructor the constructor function
     */
    function QuickReloadManager() {
        this.enabled = false;
        this.handlers = [];
        this.lastFetchTime = $('meta[name="confluence-request-time"]').attr('content') || (new Date()).getTime();
        this.timer = null;

        _.bindAll(this, 'addHandler', 'removeHandler', 'update', 'enable', 'disable', 'onUpdateSuccess', 'onUpdateError');
    }

    /**
     * Adds a handler that processes quick reload data and renders it inside the application. Each handler must be unique
     * @param {Object} handler a quick reload handler object
     * @returns {Object} the current QuickReloadManager object
     */
    QuickReloadManager.prototype.addHandler = function(handler) {
        var alreadyRegistered = false;
        var i;
        for (i = 0; i < this.handlers.length; i++) {
            if (this.handlers[i] === handler) {
                alreadyRegistered = true;
            }
        }
        if (alreadyRegistered !== true) {
            this.handlers.push(handler);
        }
    };

    /**
     * Removes a handler that processes quick reload data
     * @param {Object} handler a quick reload handler object
     */
    QuickReloadManager.prototype.removeHandler = function(handler) {
        var i;
        if (handler) {
            for (i = 0; i < this.handlers.length; i++) {
                if (this.handlers[i] === handler) {
                    this.handlers.splice(i, 1);
                    break;
                }
            }
        }
    };

    // Is there a better way to check this, but allow comment editor?
    function isFullScreenEdit() {
        return $('body').hasClass('contenteditor');
    }

    /**
     * Calls the QuickLoad service, finds any new updates since the last request and shows the notice dialog.
     */
    QuickReloadManager.prototype.update = function() {
        var pageId = Meta.get('page-id') !== undefined ? Meta.get('page-id') : '0';
        // This is assuming we never come back to the view page without refreshing
        // Ideally this we respond to an event instead of manually checking
        if (isFullScreenEdit()) {
            this.disable();
            return;
        }
        $.ajax({
            type: 'GET',
            url: Constants.CONTEXT_PATH + '/rest/quickreload/latest/' + pageId + '?since=' + encodeURIComponent(this.lastFetchTime),
            dataType: 'json'
        }).done(this.onUpdateSuccess).fail(this.onUpdateError);
    };

    /**
     * Enables quick reload polling for new page edits, comments and inline comments
     */
    QuickReloadManager.prototype.enable = function() {
        if (this.enabled) {
            return;
        }

        if (this.timer === null) {
            this.timer = new SmartTimer(this.update);
        }
        this.timer.start();
        this.enabled = true;
    };

    /**
     * Disabled quick reload polling for new page edits, comments and inline comments
     */
    QuickReloadManager.prototype.disable = function() {
        if (this.timer !== null) {
            this.timer.stop();
        }
        this.enabled = false;
    };

    /**
     * Returns a value informing the user if quick polling is enabled
     * @returns {Boolean} whether quick reload polling is enabled
     */
    QuickReloadManager.prototype.isEnabled = function() {
        return this.enabled;
    };

    /**
     * Callback on successful REST call to quick reload resource. It passes the REST data to each quick reload
     * handler for processing and rendering
     * @param {Object} data that contains the last fetch time
     * @param {String} textStatus the status string
     * @param {Object} jqXHR the Ajax response
     * @private
     */
    QuickReloadManager.prototype.onUpdateSuccess = function(data, textStatus, jqXHR) {
        if (jqXHR.status === 204) {
            return;
        }
        this.lastFetchTime = data.time;
        _.map(this.handlers, function(handler) {
            var newResults = data[handler.property];
            if (!Array.isArray(newResults)) {
                newResults = [newResults];
            }
            newResults = handler.filterNewResults(handler.results, newResults);

            if (newResults.length > 0) {
                handler.results = handler.results.concat(newResults);
                handler.render(newResults);
            }
        }, this);
    };

    /**
     * Error handler for quick reload REST resource. Disables quick reload polling if an error occurs
     * @param {Object} xhr the Ajax response
     * @private
     */
    QuickReloadManager.prototype.onUpdateError = function(xhr) {
        // Quick reload plugin will get disabled if received one of the following status codes:
        var blackListedStatusCodes = {
            404: 'not found - the plugin has been probably been removed or disabled from Confluence',
            500: 'generic server error',
            503: 'service unavailable',
            504: 'gateway timeout'
        };

        var request = xhr || {};
        var blackListedError = blackListedStatusCodes[request.status];
        if (blackListedError) {
            this.disable();
            Logger.error('Quick comment reload plugin has been disabled in this page due to a server error: "' + blackListedError + '". Please refresh the page to get it back.');
        }
    };

    // return a singleton
    return (new QuickReloadManager());
});
