(function($, _) {
    var RM = Confluence.Roadmap = Confluence.Roadmap || {};
    var USER_LOCALE = AJS.Meta.get('user-locale').replace('_', '-');

    RM.DateUtilities = {
        MILLISECONDS_A_DAY: 1000 * 60 * 60 * 24,
        MILLISECONDS_A_WEEK: 1000 * 60 * 60 * 24 * 7,

        getStartOfIsoWeek: date => new Date(date.getFullYear(), date.getMonth(), date.getDate() - date.getDay() + 1),
        getEndOfIsoWeek: date => new Date(date.getFullYear(), date.getMonth(), date.getDate() - date.getDay() + 7),
        getStartOfMonth: date => new Date(date.getFullYear(), date.getMonth(), 1),
        getEndOfMonth: date => new Date(new Date(date.getFullYear(), date.getMonth() + 1, 1).getTime() - 1), // moment.js implementation
        addMonths: (d, n) => {
            const date = new Date(d);
            const day = date.getDate();
            date.setDate(1);
            date.setMonth(date.getMonth() + n);
            const year = date.getFullYear();
            const isLeapYear = ((year % 4 == 0) && (year % 100 != 0)) || (year % 400 == 0);
            const daysInMonth = [31, (isLeapYear ? 29 : 28), 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][date.getMonth()]
            date.setDate(Math.min(day, daysInMonth));
            return date;
        },
        areInSameWeek: (date1, date2) => {
            const date1Rounded = new Date(new Date(date1).setHours(0, 0, 0, 0));
            const date2Rounded = new Date(new Date(date2).setHours(0, 0, 0, 0));
            return new Date(date1Rounded.setDate(date1Rounded.getDate() - date1Rounded.getDay())).getTime()
                === new Date(date2Rounded.setDate(date2Rounded.getDate() - date2Rounded.getDay())).getTime();
        },
        setIsoWeekday: (date, weekday) => new Date(date.setDate(date.getDate() - date.getDay() + weekday)),
        diffInYears: (fromDate, toDate) =>
            Math.abs(new Date(toDate.getTime() - fromDate.getTime()).getUTCFullYear() - 1970),
        /**
         * Try to parse a string to a date. If can not parse, return null.
         * @param {string|number|date} input input to be parsed
         * @return {Date | null} return parsed date or null
         */
        parseDate: function(input) {
            if (!input) {
                return null;
            }
            if (input instanceof Date || typeof input === 'number') {
                return new Date(input);
            }
            if (input.includes && input.includes(':')) {
                return new Date(input);
            }
            const parts = input.split('-');
            const year = parts[0].toString();
            const currentYear = new Date().getFullYear().toString();
            parts[0] = currentYear.substring(0, currentYear.length - year.length) + year;

            const timestamp = Date.parse(parts.join('-'));
            if (isNaN(timestamp)) {
                return null;
            } else {
                return new Date(new Date(parts[0], parts[1] - 1, parts[2]));
            }
        },

        /**
         * Get date string from date
         *
         * @param {Date} date value
         * @param {string} format format pattern string
        */
        getDateStringFromDate: function(date, format) {
            switch (format) {
                case Roadmap.SHORT_DATE_FORMAT:
                    return `${date.getFullYear()}-${(date.getMonth() + 1).toString().padStart(2, 0)}-${date.getDate().toString().padStart(2, 0)}`;
                case Roadmap.WEEK_FORMAT:
                    return date.getDate().toString().padStart(2, '0') + '-' + date.toLocaleString(USER_LOCALE, { month: "short" });
                case Roadmap.MONTH_FORMAT:
                    return date.toLocaleString(USER_LOCALE, { month: 'short' });
                case Roadmap.MONTH_YEAR_FORMAT:
                    return date.toLocaleString(USER_LOCALE, { month: 'short' }) + '-' + date.getFullYear();
                case Roadmap.DAY_FORMAT:
                    return date.getDate().toString().padStart(2, '0');
                default:
                    return date.getFullYear().toString() + '-'
                        + (date.getMonth() + 1).toString().padStart(2, '0') + '-'
                        + date.getDate().toString().padStart(2, '0') + ' '
                        + date.getHours().toString().padStart(2, '0') + ':'
                        + date.getMinutes().toString().padStart(2, '0') + ':'
                        + date.getSeconds().toString().padStart(2, '0');
            }
        },

        /**
         * Get number of months within a period
         *
         * @param {Date} startDate
         * @param {Date} endDate
         * @return {number} number of months within the period
         */
        getNumberOfMonths: function(startDate, endDate) {
            return endDate.getMonth() - startDate.getMonth() + (endDate.getFullYear() - startDate.getFullYear()) * 12;
        },

        /**
         * Get number of columns within a period for the given display option
         *
         * @param {Date} startDate
         * @param {Date} endDate
         * @param {string} displayOption
         * @return {number} number of columns within the period for the given display option
         */
        getNumberOfColumns: function(startDate, endDate, displayOption) {
            if (displayOption === Roadmap.TIMELINE_DISPLAY_OPTION.DAY) {
                return Math.ceil((endDate.getTime() - startDate.getTime()) / RM.DateUtilities.MILLISECONDS_A_DAY) + 1;
            } else if (displayOption === Roadmap.TIMELINE_DISPLAY_OPTION.WEEK) {
                return Math.ceil((endDate.getTime() - startDate.getTime()) / RM.DateUtilities.MILLISECONDS_A_WEEK) + 1;
            } else {
                return RM.DateUtilities.getNumberOfMonths(startDate, endDate) + 1;
            }
        },

        getMillisecondsInMonth: function(date) {
            return new Date(date.getFullYear(), date.getMonth() + 1, 0).getDate() * RM.DateUtilities.MILLISECONDS_A_DAY;
        },

        getMillisecondsFromStartMonth: function(date) {
            return date.getTime() - new Date(date.getFullYear(), date.getMonth(), 1).getTime();
        },

        /**
         * calculate duration information of a moment date
         *
         * @param {Date} date - date
         * @returns {Object} duration information object:
         *                      totalMsOfMonth: total Milliseconds of month get from date.
         *                      MsFromStartMonth: Milliseconds from the month's beginning date to date.
         *                      MsRemainingOfMonth: Milliseconds from date to the month's ending date.
         *                      durationOfMonth: month's duration from the month's beginning date to date.
         *                      remainingDurationOfMonth: month's duration from date to the month's ending date.
         */
        calculateDurationInformation: function(date) {
            var totalMsOfMonth = RM.DateUtilities.getMillisecondsInMonth(date);
            var msFromStartMonth = RM.DateUtilities.getMillisecondsFromStartMonth(date);
            var msRemainingOfMonth = totalMsOfMonth - msFromStartMonth;

            return {
                totalMsOfMonth: totalMsOfMonth,
                msFromStartMonth: msFromStartMonth,
                msRemainingOfMonth: msRemainingOfMonth,
                durationOfMonth: msFromStartMonth / totalMsOfMonth,
                remainingDurationOfMonth: msRemainingOfMonth / totalMsOfMonth
            };
        },

        /**
         * Get week duration by a start date and month duration.
         *
         * @param {Date} startDate - start date native JS object.
         * @param {number} duration - can be a floated number.
         * @param {string} oldDisplayOption - previous display option, can be 'MONTH' or 'DAY'.
         * @returns {number} week duration, can be floated number with 2 decimal digits.
         */
        convertToWeekDuration: function(startDate, duration, oldDisplayOption) {
            let durationInMs = 0;
            let currentDate = new Date(startDate);

            if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.MONTH) {
                while (duration > 0) {
                    const currentMonthMs = RM.DateUtilities.getMillisecondsInMonth(currentDate);
                    // If the remaining duration is less than a full month
                    if (duration < 1) {
                        durationInMs += duration * currentMonthMs;
                        break;
                    } else {
                        // Add full month duration
                        durationInMs += currentMonthMs;
                        duration -= 1;
                        currentDate.setMonth(currentDate.getMonth() + 1);
                    }
                }
            } else if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.DAY) {
                durationInMs = duration * RM.DateUtilities.MILLISECONDS_A_DAY;
            }
            // Convert milliseconds to weeks
            return durationInMs / RM.DateUtilities.MILLISECONDS_A_WEEK;
        },

        /**
         * Get month duration by a start date and week duration.
         *
         * @param {Date} startDate - start date native JS object
         * @param {number} duration - can be a floated number
         * @param {string} oldDisplayOption - previous display option, can be 'WEEK' or 'DAY'
         * @returns {number} month duration, can be floated number with 2 decimal digits.
         */
        convertToMonthDuration: function(startDate, duration, oldDisplayOption) {
            let durationInMs = 0;
            if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.WEEK) {
                durationInMs = duration * RM.DateUtilities.MILLISECONDS_A_WEEK;
            } else if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.DAY) {
                durationInMs = duration * RM.DateUtilities.MILLISECONDS_A_DAY;
            }
            let currentMonth = new Date(startDate);
            let totalMonths = 0;
            while (durationInMs > 0) {
                const currentMonthMs = RM.DateUtilities.getMillisecondsInMonth(currentMonth);
                if (durationInMs >= currentMonthMs) {
                    // If the remaining duration can cover a full month
                    totalMonths += 1;
                    durationInMs -= currentMonthMs;
                    currentMonth.setMonth(currentMonth.getMonth() + 1);
                } else {
                    // If there is less than a full month left
                    totalMonths += durationInMs / currentMonthMs;
                    durationInMs = 0;
                }
            }
            return totalMonths;
        },

        /**
         * Get day duration by a start date, duration and previous displayOption
         *
         * @param {Date} startDate - start date native JS object.
         * @param {number} duration - can be a floated number.
         * @param {string} oldDisplayOption - previous display option, can be 'MONTH' or 'WEEK'.
         * @returns {number} week duration, can be floated number with 2 decimal digits.
         */
        convertToDayDuration: function(startDate, duration, oldDisplayOption) {
            let durationInMs = 0;
            let currentDate = new Date(startDate);

            if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.MONTH) {
                while (duration > 0) {
                    const currentMonthMs = RM.DateUtilities.getMillisecondsInMonth(currentDate);
                    if (duration < 1) {
                        durationInMs += duration * currentMonthMs;
                        break;
                    } else {
                        durationInMs += currentMonthMs;
                        duration -= 1;
                        currentDate.setMonth(currentDate.getMonth() + 1);
                    }
                }
            } else if (oldDisplayOption === Roadmap.TIMELINE_DISPLAY_OPTION.WEEK) {
                durationInMs = duration * RM.DateUtilities.MILLISECONDS_A_WEEK;
            }
            // Convert milliseconds to days
            return durationInMs / RM.DateUtilities.MILLISECONDS_A_DAY;
        },
    };

    RM.FieldUtilities = {
        /**
         * Fix some issues for AUI date picker
         *
         * @param {$container} find date pickers in container
         */
        fixDatePickerFields: function($container) {
            // fix some issues relate to AUI datepicker by apply datepicker-patch.css
            // fix datepicker doesn't fire change when AUI versions which < 5.4.5
            var $datepickers = $container.find('input[data-aui-dp-uuid]');
            $datepickers.each(function(index, element) {
                var $dp = $(element),
                    uuid = $dp.attr('data-aui-dp-uuid');
                $dp.on('click focus', function() {
                    var $dpPopup = $('[data-aui-dp-popup-uuid=' + uuid + ']');
                    if (AJS.version <= '5.4.5') {
                        var defaultOnSelectHandler = $dpPopup.datepicker('option', 'onSelect');
                        $dpPopup.datepicker('option', 'onSelect', function(dateText, inst) {
                            defaultOnSelectHandler(dateText, inst);
                            $dp.change();
                        });
                    }
                    $dpPopup.parents('.aui-inline-dialog').addClass('datepicker-roadmap-patch');
                });
            });
        }
    };
}(AJS.$, window._));
