define("confluence-ui-components/js/confluence-select", [
    "jquery",
    "ajs",
], function ($, AJS) {
    "use strict";
    /**
       * Can be used instead of auiSelect2 if you need to add a11y fetures like:
       * - reading with AT (fixed labeling; changed attributes for focused inputs; added lots of ids and references to them for correct reading,
       *   for more details please check WCAG 1.3.1, 1.3.2, 4.1.2 or issues CONFSERVER-77305, CONFSERVER-79441, CONFSERVER-78098)
       * - improved keyboard accesibility (please check WCAG 2.1.1 or CONFSERVER-78106)
       *
       * Known issue: when .select2-no-results element rerenders, auiSelect2 has no aproriate event and AT tools don't know what happened,
       * we are missing the following call as a reaction:
       * $(".select2-no-results").each(function (index, item) {
       *     // getting AT users know if items were not found
       *     item.setAttribute("aria-live", "polite");
       * });
       * Coming soon. :)
       *
       * Please note you can also use confluence-select to call methods of the select2 jquery plugin by passing
       * the corresponding method name as the first argument.
       * For more details: https://bitbucket.org/atlassian/aui/src/3fef98c08c1d0d4f6cbabcbb39c669ed22fe699d/packages/core/src/js-vendor/jquery/plugins/jquery.select2.js#lines-3136
       *
       * @param {Object | string} options auiSelect2 options, will be applied to auiSelect2 call without any changes.
       *                                  If a string - a select2 method with the corresponding name will be called.
       * @param {Array} labelIds (optional) array of ids of elements that will be referred to select2 input via aria-labelledby attribute,
       *                         it means you can use multiple labels for your confluence-select, in addition <label for=... labeling approach
       *                         also supported, will be converted to aria-labelledby approach
       * @returns calls $.fn.auiSelect2 with options (https://aui.atlassian.com/aui/7.9/docs/auiselect2.html)
       */
    $.fn.confluenceSelect = function (options, labelIds) {
        if (typeof options === "string") {
            return $.fn.auiSelect2.apply(this, arguments);
        }

        if (!labelIds) {
            labelIds = [];
        }

        const SELECT2_CONTAINER_ID_PREFIX = "s2id_";

        const CS = "confluence-select-";
        const LISTBOX_ID_PREFIX = CS + "listbox-";
        const OPTION_ID_PREFIX = CS + "option-";
        const LABEL_ID_PREFIX = CS + "label-";
        const OPTION_CHOSEN_ID_PREFIX = OPTION_ID_PREFIX + "chosen-";
        const DESCRIPTION_ID_PREFIX = CS + "description-";

        const inputId = this.attr("id");
        if (!inputId) {
            throw "please set an id attribute to your confluenceSelect input for correct working";
        }

        const containerSelector = "#" + SELECT2_CONTAINER_ID_PREFIX + inputId;
        const listboxId = LISTBOX_ID_PREFIX + inputId;
        const descriptionId = DESCRIPTION_ID_PREFIX + inputId;

        // Appends hidden description to the select2 input for accessibility voiceover
        function appendHiddenDescription() {
            const $descriptionElement = $("#" + descriptionId);
            if (!$descriptionElement.length) {
                const $select2input = $(containerSelector + " input.select2-input");
                const $hiddenInstruction = $("<span hidden></span>");
                $hiddenInstruction.attr("id", descriptionId);
                const descriptionText = [];
                if (options.multiple) {
                    descriptionText.push(AJS.I18n.getText(
                        "confluence-ui-components.confluence-select.multiselect-description"
                    ));
                }
                descriptionText.push(AJS.I18n.getText(
                    "confluence-ui-components.confluence-select.filter-description"
                ));
                $hiddenInstruction.text(descriptionText.join(", "));
                $hiddenInstruction.addClass("assistive");
                $select2input.after($hiddenInstruction);
                const descriptionIds = [descriptionId];
                // Add any existing aria-describedby to be read after
                const existingAriaDescribedBy = $select2input.attr("aria-describedby");
                if (existingAriaDescribedBy) {
                    descriptionIds.push(existingAriaDescribedBy); 
                }
                $select2input.attr("aria-describedby", descriptionIds.join(" "));
            }
        }

        // event names were taken from here https://select2.github.io/select2/ Select2 3.5.3
        this.on("select2-focus", function () {
            // input from the hood of select2 has this selector
            const focusedInputClass = options.multiple
                ? ".select2-input"
                : ".select2-focusser";
            // get browser focus during tabbing
            const $select2Input = $(containerSelector + " input" + focusedInputClass);

            const select2InputId = $select2Input.attr("id");

            const labelId = LABEL_ID_PREFIX + select2InputId;
            const chosenId = OPTION_CHOSEN_ID_PREFIX + select2InputId;
            const labeledbyValue = labelIds.concat([labelId, chosenId]).join(" ");

            // label from different types of forms
            const $labelWithOriginalId = $("label[for=" + inputId + "]");
            const $labelWithSelect2Id = $("label[for=" + select2InputId + "]");
            const $label = $labelWithOriginalId.length
                ? $labelWithOriginalId
                : $labelWithSelect2Id;

            const $chosenOption = $(containerSelector + " span.select2-chosen");

            //add aria attributes to select2 input
            $select2Input.attr("role", "combobox");
            $select2Input.attr("aria-haspopup", "listbox");
            $select2Input.attr("aria-controls", listboxId);
            if (labelIds.includes('type-picker-custom')) {
                $select2Input.attr("aria-label", AJS.I18n.getText("confluence-ui-components.type-picker.ariaLabel"));
            } else {
                $select2Input.attr("aria-labelledby", labeledbyValue);
            }

            // add an id to refer it later
            $chosenOption.attr("id", chosenId);

            // Don't read label when offscreen
            // If label has content, its handled by aria-labelledby
            if ($label.hasClass("select2-offscreen")) {
                $label.attr("aria-hidden", "true");
            }
            $label.attr("id", labelId);

            appendHiddenDescription();
        });

        this.on("select2-loaded", function () {
            const $dropdown = $("#select2-drop .select2-results");
            const $selectItems = $(
                "#select2-drop .select2-result:not(.select2-result-unselectable)"
            );

            $dropdown.attr("role", "listbox");
            $dropdown.attr("id", listboxId);

            $selectItems.each(function (index, item) {
                const $item = $(item);
                $item.attr("role", "option");
                $item.attr("id", OPTION_ID_PREFIX + index);
            });
        });

        this.on("change", function () {
            //add keyboard support to 'X' button WCAG 2.1.1
            const $closeButtons = $(containerSelector + " .select2-search-choice-close");
            $closeButtons.attr("tabindex", 0);
            $closeButtons.attr("role", "button");
            $closeButtons.attr(
                "aria-label",
                AJS.I18n.getText("confluence-ui-components.confluence-select.deselect")
            );
        });

        this.on("select2-highlight", function () {
            const $dropdownInput = $("#select2-drop input.select2-input");
            const $select2input = $(containerSelector + " input.select2-input");
            const $input = $dropdownInput.length ? $dropdownInput : $select2input;
            const $highlightedOption = $("#select2-drop .select2-highlighted");

            //set reference for screen readers to active item
            $input.attr("aria-activedescendant", $highlightedOption.attr("id"));
        });

        this.on("select2-open", function () {
            const $dropdownInput = $("#select2-drop input.select2-input");
            const $select2input = $(containerSelector + " input.select2-input");
            const $inputAutocomplete = $dropdownInput.length ? $dropdownInput : $select2input;
            const $noResultsItem = $(".select2-no-results");

            // getting AT users know if items were not found
            $noResultsItem.attr("aria-live", "polite");

            //reset aria-activedescendant when dropdown opened
            $inputAutocomplete.attr("aria-activedescendant", '');
        });

        return $.fn.auiSelect2.call(this, options);
    };
});
