var Whitelist = window.Whitelist || {};

var DOC_URL = window.JIRA ?
    "https://confluence.atlassian.com/display/JIRA/Configuring+the+Whitelist" :
    (window.BAMBOO ?
        "https://confluence.atlassian.com/display/BAMBOO/Configuring+the+Allowlist" :
        "https://confluence.atlassian.com/display/DOC/Configuring+the+Allowlist");

(function ($, skate) {

    var Dialog2 = require('aui/dialog2');
    var Backbone = require('whitelist/lib/backbone');

    if (skate) {
        skate('whitelist-icon', {
            type: skate.types.CLASS,
            insert: function(element) {
                //after inserting to DOM we fetch favicon images if they are available, if not we use default class
                var $whitelistIcon = $(element);
                var iconUrl = $whitelistIcon.data('iconUrl');

                $whitelistIcon.removeClass(element.defaultIconClass);
                if (iconUrl) {
                    element.getImage(iconUrl);
                } else {
                    $whitelistIcon.addClass(element.defaultIconClass);
                }
            },
            prototype: {
                defaultIconClass: 'aui-icon aui-icon-small aui-iconfont-weblink',
                getImage: function(src) {
                    var $icon = $(this);
                    var img = new Image();

                    $icon.spin();

                    img.onload = function () {
                        $icon.css("background-image", "url(" + src + ")").spinStop();
                    };
                    img.onerror = function () {
                        $icon.addClass(this.defaultIconClass).spinStop();
                    }.bind(this);

                    img.src = src;
                }
            }
        });
    }


    function clearAuiMessageBar() {
        $("#aui-message-bar").empty();
    }

    function errorHandler(resp) {
        clearAuiMessageBar();
        if (resp.message === "This resource requires WebSudo.") {
            resp.message = AJS.I18n.getText("whitelist.ui.error.websudo");
        }
        AJS.messages.error({
            title: AJS.I18n.getText("whitelist.ui.error"),
            body: resp.message
        });
    }

    function createErrorResponse(errors) {
        return {
            errors: errors
        };
    }

    Whitelist.types = {
        "whitelist.application.link": {
            name: AJS.I18n.getText("whitelist.application.link"),
            test: $.noop
        },
        "whitelist.domain.name": {
            name: AJS.I18n.getText("whitelist.domain.name"),
            test: function (str) {
                return /^(https?:\/\/)?([a-zA-Z0-9\-\.]|\[[0-9a-fA-F:]+\])+(:[0-9]+)?$/.test(str);
            }
        },
        "whitelist.exact.url": {
            name: AJS.I18n.getText("whitelist.exact.url"),
            test: function (str) {
                return /^(https?:\/\/)?([a-zA-Z0-9\-\.]|\[[0-9a-fA-F:]+\])+(:[0-9]+)?([\/\w\-\.\?~&=%]*)*$/.test(str);
            }
        },
        "whitelist.wildcard.expression": {
            name: AJS.I18n.getText("whitelist.wildcard.expression"),
            test: function (str) {
                return /^[\d\[\]\w\/\.\-\?:~&=%\*]*$/.test(str) && ~str.indexOf("*");
            }
        },
        "whitelist.regular.expression": {
            name: AJS.I18n.getText("whitelist.regular.expression"),
            test: function (str) {
                try {
                    new RegExp(str);
                    return true;
                } catch (e) {
                    return false;
                }
            }
        }
    };

    var selectableTypes = _.omit(Whitelist.types, "whitelist.application.link");

    Whitelist.EntryModel = AJS.RestfulTable.EntryModel.extend({
        validate: function (attrs, options) {
            options = options || {};
            if (!options.unset && attrs.allowEdit !== false) {
                if (!attrs.expression) {
                    options.status = 400;
                    return createErrorResponse({
                        expression: AJS.I18n.getText("whitelist.ui.error.noexpression")
                    });
                } else if (Whitelist.types[attrs.type].test && !Whitelist.types[attrs.type].test(attrs.expression)) {
                    options.status = 400;
                    return createErrorResponse({
                        expression: AJS.I18n.getText("whitelist.ui.error.invalidexpression")
                    });
                }
            }
        },
        save: function (attrs, options) {
            if (attrs.expression && (attrs.type === "whitelist.exact.url" || attrs.type === "whitelist.domain.name") && !/:\/\//.test(attrs.expression)) {
                attrs.expression = "http://" + attrs.expression;
            }
            if (!this._validate(attrs, _.extend({}, options, {silent: false}))) return false;
            return AJS.RestfulTable.EntryModel.prototype.save.apply(this, arguments);
        },
        changedAttributes: function (attributes) {
            var changed = {},
                current = this.toJSON();
            $.each(attributes, function (name, value) {
                if (current[name] === undefined) {
                    if (typeof value === "string") {
                        if ($.trim(value) !== "") {
                            changed[name] = value;
                        }
                    } else if ($.isArray(value)) {
                        if (value.length !== 0) {
                            changed[name] = value;
                        }
                    } else {
                        changed[name] = value;
                    }
                } else if (current[name] !== undefined && current[name] !== value) {
                    if (typeof value === "object") {
                        if (!_.isEqual(value, current[name])) {
                            changed[name] = value;
                        }
                    } else {
                        changed[name] = value;
                    }
                }
            });
            if (!_.isEmpty(changed)) {
                this.addExpand(changed);
                return changed;
            }
        }
    });

    Whitelist.ExpressionEditView = Backbone.View.extend({
        render: function (data) {
            if (this.model.get("allowEdit") !== false) {
                return Whitelist.expressionColumnEdit(data);
            } else {
                return new Whitelist.ExpressionReadView({
                    model: this.model
                }).render(data);
            }
        }
    });

    Whitelist.ExpressionReadView = Backbone.View.extend({
        render: function (data) {
            data.iconUrl = this.model.get("iconUrl");
            var $el = $(Whitelist.expressionColumnRead(data));
            return $el;
        }
    });

    Whitelist.TypeEditView = Backbone.View.extend({
        render: function (data) {
            if (this.model.get("allowEdit") !== false) {
                return aui.form.select({
                    name: data.name,
                    options: _.map(selectableTypes, function (value, key) {
                        return {
                            text: value.name,
                            value: key,
                            selected: data.value === key
                        }
                    }),
                    extraClasses: "full-width-field"
                });
            } else {
                return new Whitelist.TypeReadView({
                    model: this.model
                }).render(data);
            }
        }
    });

    Whitelist.TypeReadView = Backbone.View.extend({
        render: function (data) {
            return Whitelist.typeColumnRead({
                value: Whitelist.types[data.value].name
            });
        }
    });

    Whitelist.AllowInboundEditView = Backbone.View.extend({
        tagName: "span",
        className: "whitelist-checkbox-container",
        events: {
            "change [type=checkbox]": "updateHiddenField"
        },
        template: Whitelist.allowInboundColumnEdit,
        render: function (data) {
            data.value = !!data.value;
            this.$el.html(this.template(data));
            return this.$el;
        },
        updateHiddenField: function (e) {
            this.$("[name=allowInbound]").val(e.target.checked);
        }
    });

    Whitelist.AllowInboundReadView = Backbone.View.extend({
        render: Whitelist.allowInboundColumnRead
    });

    Whitelist.AllowAnonymousUserEditView = Backbone.View.extend({
        tagName: "span",
        className: "whitelist-checkbox-container",
        events: {
            "change [type=checkbox]": "updateHiddenField"
        },
        template: Whitelist.allowAnonymousUserColumnEdit,
        render: function (data) {
            data.value = !!data.value;
            this.$el.html(this.template(data));
            return this.$el;
        },
        updateHiddenField: function (e) {
            this.$("[name=allowAnonymousUser]").val(e.target.checked);
        }
    });

    Whitelist.AllowAnonymousUserReadView = Backbone.View.extend({
        render: Whitelist.allowAnonymousUserColumnRead
    });

    Whitelist.RowView = AJS.RestfulTable.Row.extend({
        initialize: function () {
            AJS.RestfulTable.Row.prototype.initialize.apply(this, arguments);
            this.allowEdit = this.allowEdit && this.model.get("allowEdit");
            this.allowDelete = this.allowDelete || this.model.get("allowDelete");
        }
    });

    Whitelist.EditRowView = AJS.RestfulTable.EditRow.extend({
        events: _.extend({
            "change .select": "handleTypeChange",
            "change #allowInbound": "handleInboundChange"
        }, AJS.RestfulTable.EditRow.prototype.events),
        handleTypeChange: function (e) {
            var expression = this.$(".text").val();
            var error = this.$(".whitelist-error");

            if (expression) {
                error.toggleClass("hidden", !this.model.validate({
                    expression: expression,
                    type: $(e.target).val()
                }));
            } else {
                error.addClass("hidden");
            }
        },
        // If inbound is allowed, anonymous users must be allowed
        handleInboundChange: function() {
            var inboundCheckbox = this.$('#allowInbound');
            var allowAnonymousCheckbox = this.$('#allowAnonymousUser');

            if (inboundCheckbox.is(":checked")) {
                allowAnonymousCheckbox.prop("checked", true);
                allowAnonymousCheckbox.prop("disabled", true);
                // this will trigger a change to the hidden input
                allowAnonymousCheckbox.trigger("change");
            } else {
                allowAnonymousCheckbox.prop("disabled", false);
            }
        },
        render: function () {
            AJS.RestfulTable.EditRow.prototype.render.apply(this, arguments);
            this.$(".select").trigger("change");
            this.handleInboundChange();
            return this;
        }
    });

    Whitelist.ToggleView = Backbone.View.extend({
        events: {
            "click": "handleClick"
        },
        initialize: function (options) {
            this.$table = options.$table;
            this.enabled = this.$el.data("enabled");
        },
        handleClick: function () {
            if (this.enabled) {
                var that = this;
                var template = Whitelist.disableConfirmation();
                var dialog = new Dialog2(template).show();
                $('#disable-confirm').on('click', function (e) {
                    e.preventDefault();
                    that.toggleWhitelist();
                    dialog.hide();
                    dialog.remove();
                });
                $('#disable-cancel').on('click', function (e) {
                    e.preventDefault();
                    dialog.hide();
                    dialog.remove();
                });
            } else {
                this.toggleWhitelist();
            }
        },
        toggleWhitelist: function () {
            this.enabled = !this.enabled;
            this.trigger(this.enabled ? "enable" : "disable");
            this.$el.text(this.enabled ? AJS.I18n.getText("whitelist.ui.whitelist.disable") : AJS.I18n.getText("whitelist.ui.whitelist.enable"));
            $.ajax({
                url: AJS.contextPath() + "/rest/whitelist/latest/" + (this.enabled ? "enable" : "disable"),
                type: "POST",
                dataType: "json",
                contentType: "application/json"
            }).done(_.bind(function () {
                clearAuiMessageBar();
                AJS.messages.success({
                    body: this.enabled ? AJS.I18n.getText("whitelist.ui.whitelist.enabled") : AJS.I18n.getText("whitelist.ui.whitelist.disabled"),
                    fadeout: true
                });
            }, this)).fail(function (xhr) {
                errorHandler(JSON.parse(xhr.responseText));
            });
        }
    });

    Whitelist.SettingsDescriptionView = Backbone.View.extend({
        disable: function () {
            this.$el.addClass("hidden");
        },
        enable: function () {
            this.$el.removeClass("hidden");
        }
    })

    Whitelist.SettingsDialogView = Backbone.View.extend({
        events: {
            "click button": "handleOpenClick"
        },
        initialize: function(options) {
            this.rulesTable = options.rulesTable;
        },
        handleOpenClick: function() {
            var dialogTemplate = Whitelist.defaultSettings({data: this.model.getAttributes(), contextPath: AJS.contextPath()});
            this.dialog = new Dialog2(dialogTemplate).show();

            $('#settings-form').on('submit', this.handleFormSubmit.bind(this));
            $('#settings-cancel').on('click', this.handleCancelClick.bind(this));
        },
        handleFormSubmit: function(e) {
            e.preventDefault();
            this.closeDialog();

            var formData = $(e.target).serializeObject();

            this.model
                .save(formData)
                .done(function() {
                    clearAuiMessageBar();
                    AJS.messages.success({
                        body: AJS.I18n.getText("whitelist.ui.whitelist.settings.dialog.success"),
                        fadeout: true
                    });
                    // reset the rules restful table to reflect changes to the rules from backend
                    this.rulesTable.reset();
                }.bind(this))
                .fail(function(xhr, status, error) {
                    errorHandler({message: error});
                });
        },
        handleCancelClick: function(e) {
            e.preventDefault();
            this.closeDialog();
        },
        closeDialog: function() {
            if (!this.dialog) {
                return;
            }

            this.dialog.hide();
            this.dialog.remove();
            delete this.dialog;
        },
        disable: function () {
            this.$el.addClass("hidden");
        },
        enable: function () {
            this.$el.removeClass("hidden");
        }
    });

    Whitelist.SettingsDialogModel = Backbone.Model.extend({
        url: AJS.contextPath() + "/rest/whitelist/latest/settings",
        getAttributes: function() {
            // https://backbonejs.org/#Model-attributes
            return _.clone(this.attributes);
        },
        save: function(data) {
            return $.ajax({
                type: "PUT",
                contentType: "application/json",
                dataType: "json",
                url: this.url,
                data: JSON.stringify(data)
            }).done(function() {
                this.set(data);
            }.bind(this))
        }
    })

    Whitelist.CheckerView = Backbone.View.extend({
        events: {
            "submit": "handleSubmit",
            "focus .whitelist-checker-field": "handleFocus",
            "blur .whitelist-checker-field": "handleBlur",
            "input .whitelist-checker-field": "handleInput"
        },
        initialize: function () {
            this.container = this.$el.parent();
            this.$spinner = this.$(".whitelist-checker-spinner");
            this.$inbound = this.$(".whitelist-inbound");
            this.$outbound = this.$(".whitelist-outbound");
            this.$allowAnonymous = this.$(".whitelist-allow-anonymous");
            this.$icons = this.$spinner.add(this.$inbound).add(this.$outbound);
            this.$spinner.tooltip({
                gravity: 'se'
            });
            this.$inbound.add(this.$outbound).tooltip({
                gravity: 'w'
            });
            this.collection.on("add remove change", _.bind(this.submit, this));
        },
        handleSubmit: function (e) {
            e.preventDefault();
            this.submit();
        },
        successTemplate: Whitelist.icon({
            useIconFont: true,
            icon: "approve"
        }),
        failTemplate: Whitelist.icon({
            useIconFont: true,
            icon: "error"
        }),
        getAllowAnonymousUser: function(data) {
            if (!data) {
                return "";
            }

            var isAllowed = data.outboundWithAuth && data.outbound;

            return isAllowed ? AJS.I18n.getText("whitelist.ui.checker.allowAnonymous.yes") : AJS.I18n.getText("whitelist.ui.checker.allowAnonymous.no");
        },
        submit: _.debounce(function () {
            var params = this.$el.serializeObject();
            this.$icons.empty();
            this.$allowAnonymous.text("");
            if (params.url && params.url !== "http://") {
                if (this.$spinner.spin) {
                    this.$spinner.spin();
                }
                $.get(AJS.contextPath() + "/rest/whitelist/latest/check", params)
                    .done(_.bind(function (data) {
                        if (this.$spinner.spinStop) {
                            this.$spinner.spinStop();
                        }
                        this.$inbound.html(data.inbound ? this.successTemplate : this.failTemplate);
                        this.$outbound.html(data.outboundWithAuth ? this.successTemplate : this.failTemplate);
                        this.$allowAnonymous.text(this.getAllowAnonymousUser(data));
                        if (data.inbound) {
                            this.$inbound.removeAttr("original-title");
                        } else {
                            this.$inbound.attr("original-title", AJS.I18n.getText("whitelist.ui.checker.restricted.inbound"));
                        }
                        if (data.outboundWithAuth) {
                            this.$outbound.removeAttr("original-title");
                        } else {
                            this.$outbound.attr("original-title", AJS.I18n.getText("whitelist.ui.checker.restricted.outbound"));
                        }
                    }, this))
                    .fail(_.bind(function (xhr) {
                        this.$spinner.html(this.failTemplate).attr("original-title", JSON.parse(xhr.responseText).message);
                    }, this));
            }
        }, 500, true),
        handleFocus: function (e) {
            if (!e.target.value) {
                e.target.value = "http://";
            }
        },
        handleBlur: function (e) {
            if (e.target.value === "http://") {
                e.target.value = "";
            }
        },
        handleInput: _.debounce(function () {
            this.$el.submit();
        }, 500),
        disable: function () {
            this.container.addClass("hidden");
        },
        enable: function () {
            this.container.removeClass("hidden");
        }
    });

    Whitelist.RestfulTable = AJS.RestfulTable.extend({
        disable: function () {
            this.$el.addClass("hidden");
        },
        enable: function () {
            this.$el.removeClass("hidden");
        },
        // Reset allows to delete the collection in memory, and recreate it by fetching all resources from API.
        reset: function() {
            this.getRows().forEach(this.removeRow.bind(this));
            this.fetchInitialResources();
        }
    });

    $(document).on(AJS.RestfulTable.Events.SERVER_ERROR, function (e, resp) {
        errorHandler(resp);
    });

    $(function () {
        $("#whitelist-doc-link").attr("href", DOC_URL);

        var restfulTable = new Whitelist.RestfulTable({
            el: "#whitelist-table",
            deleteConfirmationCallback: function (model) {
                var template = Whitelist.deleteConfirmation({expression: model.expression});
                var dialog = new Dialog2(template).show();
                return new Promise(function (resolve, reject) {
                    $('#delete-confirm').on('click', function (e) {
                        resolve();
                        e.preventDefault();
                        dialog.hide();
                        dialog.remove();
                    });
                    $('#delete-cancel').on('click', function (e) {
                        reject();
                        e.preventDefault();
                        dialog.hide();
                        dialog.remove();
                    });
                });
            },
            resources: {
                all: function (callback) {
                    var initialDataElement = $("#whitelist-data");
                    // if first render, the server renders the initial data as a data attribute of a DIV element.
                    if (initialDataElement && initialDataElement.data("whitelist")) {
                        callback(initialDataElement.data("whitelist"));
                        // remove the DIV element with the initial data. Next time, fetch it from the API.
                        initialDataElement.remove();
                    } else {
                        // else, get the newest data from API.
                        $.get(AJS.contextPath() + "/rest/whitelist/latest/")
                            .done(function (entries) {
                                callback(entries.rules);
                            })
                            .fail(function(xhr, status, error) {
                                callback([]);
                                errorHandler({message: error});
                            });
                    }
                },
                self: AJS.contextPath() + "/rest/whitelist/latest/"
            },
            columns: [
                {
                    id: "type",
                    header: Whitelist.typeHeader,
                    editView: Whitelist.TypeEditView,
                    readView: Whitelist.TypeReadView
                },
                {
                    id: "expression",
                    header: AJS.I18n.getText("whitelist.ui.header.expression"),
                    editView: Whitelist.ExpressionEditView,
                    readView: Whitelist.ExpressionReadView
                },
                {
                    id: "allowInbound",
                    header: Whitelist.allowInboundHeader,
                    editView: Whitelist.AllowInboundEditView,
                    readView: Whitelist.AllowInboundReadView,
                    allowEdit: false
                },
                {
                    id: "allowAnonymousUser",
                    header: Whitelist.allowAnonymousUserHeader,
                    editView: Whitelist.AllowAnonymousUserEditView,
                    readView: Whitelist.AllowAnonymousUserReadView,
                    allowEdit: false
                }
            ],
            model: Whitelist.EntryModel,
            views: {
                row: Whitelist.RowView,
                editRow: Whitelist.EditRowView
            }
        });

        var checker = new Whitelist.CheckerView({
            el: "#whitelist-checker",
            collection: restfulTable._models
        });

        var settingsDescriptionView = new Whitelist.SettingsDescriptionView({
            el: "#whitelist-settings-description"
        });

        var settingsModel = new Whitelist.SettingsDialogModel($('#whitelist-data-settings').data('settings'));
        var settingsView = new Whitelist.SettingsDialogView({
            el: "#whitelist-settings",
            model: settingsModel,
            rulesTable: restfulTable
        });

        new Whitelist.ToggleView({
            el: "#whitelist-enable"
        }).on("disable", _.bind(checker.disable, checker))
          .on("disable", _.bind(restfulTable.disable, restfulTable))
          .on("disable", _.bind(settingsView.disable, settingsView))
          .on("disable", _.bind(settingsDescriptionView.disable, settingsDescriptionView))
          .on("enable", _.bind(checker.enable, checker))
          .on("enable", _.bind(restfulTable.enable, restfulTable))
          .on("enable", _.bind(settingsView.enable, settingsView))
          .on("enable", _.bind(settingsDescriptionView.enable, settingsDescriptionView));
    });
})(AJS.$, window.skate);
