define("tc/event-edit-dialog",
    [
        "jquery",
        "tc/event-field-handlers",
        "tc/event-types",
        "tc-backbone",
        "tc/calendar-util",
        "tc/form-state-control",
        "confluence/api/logger",
        "tc/update-event",
        "tc/event-edit-analytics"
    ],
    function (
        $,
        eventFieldHandlers,
        EventTypes,
        Backbone,
        CalUtil,
        FormStateControl,
        Logger,
        UpdateEvent,
        eventEditAnalytics
    )
    {
       "use strict";
        var CalendarPlugin,
        calendarDiv,
        dialog,
        okButtonStates = {};

        var EventEditDialog = Backbone.View.extend({
            disableCalSelect : function() {
                if ($.fn.auiSelect2) {
                    $("select#calendar").auiSelect2("disable");
                } else {
                    $("select#calendar").prop("disabled", true);
                }

                if(AJS.Meta.get("build-number") >= 4520) {
                    $('#event-type').auiSelect2("enable", false);
                } else {
                    $('#event-type').auiSelect2("disable");
                }
            },

            enableCalSelect : function() {
                if ($.fn.auiSelect2) {
                    $("select#calendar").auiSelect2("enable");
                } else {
                    $("select#calendar").prop("disabled", false);
                }

                if(AJS.Meta.get("build-number") >= 4520) {
                   $('#event-type').auiSelect2("enable", true);
                } else {
                   $('#event-type').auiSelect2("enable");
                }
            },

            showToolTip : function() {
                $("#edit-event-dialog .subcalendar-event select.event-type-select").css('display','none');
                $("#calendar-tooltip.icon-help").removeClass("hidden").tooltip({aria:true});
            },

            hideToolTip: function() {
                $("#calendar-tooltip.icon-help").addClass("hidden");
            },

            getField : function(fieldsDiv, fieldName) {
                return $("input[name='" + fieldName + "']", fieldsDiv);
            },

            getSelect : function(fieldsDiv, fieldName) {
                return $("select[name='" + fieldName + "']", fieldsDiv);
            },

            getTextSelect : function(fieldsDiv, fieldName) {
                return $("select[name='" + fieldName + "'] option:selected", fieldsDiv).text();
            },

            getTextArea : function(fieldsDiv, fieldName) {
                return $("textarea[name='" + fieldName + "']", fieldsDiv);
            },

            show : function() {
                dialog.show();
            },

            hide : function() {
                dialog.hide();
            },

            remove : function() {
                dialog.remove();
            },

            /**
             * This is the public interface. This function returns a edit event dialog.
             *
             * event : The event to edit. If creating a new event just create an object with the appropriate date fields set.
             *
             * subCalendar : The subCalendar the event belong to. If null the event will be associated with the first calendar
             * in the list of users calendars alphabetically.
             *
             * _CalendarPlugin : A reference to the CalendarPlugin object.
             *
             * _calendarDiv : The div of the calendar we are editing.
             */
            getEditEventDialog : function(event, subCalendar, _CalendarPlugin, _calendarDiv) {
                var eventEditForm,
                    submit,
                    cancel,
                    dialogPanel,
                    eventTypes,
                    fieldHandlersForType = [],
                    okButton,
                    localStorage = Confluence.TeamCalendars.LocalStorage,
                    originalEventSubCalendarId = event.originalEventSubCalendarId,
                    eventEditDialog = this;

                var width = 580, height = 700;

                var isReadonlyEvent = event && event.eventType === "jira" && event.extraProperties
                    && event.extraProperties.startDateFieldEditable === "false"
                    && (event.end === null || event.extraProperties.endDateFieldEditable === "false");


                //Sometimes the event to create the dialog is fired multiple times. If we are already showing a dialog return it.
                if (dialog && dialog.isVisible()) {
                    return dialog;
                }

                //This should never happen as we destroy the dialog when you close it.
                //But just in case log a warning and kill the existing dialog.
                if (dialog) {
                    dialog.remove();
                    dialog = null;
                    AJS.log("Warning: old event edit dialog was not destroyed before creating a new one.");
                }

                CalendarPlugin = _CalendarPlugin;
                calendarDiv = _calendarDiv;

                eventTypes = EventTypes.DefaultEventType;

                //Called when the user submits the event edit dialog. Collects and massages data from the form and then calls
                //CalendarPlugin.updateEvent to send the data to the server. Also handles errors that come back from the server.
                function submitForm(fieldsDiv, fieldHandlers) {
                    var originalSubCalendarId,
                        originalStartDate,
                        subCalendarId,
                        uid,
                        allDayEvent,
                        until,
                        eventTypeName,
                        eventType,
                        originalEventType,
                        originalCustomEventTypeId,
                        customEventTypeId,
                        childSubCalendarId,
                        isSingleJiraDate,
                        rruleStr;

                    if (CalendarPlugin.isProcessingEvent(calendarDiv)) {
                        return false;
                    }

                    CalendarPlugin.setProcessingEvent(calendarDiv, true, eventEditForm);

                    originalSubCalendarId = eventEditDialog.getField(dialogPanel, "originalSubCalendarId").val();
                    childSubCalendarId = eventEditDialog.getField(dialogPanel, "childSubCalendarId").val();
                    originalStartDate = eventEditDialog.getField(dialogPanel, "originalStartDate").val();
                    subCalendarId = $("option:selected", eventEditDialog.getSelect(dialogPanel, "calendar")).val();
                    uid = eventEditDialog.getField(dialogPanel, "uid").val();
                    eventTypeName = eventEditDialog.getSelect(dialogPanel, "event-type").val();
                    eventType = CalUtil.isCustomEventType(eventTypeName) ? EventTypes.CustomEventType : eventTypes[eventTypeName];
                    allDayEvent = eventEditDialog.getField(dialogPanel, "allDayEvent").is(":checked");
                    until = eventEditDialog.getField(dialogPanel, "until");
                    originalEventType = eventEditDialog.getField(dialogPanel, "originalEventType").val();
                    originalCustomEventTypeId = eventEditDialog.getField(dialogPanel, "originalCustomEventTypeId").val();
                    customEventTypeId = CalUtil.isCustomEventType(eventTypeName) ? eventTypeName : "";
                    isSingleJiraDate = event && (event.eventType === "jira-project-releases" || (event.eventType === "jira" && event.end == null));
                    //save calendarSelect and eventType to cookie
                    localStorage.setItem("last-selected-calendar", subCalendarId);
                    localStorage.setItem("last-selected-calendar-type", eventTypeName);

                    if (!uid) { //We are creating a new event
                        calendarDiv.data(CalendarPlugin.PREF_LAST_ALL_DAY_USED, allDayEvent.toString());
                    }

                    if(!eventType) {
                        return;
                    }

                    //Some event types, e.g. JIRA send something totally different to the server for historical reasons. So we
                    //let them have their own handler to submit the form.
                    if (eventType.customHandler) {
                        eventType.customHandler.submitForm(
                                fieldsDiv,
                                eventEditDialog.getField(fieldsDiv, "name").val(),
                                subCalendarId,
                                CalendarPlugin.getRenderedMacroCallbackHandler(calendarDiv),
                                function() {
                                    dialog.remove();
                                    CalendarPlugin.setProcessingEvent(calendarDiv, false, eventEditForm);
                                    CalendarPlugin.updateAvailableSubCalendarsInSubCalendarPanel(calendarDiv);
                                    eventEditAnalytics.analyticEventTrigger(calendarDiv, {rruleLength: 0}, uid, eventType);
                                },
                                function() {
                                    CalendarPlugin.setProcessingEvent(calendarDiv, false, eventEditForm);
                                });
                        return;
                    }

                    var eventType = CalUtil.isCustomEventType(eventTypeName) ? "custom" : eventTypeName;
                    var data = {
                        //These fields are always on the form.
                        childSubCalendarId: childSubCalendarId, //need hold this childSubCalendar for purpose check exist JIRA event type,
                        customEventTypeId: customEventTypeId,
                        eventType : eventType,
                        isSingleJiraDate: isSingleJiraDate,
                        originalSubCalendarId : originalSubCalendarId,
                        originalEventSubCalendarId : originalEventSubCalendarId,
                        originalStartDate : originalStartDate,
                        originalEventType: originalEventType,
                        originalCustomEventTypeId: originalCustomEventTypeId,
                        recurrenceId: event.recurId || "",
                        subCalendarId : subCalendarId,
                        uid : uid
                    };

                    UpdateEvent.updateEvent(CalendarPlugin, calendarDiv, fieldsDiv, eventEditForm, fieldHandlers, dialog, data);

                    return false;
                }

                //Helper function to add and populate the calendar field.
                function addCalendarField(dialogPanel, event) {
                    var updatableSubCalendars = CalendarPlugin.getSubCalendarsWhichCanAddEvents(calendarDiv);
                    updatableSubCalendars.sort(function(leftSubCalendar, rightSubCalendar) {
                        var leftSubCalendarName = leftSubCalendar.name.toLowerCase(), rightSubCalendarName = rightSubCalendar.name.toLowerCase();
                        return leftSubCalendarName.localeCompare(rightSubCalendarName);
                    });

                    var calendarSelect = eventEditDialog.getSelect(dialogPanel, "calendar");
                    $.each(updatableSubCalendars, function(_subCalendarIdx, _aSubCalendar) {
                        calendarSelect.append($("<option/>", { "value" :_aSubCalendar.id, "text" : _aSubCalendar.name}));
                    });

                    if ($.fn.auiSelect2) {
                        Logger.log("Enable select2 for calendar selection");
                        calendarSelect.auiSelect2({
                            minimumResultsForSearch: -1
                        });
                    }

                    //hidden the calendar field if just subscribed to one calendar.
                    if (updatableSubCalendars.length === 1) {
                        calendarSelect.parent().hide();
                    }

                    var setSelectedCalendar = function (subCalendarId){
                        // don't select the subCalendarId if it's not one of the subscribed calendars, it won't exist in the selector
                        if (calendarSelect.find("option[value='" + subCalendarId + "']").length > 0) {
                            Logger.log("Enable select2 for calendar selection " + subCalendarId);
                            if ($.fn.auiSelect2) {
                                calendarSelect.auiSelect2('val', subCalendarId);
                            } else {
                                calendarSelect.val(subCalendarId);
                            }
                        }
                    }

                    if (event.subCalendarId) {
                        var subCalendar = CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId);
                        if (subCalendar) {
                            var subCalendarId = subCalendar.parentId || event.subCalendarId;
                            setSelectedCalendar(subCalendarId);
                        }
                    } else {
                        //set default for calendar from cookie
                        var calendarSelectedId = localStorage.getItem("last-selected-calendar");
                        if(calendarSelectedId) {
                            Logger.log("Enable select2 for last calendar selection " + calendarSelectedId);
                            setSelectedCalendar(calendarSelectedId)
                        }
                    }

                    if ((CalUtil.isJiraEvent(event) || CalUtil.isJiraEventStream(event.eventType))) {
                        FormStateControl.disableElement(calendarSelect);
                    }

                    //change calendar option
                    calendarSelect.change( function() {
                        var originalValueSubCalendarId = eventEditDialog.getField(dialogPanel, "originalSubCalendarId").val();
                        var originalValueEventType = eventEditDialog.getField(dialogPanel, "originalEventType").val();
                        if (updatableSubCalendars.length > 1  && originalValueSubCalendarId !== "" && originalValueEventType !== "") {
                            var optionSelected = $(this).find("option:selected");
                            var typeValue = eventEditDialog.getSelect(dialogPanel, "event-type").val();
                            if(originalValueSubCalendarId === optionSelected.val() && typeValue === originalValueEventType) {
                                FormStateControl.enableElement($("#editthisinstanceonly"));
                            } else {
                                FormStateControl.disableElement($("#editthisinstanceonly"));
                            }
                        }
                        addEventTypeField(dialogPanel, true);
                    });
                }

                function formatEventType(type) {
                    var classIcon = CalUtil.isCustomEventType(type.id) ? type.css : type.id;
                    var iconTemplate = Confluence.TeamCalendars.Templates.renderCustomImagePrinting({eventType: classIcon});
                    var eventTypeItem = $("<div/>", {"class": "tc-event-type-item", "title": type.text})
                        .append(iconTemplate)
                        .append($("<span/>", { "class":"ellipsis_text", "text": type.text }));
                    return $("<div/>").append(eventTypeItem).html();
                }

                function getListDisableEvent(calendarId) {
                    var listCalendarObject = CalendarPlugin.getSubCalendarsWhichCanAddEvents(calendarDiv);
                    for (var item in listCalendarObject) {
                       if(listCalendarObject[item].id === calendarId) {
                           return listCalendarObject[item].disableEventTypes;
                       }
                    }
                }

                function getListCustomEventType(calendarId) {
                     var listCalendarObject = CalendarPlugin.getSubCalendarsWhichCanAddEvents(calendarDiv);
                     for (var item in listCalendarObject) {
                        if(listCalendarObject[item].id === calendarId) {
                            return listCalendarObject[item].customEventTypes;
                        }
                     }
                }

                function getOriginalCustomEventType(originalValueSubCalendarId, customEventTypeId) {
                    var originalCustomEventType;
                    if(!customEventTypeId) {
                        return originalCustomEventType;
                    }

                    var listOriginalCustomEventType = getListCustomEventType(originalValueSubCalendarId);
                    if(listOriginalCustomEventType && listOriginalCustomEventType.length) {
                        for(var i =0; i<listOriginalCustomEventType.length; i++) {
                            if(listOriginalCustomEventType[i].customEventTypeId === customEventTypeId) {
                                originalCustomEventType = listOriginalCustomEventType[i];
                                break;
                            }
                        }
                    }
                    return originalCustomEventType;
                }

                function sortCustomEventType(listCustomEventTypes) {
                    listCustomEventTypes.sort(function(eventA, eventB) {
                       return eventA.title.toUpperCase().localeCompare(eventB.title.toUpperCase());
                    });

                    return listCustomEventTypes;
                }

                //Helper function to add and populate the event type field.
                function addEventTypeField(dialogPanel, isChangSubcalendar) {
                    var typeSelect = eventEditDialog.getSelect(dialogPanel, "event-type");

                    if(isChangSubcalendar) {
                        typeSelect.empty();
                    }

                    var originalValueEventType = eventEditDialog.getField(dialogPanel, "originalEventType").val();
                    var originalValueSubCalendarId = eventEditDialog.getField(dialogPanel, "originalSubCalendarId").val();
                    var calendarIdSelected = eventEditDialog.getSelect(dialogPanel, "calendar").find("option:selected").val();
                    var listDisableEvent = getListDisableEvent(calendarIdSelected);
                    var listCustomEventType = sortCustomEventType(getListCustomEventType(calendarIdSelected)); //sort list custom event type
                    var availableEventType = [];

                    var originalCustomEventType = getOriginalCustomEventType(originalValueSubCalendarId, event.customEventTypeId);

                    var isDisableEvent = function(eventType) {
                        for (var item in listDisableEvent) {
                            if (eventType === listDisableEvent[item])
                            {
                                return true;
                            }
                        }
                        return false;
                    };

                    //load event type when edit
                    if(event.id) {
                        $.each(
                            eventTypes,
                            function(i, type) {
                                if ((originalValueEventType === type.id || !isDisableEvent(type.id) && (CalUtil.isJiraEvent(event) || type.hiddenFromEventCreation === undefined))) {
                                    typeSelect.append($("<option/>", { "value" : type.id, "text" : type.name } ));
                                    availableEventType.push(type.id);
                                }
                            }
                        );

                        $.each(
                            listCustomEventType,
                            function(i, item){
                                if(!isDisableEvent(item.customEventTypeId)) {
                                    typeSelect.append($("<option/>", { "value": item.customEventTypeId, "class" : item.icon, "data-custom-id": item.customEventTypeId, "text" : item.title}));
                                    availableEventType.push(item.customEventTypeId);
                                }
                            }
                        );

                        //load custom event type
                        if(isChangSubcalendar && originalCustomEventType) {
                            var existCustomEventType = false;
                            for(var i = 0; i < listCustomEventType.length; i++) {
                                if(listCustomEventType[i].title === originalCustomEventType.title) {
                                    existCustomEventType = true;
                                    break;
                                }
                            }
                            if(!existCustomEventType) {
                                typeSelect.append($("<option/>", { "value": originalCustomEventType.customEventTypeId, "class" : originalCustomEventType.icon, "data-custom-id": originalCustomEventType.customEventTypeId, "text" : originalCustomEventType.title}));
                                availableEventType.push(originalCustomEventType.customEventTypeId);
                            }
                        }

                        //Disable OK button if don't have event type is enable
                        if (typeSelect.children().length > 0) {
                            $(".field-group-event-type .type-error", dialogPanel).empty();
                            $("#event-fields-container", dialogPanel).show();
                            eventEditDialog.enableOkButton();
                        } else {
                            typeSelect.append($("<option/>", { "value" : "", "text" : "" } )); //Hack for case all event type is disabled.
                            $(".field-group-event-type .type-error", dialogPanel).append($("<span/>", {"text": AJS.I18n.getText("calendar3.error.addevent.noeventtype")}));
                            $("#event-fields-container", dialogPanel).hide();
                            eventEditForm.disableOkButton({
                                "buttonText" : AJS.I18n.getText("ok")
                            });
                        }
                    } else {//load event type when create new event
                        //load list event type exclude jira type
                        $.each(
                            eventTypes,
                            function(i, type) {
                                if (!isDisableEvent(type.id) && (CalUtil.isJiraEvent(event) || type.hiddenFromEventCreation === undefined)) {
                                    typeSelect.append($("<option/>", { "value" : type.id, "text" : type.name } ));
                                    availableEventType.push(type.id);
                                }
                            }
                        );

                        //load custom event type
                        $.each(
                            listCustomEventType,
                            function(i, item){
                                if(!isDisableEvent(item.customEventTypeId)) {
                                    typeSelect.append($("<option/>", { "value": item.customEventTypeId, "class" : item.icon, "data-custom-id": item.customEventTypeId, "text" : item.title}));
                                    availableEventType.push(item.customEventTypeId);
                                }
                            }
                        );

                        //load list event jira type
                        $.each(
                            eventTypes,
                            function(i, type) {
                                if (type.hiddenFromEventCreation !== undefined && type.hiddenFromEventCreation === false && !isDisableEvent(type.id) ) {
                                    typeSelect.append($("<option/>", { "value" : type.id, "text" : type.name } ));
                                    availableEventType.push(type.id);
                                }
                            }
                        );

                        //Disable OK button if don't have event type is enable
                        if (typeSelect.children().length > 0) {
                            $(".field-group-event-type .type-error", dialogPanel).empty();
                            $("#event-fields-container", dialogPanel).show();
                            eventEditDialog.enableOkButton();
                        } else {
                            typeSelect.append($("<option/>", { "value" : "", "text" : "" } )); //Hack for case all event type is disabled.
                            $(".field-group-event-type .type-error", dialogPanel).append($("<span/>", {"text": AJS.I18n.getText("calendar3.error.addevent.noeventtype")}));
                            $("#event-fields-container", dialogPanel).hide();
                            eventEditForm.disableOkButton({
                                "buttonText" : isReadonlyEvent ? AJS.I18n.getText("ok") : AJS.I18n.getText("calendar3.save")
                            });
                        }

                        //add link custom event type
                        typeSelect.append($("<option/>", { "value": "addNewCustomEventType", "class" : "addNewCustomEventType", "text" :  AJS.I18n.getText("com.atlassian.confluence.extra.team-calendars.eventtypes.custom.addeventtype")}));
                    }

                    if(AJS.$.fn.auiSelect2){
                        $('.event-edit #event-type').auiSelect2({
                            formatSelection: formatEventType,
                            formatResult: formatEventType,
                            minimumResultsForSearch: -1
                        });

                        if(event.id && CalUtil.isJiraEvent(event)) {
                            //support new method disable auislect2 from confluence 5.3
                            if(AJS.Meta.get("build-number") >= 4520) {
                                $('#event-type').auiSelect2("enable", false);
                            } else {
                                $('#event-type').auiSelect2("disable");
                            }
                        }
                    }

                    if(event.eventType && event.eventType === "custom") {
                        typeSelect.change(handleEventTypeChange).val(event.customEventTypeId).trigger("change");
                    } else {
                        var selectedEventType = event.eventType || localStorage.getItem("last-selected-calendar-type");
                        if (!selectedEventType || selectedEventType === ""
                            || selectedEventType === "null") {
                            selectedEventType = "other";
                        }

                        if (!availableEventType.includes(selectedEventType) || isDisableEvent(selectedEventType)) {
                            // we allow to move event even though event type on target calendar is disable
                            if (event || !isDisableEvent("other")) {
                                selectedEventType = "other";
                            } else {
                                // select what ever type we are having
                                selectedEventType = typeSelect.val();
                            }
                        }

                        typeSelect.change(handleEventTypeChange).val(selectedEventType).trigger("change");
                    }
                }

                //Set hidden fields that are always present on the dialog.
                function setHiddenFields() {
                    eventEditDialog.getField(dialogPanel, "uid").val(event.id || "");
                    eventEditDialog.getField(dialogPanel, "originalStartDate").val(event.originalStart || "");
                    eventEditDialog.getField(dialogPanel, "originalSubCalendarId").val(event.subCalendarId || "");
                    eventEditDialog.getField(dialogPanel, "originalEventType").val(event.eventType || "");
                    eventEditDialog.getField(dialogPanel, "originalCustomEventTypeId").val(event.customEventTypeId || "");
                    eventEditDialog.getField(dialogPanel, "childSubCalendarId").val(event.childSubCalendarId || "");
                }

                //Handle event type change
                function handleEventTypeChange(event) {
                    if(!dialog) {
                        return;
                    }
                    var type = event.target.value;

                    if(type === "addNewCustomEventType") {
                        var subCalendarId = $("option:selected", eventEditDialog.getSelect(dialogPanel, "calendar")).val();
                        dialog.remove();
                        var subCalendarSelectObject = CalendarPlugin.getSubCalendar(calendarDiv, subCalendarId);
                        var subCalendarEditDialog = CalendarPlugin.showCustomEventEditForm(calendarDiv, subCalendarSelectObject);
                        return;
                    }

                    changeEventType(type);
                    //check event != jira and set attribute for select element
                    if (!CalUtil.isJiraEventStream(type)) {
                        eventEditDialog.getSelect(dialogPanel, "event-type").data("previous", type);
                    }

                    calculateDialogHeight();
                }

                function calculateDialogHeight() {
                    if (dialog) {
                        dialog.getCurrentPanel().body.css({height:"auto"});
                        $("#" + dialog.id).css({height:"auto"});
                    }
                }

                //Called when the event type changes.
                function changeEventType(type) {
                    if(!type) {
                        return;
                    }

                    var fields = CalUtil.isCustomEventType(type) ? EventTypes.CustomEventType.fields : eventTypes[type].fields,
                        fieldHandlers = [],
                        fieldsContainer = $('#event-fields-container'),
                        eventTypeIsJira = CalUtil.isJiraEventStream(type),
                        fieldsDiv = eventTypeIsJira ? $('#' + type + '-event-fields', fieldsContainer) : $('#tc-event-fields', fieldsContainer),
                        eventTypeName = eventEditDialog.getSelect(dialogPanel, "event-type").val(),
                        previousType = eventEditDialog.getSelect(dialogPanel, "event-type").data("previous"),
                        previousValueSummaryHolder,
                        originalEventType = eventEditDialog.getField(dialogPanel, "originalEventType").val();

                    if (event && CalUtil.isJiraEventStream(event.eventType)) {
                        // An existing event. Disable event type changing.
                        // TODO: Discuss feasibility for the ability to change existing event's type.
                        eventEditDialog.getSelect(dialogPanel, "event-type").prop("disabled", true);
                    }

                    if(!eventTypeIsJira && previousType) {// get field value previous value of summary or what
                        var previousFields = CalUtil.isCustomEventType(previousType) ? EventTypes.CustomEventType.fields : eventTypes[previousType].fields;
                        $.each(
                            $.grep(
                                previousFields,
                                function(aField) {
                                    return aField.type === "text" && aField.copyto;
                                }
                            ),
                            function(id, fieldSelected) {
                                previousValueSummaryHolder = eventEditDialog.getField(fieldsDiv, fieldSelected.id).val();
                            }
                        );
                    }

                    //Hide the current fieldsDiv.
                    $(".event-fields", fieldsContainer).each(function() {
                        $(this).hide();
                    });

                    //enable fieldsDiv for event type is JIRA
                    if (fieldsDiv.length && eventTypeIsJira ) {
                        fieldsDiv.show();
                        fieldHandlers = fieldHandlersForType[type];
                    } else {
                        //disable all field event have fieldsDiv is div#tc-event-fields
                        if (fieldsDiv.length) {
                            fieldsDiv.show();
                            $("#tc-event-fields >div").hide();
                        } else {
                            //create new fieldsDiv for event type
                            if (eventTypeIsJira) {
                                fieldsDiv = $("<div/>", {
                                    "id" : type + "-event-fields",
                                    "class" : "event-fields"
                                });
                            } else {
                                fieldsDiv = $("<div/>", {
                                    "id" : "tc-event-fields",
                                    "class" : "event-fields"
                                });
                            }
                            fieldsContainer.append(fieldsDiv);
                        }

                        //For each field in event type add field to panel
                        $.each(fields, function(i, field) {
                            if (eventTypeIsJira) {
                                fieldHandlers[i] = eventFieldHandlers[field.type](field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                                fieldHandlers[i].add();
                                fieldHandlers[i].init();
                            } else {
                                fieldHandlers[i] = eventFieldHandlers.getEventField(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog, previousValueSummaryHolder);
                            }
                        });
                        fieldHandlersForType[type] = fieldHandlers;
                    }

                    var getOkButtonState = function() {
                        return okButtonStates[eventTypeName] !== false;
                    };

                    if (getOkButtonState())
                        eventEditForm.enableOkButton();
                    else
                        eventEditForm.disableOkButton({
                            "buttonText" : isReadonlyEvent ? AJS.I18n.getText("ok") : AJS.I18n.getText("calendar3.save")
                        });

                    //attach new onSubmit handler for current fieldsDiv
                    eventEditForm.unbind("submit").submit(function() {
                         submitForm(fieldsDiv, fieldHandlers);
                        return false;
                    });

                    //Disable/enable this instance only field
                    if (originalEventType !== undefined && originalEventType !== "") {
                        var originalValueSubCalendarId = eventEditDialog.getField(dialogPanel, "originalSubCalendarId").val();
                        var calendarValSelect = eventEditDialog.getSelect(dialogPanel, "calendar").val();
                        if(originalEventType === type && calendarValSelect === originalValueSubCalendarId) {
                            FormStateControl.enableElement($("#editthisinstanceonly"));
                        } else {
                            FormStateControl.disableElement($("#editthisinstanceonly"));
                        }
                    }

                    //click to link setting reminder in add event
                    $(fieldsDiv)
                        .off("click.calendar-reminder-setting")
                        .on("click.calendar-reminder-setting", "#calendar-reminder-setting", function() {
                            var calendarValSelect = eventEditDialog.getSelect(dialogPanel, "calendar").val();
                            dialog.hide();
                            var subCalendarSelectObject = CalendarPlugin.getSubCalendar(calendarDiv, calendarValSelect);
                            CalendarPlugin.showCustomEventEditForm(calendarDiv, subCalendarSelectObject, eventTypeName);
                            return;
                        });
                }

                AJS.unbind("show-dialog-edit-event.teamcalendar").bind("show-dialog-edit-event.teamcalendar", function(event) {
                    var eventContainer = CalUtil.isJiraEventStream(eventEditDialog.getSelectedEventType()) ? "#" + eventEditDialog.getSelectedEventType() + "-event-fields" : "#tc-event-fields";
                    var fieldReminder = $("#field-text-reminder .bodyInforReminderLabel", $(eventContainer, dialogPanel));
                    var eventTypeReminder = CalendarPlugin.getCurrentReminderForEventType(calendarDiv, eventEditDialog);
                    var hasReminderPeriod = eventTypeReminder && eventTypeReminder.periodInMins > 0;
                    //update field reminder when edit/create
                    $(fieldReminder).empty().append(
                        Confluence.TeamCalendars.Templates.Fields.bodyInforReminderLabel({
                            "reminderPeriodText" : hasReminderPeriod ? AJS.format(AJS.I18n.getText('calendar3.reminder.event.editdialog.period.text'), CalUtil.showInforReminder(eventTypeReminder.periodInMins)) : AJS.I18n.getText('calendar3.reminder.event.editdialog.period.none.text'),
                            "reminderPeriod": hasReminderPeriod ? eventTypeReminder.periodInMins : 0
                        })
                    );
                    dialog.show();
                });

                AJS.unbind("remove-dialog-edit-event.teamcalendar").bind("remove-dialog-edit-event.teamcalendar", function(event) {
                    if(dialog) {
                        dialog.remove();
                    }
                });

                AJS.bind("remove.dialog", function(e, data) {
                    if (data.dialog.id == "edit-event-dialog") {
                        dialog = null;
                    }
                });

                submit = function() {
                    eventEditForm.submit();
                    AJS.unbind("show-dialog-edit-event.teamcalendar");
                    return false
                };

                cancel = function() {
                    dialog.remove();
                    AJS.unbind("show-dialog-edit-event.teamcalendar");
                    return false;
                };

                dialog = new AJS.ConfluenceDialog({
                    "id": "edit-event-dialog",
                    "width": width,
                    "height": height,
                    "onSubmit" : submit,
                    "onCancel": cancel,
                    "focusSelector": ".dialog-components"
                });

                dialog.addHeader(event.title || AJS.I18n.getText("calendar3.heading.createevent"));
                if (isReadonlyEvent) {
                    dialog.addButton(AJS.I18n.getText("ok"), submit, "submit");
                } else {
                    dialog.addLink(AJS.I18n.getText("cancel.name"), cancel);
                    dialog.addButton(AJS.I18n.getText("calendar3.save"), submit, "submit");
                }

                okButton = dialog.popup.element.find(".submit");

                dialog.addPanel(
                    "",
                    Confluence.TeamCalendars.Templates.eventEdit(),
                    "calendar-dialog-panel");


                dialogPanel = dialog.getCurrentPanel().body;
                dialog.popup.element.find(".dialog-components").attr('tabindex', '0');


                var setOkButtonState = function(enabled) {
                    okButtonStates[eventEditDialog.getSelect(dialogPanel, "event-type").val()] = enabled;
                };

                eventEditForm = $("form[name='event-edit']", dialogPanel);
                eventEditForm.disableOkButton = function(options) {
                    var mergedOptions = $.extend({
                        "buttonText" : AJS.I18n.getText("calendar3.adding")
                    }, options);

                    okButton.text(mergedOptions.buttonText)
                        .prop("disabled", true)
                        .addClass("ui-state-disabled");

                    setOkButtonState(false);
                };

                eventEditForm.enableOkButton = function(options) {
                    var mergedOptions = $.extend({
                        "buttonText" : isReadonlyEvent? AJS.I18n.getText("ok") : AJS.I18n.getText("calendar3.save")
                    }, options);

                    okButton.prop("disabled", false)
                        .removeClass("ui-state-disabled")
                        .text(mergedOptions.buttonText);

                    setOkButtonState(true);
                };

                //Add calendar field
                addCalendarField(dialogPanel, event);

                dialog.disableOkButton = eventEditForm.disableOkButton;
                dialog.enableOkButton = eventEditForm.enableOkButton;

                //Add event type field
                addEventTypeField(dialogPanel);

                setHiddenFields();

                return dialog;
            },

            disableOkButton : function(options) {
                dialog.disableOkButton(options);
            },
            enableOkButton : function(options) {
                dialog.enableOkButton(options);
            },

            getSelectedEventType : function() {
                return this.getSelect(dialog.popup.element, "event-type").val();
            },

            getSelectedSubCalendarId: function() {
                return this.getSelect(dialog.popup.element, "calendar").val();
            },

            getTextSelectedEventType: function() {
                return this.getTextSelect(dialog.popup.element, "event-type");
            },

            getTextSelectedSubCalendar: function() {
                return this.getTextSelect(dialog.popup.element, "calendar");
            }
        });

        return EventEditDialog;
    }
);
