/**
 * This code creates the FieldHandler objects. Each FieldHandler object must implement the following methods:
 *   add() - This method adds a field to the fieldsDiv and sets up any Javascript needed to make the field work.
 *   init() - This initializes the field to either its default value, or a value taken from the event object.
 *   getData() - This returns the data contained in the fields as key-value pairs.
 *
 * Each field handler expects the field object to have different properties defined.
 *
 * See also:
 *   event-types.js - This is currently where the mappings of fields to event types lives.
 *   event-edit-dialog.js - The code that uses these FieldHandlers.
 *   event-fields.soy - The soy templates for the fields.
 */
define("tc/event-field-handlers",
    [
        "jquery",
        "confluence/date-time",
        "confluence/dark-features",
        "tc/jira-event-field-process-united",
        "tc/jira-event-field-process-divided",
        "tc/calendar-util",
        "tc/repeat-model",
        "tc/repeat-component",
        "tc/form-state-control"
    ],
    function (
        $,
        dateTime,
        DarkFeatures,
        JiraEventFieldProcessUnited,
        JiraEventFieldProcessDivided,
        CalUtil,
        RepeatModel,
        RepeatComponentView,
        FormStateControl
    )
    {
        "use strict";

        var eventFieldHandlers = {

            /**
             * A simple text field. Expects the field object to have the following properties:
             *   id - The id of the field.
             *   label - A label for the field.
             *   required - Is the field required?
             *   readOnly - Is the field non-editable?
             *   fieldText - The text to be rendered in the field.
             *
             */
            text : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.text({
                                "id" : field.id,
                                "label" : field.label,
                                "required" : field.required,
                                "readOnly" : field.readOnly,
                                "fieldText" : event[field.property] || "",
                            })
                        );
                    },
                    init : function () {
                        $("#" + field.id + "-text", fieldsDiv).val(event[field.property] || "").keydown(function() {
                            // This is to set a flag into the input field indicating that the value has been changed by the user.
                            // The use of this is by JIRA event types to not change the event series name after the user
                            // has typed in something for the name.
                            var nameInputField = $(this),
                                name = nameInputField.val();

                            if (!nameInputField.data("valueChangedByUser")) {
                                setTimeout(function() {
                                    if (name !== nameInputField.val()) {
                                        nameInputField.data("valueChangedByUser", true);
                                    }
                                }, 300);
                            }
                        });
                    },
                    setData: function(valueDefault) {
                        eventEditDialog.getField(fieldsDiv, field.id).val(valueDefault);
                    },
                    getData : function() {
                        var data = {};

                        data[field.id] = eventEditDialog.getField(fieldsDiv, field.id).val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-text-" + field.id;
                    }
                };
            },

            /**
             * A textarea field. Expects the field object to have the following properties:
             *   id - The id of the field.
             *   size - The size of the textarea in lines.
             *   required - Is the field required?
             *   label - A label for the field.
             *   readOnly - Is the field non-editable?
             *   fieldText - The text to be rendered in the field.
             *
             */
            textarea : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.textarea({
                                "id" :  field.id,
                                "size" : field.size,
                                "required" : field.required,
                                "label" : field.label,
                                "readOnly" : field.readOnly,
                                "fieldText" : event[field.property] || "",
                            })
                        );
                    },
                    init : function() {
                        $("#" + field.id + "-textarea", fieldsDiv).val(event[field.property] || "");
                    },
                    getData : function() {
                        var data = {};
                        data[field.id] = $("textarea[name=" + field.id + "]", fieldsDiv).val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-textarea-" + field.id;
                    }
                };
            },

            /**
             * A Page field, can be used to enter a url or confluence page. Includes confluence page drop down.
             * Expects the field object to have the following properties:
             *   id - The id of the field.
             *   label - A label for the field.
             *   required - Is the field required?
             *   property - The property on the event object that the field value is taken from.
             *
             *   TODO: make id and property the same. Need to fix in event object.
             */
            page : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.page({
                                "id" : field.id,
                                "label" : field.label,
                                "required" : field.required,
                                "readOnly" : field.readOnly,
                                "fieldText" : event[field.property],
                            })
                        );
                    },
                    init : function () {
                        $("#" + field.id + "-text", fieldsDiv).val(event[field.property] || "");

                        CalendarPlugin.makeAutoCompleteSearch(
                                calendarDiv,
                                $("input[name='" + field.id + "']", fieldsDiv),
                                { type: [ "page", "blogpost" ] },
                        );

                    },
                    getData : function() {
                        var data = {},
                            theField = eventEditDialog.getField(fieldsDiv, field.id);

                        data[field.id] = theField.hasClass("with-hint") ? "" : theField.val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-page-" + field.id;
                    }
                };
            },

            getRepeatModel: function(event) {
                if (!event) {
                    throw 'eventFieldHandlers.getRepeatModel: event is empty';
                }

                if (!this._repeats2Models) {
                    this._repeats2Models = new Map();
                }

                if (!this._repeats2Models.has(event)) {
                    this._repeats2Models.set(
                        event,
                        new RepeatModel({
                            rruleStr: event.rruleStr ? event.rruleStr : "",
                            event: event
                        })
                    );
                }

                return this._repeats2Models.get(event);
            },
            /**
             * Adds a 'when' field to the form. Consists of a start and end date and time pickers, and an all-day checkbox.
             * Expects the field object to have the following properties:
             *   id - id of the field.
             *
             */
            when : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        var startTimeField, endTimeField, allDayEventCheckbox;
                        var isSingleJiraDate = event && (event.eventType === "jira-project-releases" || (event.eventType === "jira" && event.end == null));

                        // handles Jira issue datefield names
                        var dateFieldStartName, dateFieldEndName, startDateFieldEditable, startDateFieldHasTime,
                            endDateFieldEditable, endDateFieldHasTime;
                        if (event && event.extraProperties) {
                            dateFieldStartName = event.extraProperties.dateFieldStartName;
                            dateFieldEndName = event.extraProperties.dateFieldEndName;
                            // only set to false if property is explicitly set to false
                            startDateFieldEditable = event.extraProperties.startDateFieldEditable !== "false";
                            startDateFieldHasTime = event.extraProperties.startDateFieldHasTime !== "false";
                            endDateFieldEditable = event.extraProperties.endDateFieldEditable !== "false";
                            endDateFieldHasTime = event.extraProperties.endDateFieldHasTime !== "false";
                        }
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.when({
                            "id" : field.id,
                            "disableAllDay" : field.disableAllDay,
                            start : {
                                fieldName : dateFieldStartName || null,
                                disableField : field.disableStartDateTime || startDateFieldEditable === false,
                                hideTime : startDateFieldHasTime === false,
                                hideField : false,
                                startDateString: event.localizedStartDate + " " + event.localizedStartTime,
                            },
                            end : {
                                fieldName : dateFieldEndName || null,
                                disableField : field.disableEndDateTime || endDateFieldEditable === false,
                                hideTime : endDateFieldHasTime === false,
                                hideField : isSingleJiraDate,
                                endDateString: event.localizedEndDate + " " + event.localizedEndTime,
                            }
                        }));

                        startTimeField = eventEditDialog.getField(fieldsDiv, "startTime");
                        endTimeField = eventEditDialog.getField(fieldsDiv, "endTime");
                        allDayEventCheckbox = eventEditDialog.getField(fieldsDiv, "allDayEvent");

                        //Set up All Day Event checkbox
                        const updateTimeFields = function(checked) {
                            if (checked) {
                                startTimeField.val('').addClass('hidden');
                                endTimeField.val('').addClass('hidden');
                            } else {
                                startTimeField.removeClass('hidden').removeAttr('tabindex');
                                endTimeField.removeClass('hidden').removeAttr('tabindex');
                                if (!startTimeField.val()) {
                                    startTimeField.val(CalendarPlugin.getDefaultStartTime());
                                }
                                if (!endTimeField.val()) {
                                    endTimeField.val(CalendarPlugin.getDefaultEndTime());
                                }
                            }
                        }

                        updateTimeFields(allDayEventCheckbox.is(':checked'));

                        allDayEventCheckbox.click(function() {
                            updateTimeFields(this.checked);
                        });
                    },
                    init : function() {
                        const startDatepicker = $('.datepicker-field.start');
                        const endDatePicker = $('.datepicker-field.end');
                        startDatepicker.on('change', function(event) {
                            const isoDateString = event.target.value;
                            eventFieldHandlers.getRepeatModel(event).set({
                                startDate: dateTime.parse(isoDateString)
                            });
                            endDatePicker.val(isoDateString);
                        });

                        const startTimeField = eventEditDialog.getField(fieldsDiv, "startTime");
                        const endTimeField = eventEditDialog.getField(fieldsDiv, "endTime");
                        const allDayEventField = eventEditDialog.getField(fieldsDiv, "allDayEvent");

                        if (event.start) {
                            eventFieldHandlers.getRepeatModel(event).set({
                                startDate: event.start,
                            });
                            startDatepicker.val(dateTime.formatPlainDate(event.start));
                            endDatePicker.val(dateTime.formatPlainDate(event.end || event.start));
                        }

                        if (event.allDay) {
                            startTimeField.val("").addClass("hidden");
                            endTimeField.val("").addClass("hidden");
                            allDayEventField.prop("checked", true);
                        } else {
                            const startTime = CalUtil.getTime(event.start);
                            const endTime = CalUtil.getTime(event.end || new Date(event.start.getTime() + 1800000));
                            startTimeField.val(startTime);
                            endTimeField.val(endTime);
                            startTimeField.removeClass('hidden').removeAttr('tabindex');
                            endTimeField.removeClass('hidden').removeAttr('tabindex');
                            allDayEventField.prop("checked", false);
                        }
                    },
                    getData : function() {
                        return {
                            startDate : eventEditDialog.getField(fieldsDiv, "startDate").val(),
                            endDate : eventEditDialog.getField(fieldsDiv, "endDate").val(),
                            startTime: eventEditDialog.getField(fieldsDiv, "startTime").val(),
                            endTime: eventEditDialog.getField(fieldsDiv, "endTime").val(),
                            allDayEvent : eventEditDialog.getField(fieldsDiv, "allDayEvent").is(":checked")
                        };
                    },
                    getElementId : function() {
                        return "#field-when";
                    }
                };
            },
            repeats2 : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                var that = this;
                return {
                    model: null,
                    view: null,
                    add : function () {
                        this.view = new RepeatComponentView({
                            model: that.getRepeatModel(event),
                            fieldsDiv: fieldsDiv,
                            _calendarDiv: calendarDiv,
                            _calendarPlugin: CalendarPlugin,
                        });
                    },
                    init : function() {
                        const startDatepicker = $('.datepicker-field.start');
                        const endDatePicker = $('.datepicker-field.end');
                        var disableFields = function(that, subCalendarSelect){
                            eventEditDialog.disableCalSelect();
                            eventEditDialog.showToolTip();
                            subCalendarSelect.prop("disabled", true);
                            if (event.start) {
                                startDatepicker.val(dateTime.formatPlainDate(event.start));
                                endDatePicker.val(dateTime.formatPlainDate(event.end || event.start));
                            }
                            // Can't change allday state for individual recurring event, this causes many problems with reschedule event matching because originalStartTime changes
                            eventEditDialog.getField(fieldsDiv, "allDayEvent").disable();
                        };

                        var enableFields = function(that, subCalendarSelect) {
                            eventEditDialog.enableCalSelect();
                            eventEditDialog.hideToolTip();
                            subCalendarSelect.prop("disabled", false);
                            startDatepicker.val(event.originalStartDateTime.split('T')[0]);
                            endDatePicker.val(event.originalEndDateTime.split('T')[0]);
                            eventEditDialog.getField(fieldsDiv, "allDayEvent").enable();
                        };

                        this.view.render();

                        var subCalendarSelect = $("select[name='calendar']", fieldsDiv);
                        if (event.recur) {
                            disableFields(that, subCalendarSelect);
                        }

                        $("input[name='editthisinstanceonly']", fieldsDiv).change(function() {
                            if ($(this).is(":checked")) {
                                disableFields(that, subCalendarSelect);
                            } else {
                                enableFields(that, subCalendarSelect);
                            }
                        });
                    },
                    getData : function() {
                        var model = that.getRepeatModel(event);
                        var rruleStr = model.getRRuleAsString();
                        var untilDateStr = model.attributes.untilDateOnly || "";
                        console.log("Final RRULE string is " + rruleStr + " and until date is " + untilDateStr);
                        return {
                            "rruleStr": rruleStr,
                            // RRULE will not include UNTIL option. We handle the full UNTIL DateTime on the backend to prevent
                            // javascript Date() object in browser timezone being converted to UTC date (with the wrong time for actual event series)
                            "until": untilDateStr, // Until date is selected in user timezone
                            "editAllInRecurrenceSeries": !model.attributes.isDisable
                        }
                    },
                    getElementId : function() {
                        return "#field-repeats";
                    }
                };
            },
            /**
             * Allows selection of multiple users.
             */
            user : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function() {
                        fieldsDiv.append (
                            Confluence.TeamCalendars.Templates.Fields.user({
                                id :  field.id,
                                label : field.label,
                                required : field.required
                            })
                        );
                        //This makes the user dropdown component work.
                        Confluence.runBinderComponents();
                    },
                    init : function() {
                        var inviteeList = $(".invitees", fieldsDiv),

                            deleteInviteeHandler = function() {
                                $(this).closest("li").remove();

                                if (!$("li", inviteeList).length) {
                                    inviteeList.addClass("hidden");
                                }
                            },

                            ellipsisInviteeNames = function() {
                                $(".invitee-left", inviteeList).filter(function() {
                                    return !$(this).attr("threedots");
                                }).ThreeDots({
                                    max_rows : 1,
                                    text_span_class: "invitee-name-display",
                                    whole_word: false
                                });
                            };

                        $("#" + field.id + "-user-picker", fieldsDiv).bind("selected.autocomplete-user", function(e, selection) {
                            var selectedUserId = selection.content.userKey || selection.content.username,
                                userSelection,
                                userAlreadyAdded = $(".invitee", inviteeList).filter(function() {
                                    return $(this).data("entity").id === selectedUserId;
                                }).length > 0;

                            if (!userAlreadyAdded) {
                                if ($("li", inviteeList).length === 3) {
                                    inviteeList.addClass("boxed");
                                }

                                userSelection = $(Confluence.TeamCalendars.Templates.Fields.userSelection({
                                    "fullName" : selection.content.title,
                                    "userName" : selection.content.username,
                                    "imgPath" : selection.content.thumbnailLink.href
                                }));

                                userSelection.data("entity", {
                                    "id" : selectedUserId,
                                    "displayName" : selection.content.title
                                }).find(".delete-invitee").click(deleteInviteeHandler);

                                inviteeList.prepend(userSelection).removeClass("hidden");

                                ellipsisInviteeNames();
                            }

                            // focus() to workaround https://studio.plugins.atlassian.com/browse/TEAMCAL-229
                            $(this).val("").focus();

                            var whoErrorDiv = $(".who-error", fieldsDiv);
                            if (whoErrorDiv.children().length) {
                                // Only empty and resize dialog if the error div has elements in it to fix the
                                // problem where the dialog footer is longer than it should be in Chrome 12 + OS X Lion
                                whoErrorDiv.empty();
                            }
                        });

                        if (event.invitees && event.invitees.length) {
                            $.each(event.invitees, function(inviteeIdx, invitee) {
                                inviteeList.append(
                                    $(Confluence.TeamCalendars.Templates.Fields.userSelection({
                                        "fullName" : invitee.displayName,
                                        "userName" : invitee.name,
                                        "imgPath" : invitee.avatarIconUrl
                                    })).data("entity", {
                                            "id" : invitee.id,
                                            "displayName" : invitee.displayName
                                    })
                                );
                            });

                            $(".delete-invitee", inviteeList.removeClass("hidden")).unbind("click").click(deleteInviteeHandler);
                            ellipsisInviteeNames();
                        }
                    },
                    getData : function() {
                        var data = {};
                        var invitees = $(".invitee", fieldsDiv);

                        if (invitees.length) {
                            data.person = $.map(invitees, function(invitee) {
                                return $(invitee).data("entity").id;
                            });
                        }

                        if (field.setTitle) {
                            data.what = invitees.length ? $.map(invitees, function(invitee) {
                                return $(invitee).data("entity").displayName;
                            }).join(", ") : "";
                        }

                        return data;
                    },
                    getElementId : function() {
                        return "#field-user-" + field.id;
                    }
                };
            },

            getEventField : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog, valueDefault) {
                var fieldHandler = eventFieldHandlers[field.type](field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);

                var fieldSelected = $(fieldHandler.getElementId(), fieldsDiv);
                //check if exist field, enable that field.
                if (fieldSelected.length) {
                    //check dose who field has required?
                    if (field.required) {
                        var fieldRequired = fieldSelected.find("label:first span.icon-required");
                        if (fieldRequired.length === 0) {
                            fieldSelected.find("label:first").append($("<span/>", {"class": "aui-icon icon icon-required" }));
                        }
                    } else {
                        fieldSelected.find("label:first span.icon-required").remove();
                    }
                    fieldsDiv.append(fieldSelected.show());

                    if (field.type === "repeats2") {
                        if (field.defaultRecur) {
                            eventFieldHandlers.getRepeatModel(event).set({frequencySelection: field.defaultRecur.freq});
                        }
                    }
                } else {
                    fieldHandler.add();
                    fieldHandler.init();
                }

                if(field.copyto && valueDefault !== undefined) {
                    fieldHandler.setData(valueDefault);
                }

                return fieldHandler;
            },

            reminder : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                   add : function () {
                       var eventTypeReminder = CalendarPlugin.getCurrentReminderForEventType(calendarDiv, eventEditDialog);
                       $(fieldsDiv).find("#field-text-" + field.id).remove(); //remove old field reminder
                       var hasReminderPeriod = eventTypeReminder && eventTypeReminder.periodInMins > 0;
                       //Add field reminder when edit/create
                       fieldsDiv.append(
                           Confluence.TeamCalendars.Templates.Fields.inforReminderLabel({
                               "id" : field.id,
                               "label" : field.label,
                               "reminderPeriodText" : hasReminderPeriod ? AJS.format(AJS.I18n.getText('calendar3.reminder.event.editdialog.period.text'), CalUtil.showInforReminder(eventTypeReminder.periodInMins)) : AJS.I18n.getText('calendar3.reminder.event.editdialog.period.none.text'),
                               "reminderPeriod": hasReminderPeriod ? eventTypeReminder.periodInMins : 0
                           })
                       );
                   },
                   init : function() {
                   },
                   getData : function() {
                       return {};
                   },
                   getElementId : function() {
                       return ""; //always create new field reminder when change event type
                   }
                };
            },

            /**
            * Adds hidden fields found on all JIRA event forms
            */
            "jira-hidden-fields" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.hiddenJiraFields());
                    },
                    init : function() {
                        var subCalendar = JiraEventFieldProcessUnited.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event),
                            callbackHandler = CalendarPlugin.getRenderedMacroCallbackHandler(calendarDiv),
                            helper = Confluence.TeamCalendars.getEditCalendarDialogHelper(fieldsDiv, subCalendar, callbackHandler);

                        if (!subCalendar || !subCalendar.id) {
                            helper.setSubCalendarIdField("");
                            helper.setNameField("");
                            helper.setColorField("");
                            helper.setSpaceToDefault();
                        } else {
                            helper.setSubCalendarIdField(subCalendar.id);
                            helper.setNameField(subCalendar.name);
                            helper.setColorField(subCalendar.color);
                            helper.setSpaceKeyField(subCalendar.spaceKey);
                        }
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
            * A jira server selector
            */
            "jira-server-select" : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.serverSelector());
                        FormStateControl.disableElement(JiraEventFieldProcessUnited.getServerSelect(fieldsDiv));
                    },
                    init : function() {
                        var serverSelect = JiraEventFieldProcessUnited.getServerSelect(fieldsDiv),
                            spinner = $(".field-group-jira-server .spinner");

                        // clean up callbacks
                        AJS.unbind("server-select-ready.teamcalendars");
                        AJS.unbind("server-select-change.teamcalendars");

                        function populateDropDown (data) {
                            var subCalendar = JiraEventFieldProcessUnited.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event);

                            $.each(data.jiraLinks, function (jiraLinkIdx, jiraLink) {
                                var optionItem = $("<option/>", {
                                    "value":            jiraLink.id,
                                    "text":             jiraLink.name,
                                    "data-display-url": jiraLink.displayUrl
                                });

                                if (data.jiraLinks.length === 1) {
                                    //TEAMCAL-1719 - if there is only one option an then selected by default
                                    optionItem.prop('selected', true);
                                    var fieldGroup = serverSelect.parent();
                                    fieldGroup.children().hide();
                                    //stype layout when hide all children field
                                    fieldGroup.css("min-height", 0);
                                    // if already hide don't need to hide it again
                                    $('.oauth-error', fieldGroup).show();
                                }
                                optionItem.appendTo(serverSelect);
                            });

                            if (data.jiraLinks.length > 1)
                                $(".field-group-jira-server", fieldsDiv).removeClass("hidden");

                            if(subCalendar) {
                                if(subCalendar.applicationId) {
                                    serverSelect.val(subCalendar.applicationId);
                                }
                                else if(subCalendar.sourceSubCalendar && subCalendar.sourceSubCalendar.applicationId){
                                        serverSelect.val(subCalendar.sourceSubCalendar.applicationId);
                                }
                            }
                        }

                        serverSelect.change(function() {
                            AJS.trigger("server-select-change.teamcalendars", fieldsDiv);
                        });

                        //Do ajax call to get available JIRA servers
                        $.ajax({
                            cache: false,
                            dataType: 'json',
                            error: function(jqXHR, textStatus, errorThrown) {
                                CalendarPlugin.showAjaxError(fieldsDiv.parent().parent(), jqXHR, textStatus, errorThrown);
                            },
                            success: function(data) {
                                populateDropDown(data);
                                //Fire an event for anyone waiting for the drop down to be populated
                                AJS.trigger("server-select-ready.teamcalendars", fieldsDiv);
                            },
                            complete: function() {
                                FormStateControl.enableElement(serverSelect);
                                spinner.hide();
                            },
                            timeout : Confluence.TeamCalendars.ajaxTimeout,
                            url: Confluence.TeamCalendars.getCalendarServiceBaseUrl("/jira/jiraLinks")
                        });
                        //Set default value
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
            * A JIRA project selector.
            */
            "jira-project-select" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.projectSelector({useOptimizedProjectPicker: !DarkFeatures.isEnabled('confluence.team-calendar.jira-combined-query-dialog')}));
                        FormStateControl.disableElement($(".tc-jira-project", fieldsDiv));
                    },
                    init : function() {
                        function serverChanged(context) {
                            if (context.id === fieldsDiv[0].id) { //event may have been fired from another panel
                                $(".oauth-approve", fieldsDiv).remove();
                                const JiraEventFieldProcess = DarkFeatures.isEnabled('confluence.team-calendar.jira-combined-query-dialog') ? JiraEventFieldProcessUnited : JiraEventFieldProcessDivided;
                                JiraEventFieldProcess.populateProjectList(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                            }
                        }

                        AJS.bind("server-select-ready.teamcalendars", function(event, context) {
                            serverChanged(context);
                        });

                        AJS.bind("server-select-change.teamcalendars", function(event, context) {
                            serverChanged(context);
                        });
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
             * Handles most of the fields on the JIRA issues date fields form. Lumped together for historical reasons.
             */
            "jira-dates" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.editJiraCalendar({isDividedFilterOptionsEnabled: !DarkFeatures.isEnabled('confluence.team-calendar.jira-combined-query-dialog')}));
                    },
                    init : function () {
                        const JiraEventFieldProcess = DarkFeatures.isEnabled('confluence.team-calendar.jira-combined-query-dialog') ? JiraEventFieldProcessUnited : JiraEventFieldProcessDivided;

                        //This is fired by the jira-server-select field in event-field-handlers.js
                        AJS.bind("server-select-ready.teamcalendars", function() {
                            JiraEventFieldProcess.initJiraDate(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                        });

                        AJS.bind("server-select-change.teamcalendars", function() {
                            JiraEventFieldProcess.initJiraDate(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                        });
                    },
                    getData : function() {
                         //Should never be called as JIRA events use a custom handler
                    }
                };
            }
        };

        return eventFieldHandlers;
    }
);
