/* globals Confluence, AppLinks, JIRA */

/**
 * Module responsible for handling form for "Jira Issue Dates" event type in Team Calendar dialog.
 * Based on old {@link module:tc/jira-event-field-process-united} module
 * but divides the flow between query filter options to make them performant.
 *
 * Note: Related Event types are bugged on the outside (out of scope of this file),
 * as only on the first render of each view the correct `fieldsDiv` are passed down to the handlers.
 * After the view is initially rendered, subsequent triggers are passing the last view from DOM.
 *
 * @module tc/jira-event-field-process-divided
 */
define('tc/jira-event-field-process-divided',
    [
        'jquery',
        'tc/calendar-util',
        'tc/form-state-control',
        'tc-backbone',
        'tc/jira-event-field-process-common'
    ],
    function(
        $,
        CalUtil,
        FormStateControl,
        Backbone,
        JiraEventFieldProcessCommon
    ) {
        'use strict';
        // Maximum number of results to be displayed when querying projects
        const PROJECT_QUERY_OPTION_RESULTS_LIMIT = 100;

        /**
         * Defines type of event selected in the dialog.
         * @type {Readonly<{PROJECT_RELEASES: string, AGILE_SPRINTS: string, ISSUE_DATES: string}>}
         */
        const DialogEventType = Object.freeze({
            ISSUE_DATES: 'jira-calendar',
            AGILE_SPRINTS: 'jira-agile-sprint-calendar',
            PROJECT_RELEASES: 'jira-project-releases-calendar'
        });

        /**
         * Defines the request error types.
         * @type {Readonly<{DEFAULT: string, OAUTH: string}>}
         */
        const RequestErrorType = Object.freeze({
            OAUTH: 'oauth',
            DEFAULT: 'default'
        });

        /**
         * Defines the mode in which the dialog is being opened.
         * @type {Readonly<{CREATE: string, EDIT: string}>}
         */
        const DialogMode = Object.freeze({
            CREATE: 'create',
            EDIT: 'edit'
        });

        /**
         * Defines the possible query filter options.
         *
         * Note: Values match the ones used in HTML.
         * @type {Readonly<{PROJECT: string, FILTER: string, JQL: string}>}
         */
        const DatesQueryType = Object.freeze({
            JQL: 'option-jql', // Filter by custom JQL
            PROJECT: 'option-project', // Filter by Jira project
            FILTER: 'option-search-filter' // Filter by Jira saved filter
        });

        /**
         * Defines the data state of a single filter query option.
         * @type {Readonly<{LOADING: string, ERROR: string, EMPTY: string, LOADED: string}>}
         */
        const QueryDataState = Object.freeze({
            EMPTY: 'empty',     // Data needed for query option is not loaded
            LOADING: 'loading', // Data is being loaded
            LOADED: 'loaded',   // Data needed to serve query filter option is ready
            ERROR: 'error',     // Error when loading data necessary for given option
        });

        /**
         * Defines the JQL input validation state.
         * @type {Readonly<{VALID: boolean, PROCESSING: string, ERROR: string, INVALID: boolean, EMPTY: string}>}
         */
        const JqlInputValidityState = Object.freeze({
            EMPTY: 'empty', // JQL input is empty, nothing to validate
            PROCESSING: 'processing', // JQL input is being validated
            VALID: true, // JQL input is valid
            INVALID: false, // JQL input is invalid
            ERROR: 'error' // JQL input could not be validated
        });

        const translations = {
            projectPickerPlaceholder: AJS.I18n.getText('calendar3.jira.fields.project.placeholder'),
            projectPickerLoading: AJS.I18n.getText('calendar3.jira.fields.project.message.loading'),
            projectPickerNoMatches: AJS.I18n.getText('calendar3.jira.fields.project.no-matches.message'),
            projectPickerLoadingError: AJS.I18n.getText('calendar3.jira.fields.project.message.error'),
            jqlValidationError: AJS.I18n.getText('calendar3.jira.error.jql.cannotvalidateagainstselectedjira'),
            savedFilterLoadingError: AJS.I18n.getText('calendar3.jira.fields.search-filter.message.error'),
            dateFieldsLoadingError: AJS.I18n.getText('calendar3.jira.error.datefields.unabletoshow'),
            dateFieldNotSearchable: AJS.I18n.getText('calendar3.jira.warning.fields.fieldnotsearchable'),
            addDurationDialogTitle: AJS.I18n.getText('calendar3.jira.duration.title'),
            oauthApprovalMessage: AJS.I18n.getText('calendar3.oauth.loginapprovetoseeprojects'),
            oauthLoginApproveText: AJS.I18n.getText('calendar3.oauth.loginandauthorize'),
            oauthApprovalUnsuccessful: AJS.I18n.getText('calendar3.oauth.trydancingagain'),
            wordAdd: AJS.I18n.getText('add.word'),
            wordCancel: AJS.I18n.getText('cancel.name'),
            wordOk: AJS.I18n.getText('ok'),
            dynamic: {
                eventName: new Map([
                    [DialogEventType.ISSUE_DATES, function(projectName) {
                        return AJS.I18n.getText('calendar3.event.field.name.jira.issues', projectName);
                    }],
                    [DialogEventType.AGILE_SPRINTS, function(projectName) {
                        return AJS.I18n.getText('calendar3.event.field.name.jira.sprints', projectName);
                    }],
                    [DialogEventType.PROJECT_RELEASES, function(projectName) {
                        return AJS.I18n.getText('calendar3.event.field.name.jira.releases', projectName);
                    }]])
            },
        };
        const selectors = {
            projectPicker: '.project-filter-picker:not(.select2-container)',
        };

        /**
         * Returns string representation of select option in Project picker.
         * If one property is missing, only the one present is used.
         * If two properties are missing, empty string is returned.
         * @param {string} projectKey - Project key
         * @param {string} projectName - Project name
         * @returns {string}
         */
        function getProjectSelectOption(projectKey, projectName) {
            if (projectKey && projectName) {
                return projectName + ' (' + projectKey + ')';
            } else {
                return projectName || projectKey || '';
            }
        }

        // region <Backbone structure>

        // Jira Issue Dates model (serving whole dialog)
        const JiraEventDatesModel = Backbone.Model.extend({
            defaults: {
                mode: DialogMode.CREATE,
                queryType: DatesQueryType.PROJECT,
                jiraServer: '',
                initValue: null,
                selectedDateFields: [],
                selectedDateDurations: [],
            },
            getDialogMode: function(){
                return this.get('mode');
            },
            getQueryType: function() {
                return this.get('queryType');
            },
        });

        const JqlQueryOptionModel = Backbone.Model.extend({
            defaults: {
                queryData: QueryDataState.EMPTY,
                jqlInputValue: '',
                jqlInputValidity: JqlInputValidityState.EMPTY,
            },

            initialize: function() {
                this.on('change:jqlInputValue', this._validateJqlInput);
            },

            loadData: function() {
                this.set('queryData', QueryDataState.LOADING);
                const model = this;
                const $container = data.$dialogContainer;

                const onJqlOptionsLoadSuccess = function(queryOptions) {
                    const jiraQueryOptionsContainer = $('.jira-query-options', $container);

                    const $currentJqlInput =  $('textarea[name="jql"]', jiraQueryOptionsContainer);
                    const currentJqlInput = $currentJqlInput.val();
                        $currentJqlInput
                            .before($(Confluence.TeamCalendars.Templates.jqlInput()).val(currentJqlInput).trigger('focus'))
                        .remove();

                    const jqlAutoCompleteUrlRegex = /.+\/jql\/autocompletedata\/suggestions.json\?fieldName=[^&]+&fieldValue=.*/;
                    const jqlAutoCompleteSpinner = $('.jqlAutoCompleteSpinner', jiraQueryOptionsContainer);
                    $(document)
                        .ajaxSend(function(event, jqXHR, ajaxOptions) {
                            if (jqlAutoCompleteUrlRegex.test(ajaxOptions.url || '')) {
                                jqlAutoCompleteSpinner.removeClass('hidden');
                            }

                        })
                        .ajaxComplete(function(event, jqXHR, ajaxOptions) {
                            if (jqlAutoCompleteUrlRegex.test(ajaxOptions.url || '')) {
                                jqlAutoCompleteSpinner.addClass('hidden');
                            }
                        });

                    const jqlErrorId = 'jql-error';
                    JIRA.JQLAutoComplete({
                        fieldID: 'jql',
                        parser: JIRA.JQLAutoComplete.MyParser(JSON.parse(queryOptions.jqlReservedKeywordsJson || '[]')),
                        queryDelay: 0.65,
                        jqlFieldNames: JSON.parse(queryOptions.visibleFieldNamesJson || '[]'),
                        jqlFunctionNames: JSON.parse(queryOptions.visibleFunctionNamesJson || '[]'),
                        minQueryLength: 0,
                        allowArrowCarousel: true,
                        autoSelectFirst: false,
                        errorID: jqlErrorId,
                        jiraApplicationId: JiraEventFieldProcess.getServerSelect()
                            .val(),
                        baseUrl: Confluence.getBaseUrl()
                    });
                };

                const selectedJiraLink = JiraEventFieldProcess.getServerSelect(this.$el)
                    .val();
                $.ajax({
                    cache: false,
                    dataType: 'json',
                    error: function(jqXHR) {
                        const errorData = handleRequestError(jqXHR);
                        model.set('queryData', QueryDataState.ERROR, errorData);
                    },
                    success: function(queryOptions) {
                        model.set('queryData', QueryDataState.LOADED);
                        onJqlOptionsLoadSuccess(queryOptions);
                    },
                    timeout: Confluence.TeamCalendars.ajaxTimeout,
                    url: Confluence.getBaseUrl() + '/rest/calendar-services/1.0/calendar/jira/' + encodeURIComponent(selectedJiraLink) + '/jql/autocompletedata'
                });
            },

            _validateJqlInput: function(model, value) {
                if (value === '') {
                    model.set('jqlInputValidity', JqlInputValidityState.EMPTY);
                    return;
                }

                model.set('jqlInputValidity', JqlInputValidityState.PROCESSING);
                $.ajax({
                    cache: false,
                    data: { jql: value },
                    error: function() {
                        model.set('jqlInputValidity', JqlInputValidityState.ERROR);
                    },
                    success: function(jqlValidationResult) {
                        model.set('jqlInputValidity', jqlValidationResult.valid);
                        if (jqlValidationResult.valid) {
                            model.trigger('queryDateFields');
                        }
                    },
                    type: 'PUT',
                    url: Confluence.getBaseUrl()
                        .concat(
                            '/rest/calendar-services/1.0/calendar/jira/',
                            encodeURIComponent(JiraEventFieldProcess.getServerSelect(this.$el)
                                .val()), '/jql/validate.json'
                        ),
                    timeout: Confluence.TeamCalendars.ajaxTimeout
                });
            },
        });

        const ProjectQueryOptionModel = Backbone.Model.extend({
            defaults: {
                queryData: QueryDataState.EMPTY,
                selectedProject: null,
            },
            initialize: function() {
                this.on('change:selectedProject', this.onProjectSelectionChange);
            },
            onProjectSelectionChange: function() {
                this.trigger('queryDateFields');
            },
        });

        // Uses deprecated query options that return whole data model for the dialog
        const FilterQueryOptionModel = Backbone.Model.extend({
            defaults: {
                queryData: QueryDataState.EMPTY,
                selectedFilter: null,
            },
            loadData: function() {
                this.set('queryData', QueryDataState.LOADING);
                this._queryOptionsData();
            },
            _queryOptionsData: function() {
                const $container = data.$dialogContainer;
                const model = this;
                const selectedJiraLink = JiraEventFieldProcess.getServerSelect($container)
                    .val();
                const handleGetOptionsSuccess = function(queryOptions) {
                    $('.compatibility-group, .field-group-jira-show', $container).removeClass('hidden');

                    // Initialize saved filters query data
                    const searchFiltersData = queryOptions.searchRequests;
                    const searchFilterOption = $('.option-search-filter', $container);
                    const searchFilterErrors = $('.option-search-filter .error', $container);
                    const searchFilterInput = $('input[name=\'searchFilter\']', searchFilterOption);

                    JiraEventFieldProcessCommon.setupSearchFilterSelect(searchFilterInput, searchFiltersData, () => {
                        searchFilterErrors.empty();
                    });

                    if (searchFiltersData && $.isArray(searchFiltersData) && searchFiltersData.length) {
                        $('.option-content', searchFilterOption.removeData('hideDateFields'))
                            .removeClass('disabled');
                    } else {
                        JiraEventFieldProcessCommon.disableSearchFilterSelect(searchFilterInput);
                        $('.option-content', searchFilterOption.data('hideDateFields', true))
                            .addClass('disabled');
                    }

                    $('.duration', $container)
                        .css('visibility', 'visible');


                    data.eventDialogView.enableOkButton();

                    if (data.subCalendar) {
                        const jiraSubCalendar = data.subCalendar.sourceSubCalendar || data.subCalendar;
                        if (jiraSubCalendar.searchFilterId) {
                            const $searchFilterInput = $('input[name=\'searchFilter\']', searchFilterOption);
                            JiraEventFieldProcessCommon.setValueOfSearchFilterSelect(
                                $searchFilterInput,
                                jiraSubCalendar.searchFilterId);
                            // triggers Backbone event listener on FilterQueryOptionView
                            $searchFilterInput.trigger('change');
                        }
                    }
                    model.set('queryData', QueryDataState.LOADED);
                };

                $.ajax({
                    cache: false,
                    dataType: 'json',
                    error: function(jqXHR, textStatus, errorThrown) {
                        model.set('queryData', QueryDataState.ERROR);
                        handleRequestError(jqXHR, textStatus, errorThrown);
                    },
                    success: function(queryOptions) {
                        handleGetOptionsSuccess(queryOptions);
                        const message = AJS.I18n.getText('calendar3.jira.error.project.nonefound', $('.tc-jira-server option:selected')
                            .data('display-url'));
                        $('.option-project .unavailable-message')
                            .html(message);
                    },
                    timeout: Confluence.TeamCalendars.ajaxTimeout,
                    url: Confluence.getBaseUrl() + '/rest/calendar-services/1.0/calendar/jira/' + encodeURIComponent(selectedJiraLink) + '/query/options.json'
                });
            }
        });

        const BaseOptionView = Backbone.View.extend({
            onQueryTypeChange: function(_, queryType) {
                if (queryType === this.type) {
                    this._initView();
                }
            },

            /**
             * Forces the query data reload. Used in case of successful authorization to Jira instance.
             */
            reloadQueryData: function() {
                if (this.model.loadData && $.isFunction(this.model.loadData)) {
                    this.model.loadData();
                }
            },

            cleanup: function() {
                this.undelegateEvents();
            }
        });

        const JqlQueryOptionView = BaseOptionView.extend({
            type: DatesQueryType.JQL,
            events: {
                'keydown textarea[name="jql"]': 'onJqlInputKeydown'
            },
            initialize: function(options) {
                this.dialogModel = options.dialogModel;
                this.jqlInputValue = '';

                this.listenTo(this.model, 'change:queryData', this.updateLoadingState);
                this.listenTo(this.model, 'change:jqlInputValidity', this.updateJqlValidityState);
                this.listenTo(this.dialogModel, 'change:queryType', this.onQueryTypeChange);

                if (this.dialogModel.getQueryType() === this.type) {
                    const initValue = options.dialogModel.get('initValue');
                    if (initValue) {
                        this.$el.find('textarea[name="jql"]')
                            .val(initValue);
                        setTimeout(function() {
                            const jiraQueryOptionsContainer = $('.jira-query-options', data.$dialogContainer);
                            const $jqlInput =  $('textarea[name="jql"]', jiraQueryOptionsContainer);
                            $jqlInput.trigger('keydown', [true]);
                        });
                    }
                    this._initView();
                }
            },

            onJqlInputKeydown: $.debounce(function(event, validateRegardless) {
                const newJqlInputValue = event.target.value;
                const isEscapePressed = event.keyCode === 27;

                if (validateRegardless || (!isEscapePressed && newJqlInputValue !== this.model.get('jqlInputValue'))) {
                    this.model.set('jqlInputValue', newJqlInputValue);
                    setDateFieldsHidden(true, this.$el);
                }
            }, 300),

            _initView: function() {
                if (this.model.get('queryData') === QueryDataState.EMPTY) {
                    this.model.loadData(this.$el);
                }
            },

            updateJqlValidityState: function(model) {
                const jqlInputValidity = model.get('jqlInputValidity');
                const jqlOk = $('.jql-ok', this.$el);
                const jqlBad = $('.jql-bad', this.$el);
                const jqlValidating = $('.jql-validating', this.$el);
                const hideAllHints = function(){
                    jqlValidating.addClass('hidden');
                    jqlOk.addClass('hidden');
                    jqlBad.addClass('hidden');
                };
                const jqlErrorContainer = $('.jql-error', this.$el);
                jqlErrorContainer.empty();

                switch (jqlInputValidity) {
                case JqlInputValidityState.PROCESSING:
                    jqlValidating.removeClass('hidden');
                    jqlOk.addClass('hidden');
                    jqlBad.addClass('hidden');
                    break;
                case JqlInputValidityState.VALID:
                    jqlValidating.addClass('hidden');
                    jqlOk.removeClass('hidden');
                    jqlBad.addClass('hidden');

                    $('textarea[name="jql"]', this.$el)
                        .focus();
                    break;
                case JqlInputValidityState.INVALID:
                    jqlValidating.addClass('hidden');
                    jqlOk.addClass('hidden');
                    jqlBad.removeClass('hidden');
                    break;
                case JqlInputValidityState.ERROR:
                    hideAllHints();
                    jqlErrorContainer.text(translations.jqlValidationError);
                    break;
                case JqlInputValidityState.EMPTY:
                    hideAllHints();
                    break;
                }
            },

            updateLoadingState: function() {
                const queryDataState = this.model.get('queryData');
                if (queryDataState === QueryDataState.LOADED || queryDataState === QueryDataState.ERROR) {
                    // In case of error fail silently as Jira does
                    // Autocomplete suggestions will not work but still user can enter JQL
                    this.$el.find('.option-content-loading')
                        .addClass('hidden');
                    this.$el.find('.option-jql .option-content')
                        .removeClass('hidden');
                } else {
                    this.$el.find('.option-jql .option-content')
                        .addClass('hidden');
                    this.$el.find('.option-content-loading')
                        .removeClass('hidden');
                }
            },
        });

        const ProjectQueryOptionView = BaseOptionView.extend({
            type: DatesQueryType.PROJECT,
            events: function (){
                const events = {};
                const eventKey = 'change ' + selectors.projectPicker;
                events[eventKey] = 'onProjectSelectionChange';
                return events;
            },

            initialize: function(options) {
                this.dialogModel = options.dialogModel;
                this.projectSelect = null;

                this.listenTo(this.model, 'change:selectedProject', this._updateJiraEventSeriesName);
                this.listenTo(this.dialogModel, 'change:queryType', this.onQueryTypeChange);

                if (this.dialogModel.getQueryType() === this.type) {
                    this._initView(this.dialogModel.get('initValue'));
                }
            },

            onProjectSelectionChange: function(changeEvent) {
                this.model.set('selectedProject', changeEvent.added);
            },

            updateLoadingState: function() {
                // Project picker does not need to load query data to be ready
                this.$el.find('.option-content-loading')
                    .addClass('hidden');
            },

            _initView: function(initialValue) {
                if (this.model.get('queryData') === QueryDataState.EMPTY || !this.projectSelect) {
                    const $projectPicker = this.$el.find(selectors.projectPicker);
                    // Set up project picker
                    this.projectSelect = $projectPicker
                        .auiSelect2({
                            placeholder: translations.projectPickerPlaceholder,
                            minimumInputLength: 0,
                            allowClear: true,
                            formatSearching: '<div class="project-select-dropdown-loading"><aui-spinner size="small" class="spinner"></aui-spinner>' +
                                '<span>'+translations.projectPickerLoading + '</span></div>',
                            ajax: {
                                url: Confluence.getBaseUrl()
                                    .concat(
                                        '/rest/calendar-services/1.0/calendar/jira/',
                                        encodeURIComponent(JiraEventFieldProcess.getServerSelect(this.$el)
                                            .val()),
                                        '/projects/picker'),
                                dataType: 'json',
                                quietMillis: 300,
                                cache: true,
                                data: function(query) {
                                    return {
                                        query: query,
                                        maxResults: PROJECT_QUERY_OPTION_RESULTS_LIMIT,
                                        allowEmptyQuery: true,
                                    };
                                },
                                processResults: function(projectsResponseData, _, queryData) {
                                    if (projectsResponseData.total > 0) {
                                        const projectSelectItems = projectsResponseData.projects.map(function(projectData) {
                                            return {
                                                id: projectData.id,
                                                text: getProjectSelectOption(projectData.key, projectData.name),
                                                projectKey: projectData.key,
                                                projectName: projectData.name,
                                            };
                                        });
                                        if (projectsResponseData.total > PROJECT_QUERY_OPTION_RESULTS_LIMIT) {
                                            // Display header when not all matching options are displayed
                                            projectSelectItems.unshift({
                                                id: -1,
                                                text: AJS.I18n.getText('calendar3.jira.fields.project.message.more-results', PROJECT_QUERY_OPTION_RESULTS_LIMIT, projectsResponseData.total),
                                                disabled: true,
                                            });
                                        }
                                        return {
                                            results: projectSelectItems
                                        };
                                    } else {
                                        if (queryData.term) {
                                            return {
                                                results: []
                                            };
                                        } else {
                                            return {
                                                results: [{
                                                    id: -1,
                                                    text: AJS.I18n.getText('calendar3.jira.fields.project.no-data'),
                                                    disabled: true,
                                                }]
                                            };
                                        }
                                    }
                                },
                            },
                            formatAjaxError: function(jqXHR) {
                                const errorData = handleRequestError(jqXHR);
                                if (errorData.type === RequestErrorType.OAUTH) {
                                    $projectPicker.auiSelect2('close');
                                    return '';
                                } else {
                                    return translations.projectPickerLoadingError;
                                }
                            },
                            formatNoMatches: translations.projectPickerNoMatches,
                            formatResultCssClass: function(option) {
                                if(option.id === -1) {
                                    return 'project-select-dropdown-header';
                                }
                            },
                            initSelection: function(element, callback) {
                                const selectedProjectData = element.val();
                                const selectedOptionLabel = getProjectSelectOption(selectedProjectData.projectKey, selectedProjectData.projectName);
                                const data = {
                                    id: -1,
                                    text: selectedOptionLabel,
                                    projectKey: selectedProjectData.projectKey,
                                    projectName: selectedProjectData.projectName,
                                };
                                callback(data);
                            }
                        });

                    // Set initial selection or clear it to bring fresh state (between Jira link changes)
                    const initSelection = initialValue || '';
                    this.projectSelect.auiSelect2('val', initSelection);
                    if (initSelection) {
                        this.model.set('selectedProject', initSelection);
                    }
                    this.model.set('queryData', QueryDataState.LOADED);
                }
            },

            /**
             * Updates the Event name based on the project selection (only for Create mode).
             * If user modified the input, the name is not updated.
             * If selection is cleared, the name input is also cleared.
             * @private
             */
            _updateJiraEventSeriesName: function() {
                if (this.dialogModel.getDialogMode() === DialogMode.CREATE) {
                    const $eventNameInput = this.$el.find('input[name="name"]');
                    const selectedProjectData = this.projectSelect.auiSelect2('data');
                    const eventType = this.dialogModel.get('eventType');

                    $eventNameInput.val(function() {
                        if ($eventNameInput.data('valueChangedByUser')) {
                            return $eventNameInput.val();
                        } else if (selectedProjectData) {
                            return translations.dynamic.eventName.get(eventType)(selectedProjectData.projectName);
                        } else {
                            return '';
                        }
                    });
                    this._cleanErrorInForm();
                }
            },

            _cleanErrorInForm: function (){
                this.$el.find('.project-error')
                    .empty();
                this.$el.find('.name-error')
                    .empty();
                this.$el.find('.dateFieldName-error')
                    .empty();
            },
        });

        const FilterQueryOptionView = BaseOptionView.extend({
            type: DatesQueryType.FILTER,
            events: {
                'change input[name="searchFilter"]': 'onSearchFilterInputChange'
            },
            initialize: function(options) {
                this.dialogModel = options.dialogModel;

                this.listenTo(this.model, 'change:queryData', this.updateLoadingState);
                this.listenTo(this.dialogModel, 'change:queryType', this.onQueryTypeChange);

                if (this.dialogModel.getQueryType() === this.type) {
                    this._initView();
                }
            },

            onSearchFilterInputChange: function(changeEvent) {
                this.model.set('selectedFilter', changeEvent.target.value);
                this.model.trigger('queryDateFields');
            },

            _initView: function() {
                if (this.model.get('queryData') === QueryDataState.EMPTY) {
                    this.model.loadData(this.$el);
                }
            },

            updateLoadingState: function() {
                const queryDataState = this.model.get('queryData');
                if (queryDataState === QueryDataState.LOADED || queryDataState === QueryDataState.ERROR) {
                    this.$el.find('.option-content-loading')
                        .addClass('hidden');
                    this.$el.find('.option-search-filter .option-content')
                        .removeClass('hidden');
                    if (queryDataState === QueryDataState.ERROR) {
                        this.$el.find('.option-search-filter .option-content').addClass('disabled');
                        const savedFilterErrorContainer = this.$el.find('.unavailable-message');
                        savedFilterErrorContainer.empty().text(translations.savedFilterLoadingError);
                        this.$el.find('.search-filter-alternative').removeClass('hidden');
                    }
                } else {
                    this.$el.find('.search-filter-alternative').addClass('hidden');
                    this.$el.find('.option-search-filter .option-content')
                        .addClass('hidden');
                    this.$el.find('.option-content-loading')
                        .removeClass('hidden');
                }
            },
        });

        const JiraIssueDatesView = Backbone.View.extend({
            events: {
                'change select[name="tc-jira-server"]': 'onJiraServerChange',
                'change input[name="query-type"]': 'onQueryTypeChange'
            },

            initialize: function() {
                this.views = this._setupQueryOptionViews(this.model);

                const initialQueryType = this.model.getQueryType();
                this.$el.find('input[name="query-type"][value="' + initialQueryType + '"]')
                    .prop('checked', true);
                this._updateQueryTypeRadioChecked(initialQueryType);
            },

            onJiraServerChange: function(event) {
                // Invalidate data loaded
                this.views.forEach(function(view){
                    view.model.set('queryData', QueryDataState.EMPTY);
                });
                const activeView = this.views.get(this.model.get('queryType'));
                setTimeout(function() {
                    activeView._initView();
                });
            },

            onQueryTypeChange: function(event) {
                const selectedQueryType = event.target.value;
                this.model.set({ queryType: selectedQueryType });

                this._updateQueryTypeRadioChecked(selectedQueryType);
                this._updateActiveQueryOptionState();
            },

            cleanup: function() {
                this.views.forEach(function(view){
                    view.cleanup();
                });
                this.undelegateEvents();
            },

            reloadActiveView: function() {
                const activeView = this.views.get(this.model.get('queryType'));
                activeView.reloadQueryData();
            },

            _updateQueryTypeRadioChecked: function(selectedQueryType) {
                this.$el.find('.jira-query-options')
                    .removeClass('option-project-selected option-search-filter-selected option-jql-selected')
                    .addClass(selectedQueryType + '-selected');
            },

            _setupQueryOptionViews: function(model) {
                const jqlQueryOptionModel = new JqlQueryOptionModel();
                const projectQueryOptionModel = new ProjectQueryOptionModel();
                const filterDeprecatedDataModel = new FilterQueryOptionModel();

                this.listenTo(jqlQueryOptionModel, 'queryDateFields', this._queryDateFieldsData);
                this.listenTo(projectQueryOptionModel, 'queryDateFields', this._queryDateFieldsData);
                this.listenTo(filterDeprecatedDataModel, 'queryDateFields', this._queryDateFieldsData);

                return new Map([
                    [DatesQueryType.JQL, new JqlQueryOptionView({
                        model: jqlQueryOptionModel,
                        el: this.el,
                        dialogModel: model
                    })],
                    [DatesQueryType.PROJECT, new ProjectQueryOptionView({
                        model: projectQueryOptionModel,
                        el: this.el,
                        dialogModel: model
                    })],
                    [DatesQueryType.FILTER, new FilterQueryOptionView({
                        model: filterDeprecatedDataModel,
                        el: this.el,
                        dialogModel: model
                    })],
                ]);
            },

            _updateActiveQueryOptionState: function() {
                const activeView = this.views.get(this.model.get('queryType'));
                activeView.updateLoadingState();
                activeView.model.trigger('queryDateFields');
            },

            _queryDateFieldsData: function() {
                Confluence.TeamCalendars.setFieldErrors(this.$el, null);
                data.eventDialogView.enableOkButton();

                const selectedDateFields = this.model.get('selectedDateFields');
                const selectedDateDurations = this.model.get('selectedDateDurations');
                // As logic adds new items reset the data after first load
                // to avoid duplicated data when switching event types or query options
                this.model.set({
                    selectedDateFields: null,
                    selectedDateDurations: null
                });

                const $container = this.$el;
                const preselectDateFieldsCallback = function() {
                    const durationContainer = $('.duration', $('#jira-calendar-event-fields'));
                    durationContainer.addClass('can-add-duration');
                    if ($('.date-fields div.checkbox', durationContainer).length) {
                        durationContainer.addClass('has-selected-duration');
                    } else {
                        durationContainer.removeClass('has-selected-duration');
                    }

                    // Check selected date fields
                    const $dateFields = $container.find('.field-group-jira-show .single-dates .date-fields div.checkbox');
                    const availableDateFields = {};
                    $dateFields.each(function() {
                        const $dateFieldInput = $(this)
                            .find('input');
                        const dateFieldKey = $dateFieldInput.val();

                        if ($.inArray(dateFieldKey, selectedDateFields) !== -1) {
                            $dateFieldInput.prop('checked', true);
                        }

                        availableDateFields[dateFieldKey] = $('label', this)
                            .text();
                    });
                    // Set selected durations
                    $.each(selectedDateDurations || [],
                        function(durationIdx, subCalendarDuration) {
                            var startDateFieldLabel = availableDateFields[subCalendarDuration.startDateFieldName || ''],
                                endDateFieldLabel = availableDateFields[subCalendarDuration.endDateFieldName || ''];

                            if (startDateFieldLabel && endDateFieldLabel) {
                                // Both fields are valid
                                var durationFieldsListContainer = $('.field-group-jira-show .duration .date-fields', $container),
                                    durationFieldKey = subCalendarDuration.startDateFieldName + '/' + subCalendarDuration.endDateFieldName,
                                    durationFieldId = 'duration-' + durationFieldKey;

                                $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDateDuration({
                                    durationFieldId: durationFieldId,
                                    isChecked: true,
                                    selectedDurationFieldKey: durationFieldKey,
                                    selectedStartDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.start', startDateFieldLabel),
                                    selectedEndDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.end', endDateFieldLabel),
                                }))
                                    .appendTo(durationFieldsListContainer);

                                durationFieldsListContainer.parent()
                                    .addClass('has-selected-duration');
                            }
                        }
                    );
                };
                populateFieldsList(preselectDateFieldsCallback, $('#jira-calendar-event-fields'));
            },
        });

        const JiraSprintsOrReleasesView = Backbone.View.extend({
            initialize: function() {
                const projectQueryOptionModel = new ProjectQueryOptionModel();
                this.listenTo(projectQueryOptionModel, 'queryDateFields', this.onQueryDateFields);

                this.view = new ProjectQueryOptionView({
                    model: projectQueryOptionModel,
                    el: this.el,
                    dialogModel: this.model
                });
            },

            onQueryDateFields: function() {
                Confluence.TeamCalendars.setFieldErrors(this.$el, null);
                data.eventDialogView.enableOkButton();
            },

            cleanup: function() {
                this.view.cleanup();
                this.undelegateEvents();
            },
        });
        // endregion

        // Holds some state to avoid "props drilling" pattern
        var data = {};
        var jiraIssueDatesView = null;
        var jiraSprintsOrReleasesView = null;
        var activeView = null; // Either JiraIssueDatesView or SprintsOrReleasesView

        /**
         * Handles the request if it returned error response.
         * If it's OAuth error, displays corresponding message.
         * In case of OAuth error processing or in other case, fallbacks to provided callback.
         * @param jqXHR - jQuery XMLHttpRequest
         * @returns {{type: string, [data]: string}}
         */
        function handleRequestError(jqXHR) {
            if (Confluence.TeamCalendars.isRequireOauth(jqXHR)) {
                CalUtil.fireEventForAnalytics('oauth.client.fetch.failed');
                const oAuthUrl = JiraEventFieldProcess.showOauthRequiredMessage(jqXHR);
                if (oAuthUrl) {
                    return {
                        type: RequestErrorType.OAUTH,
                        data: oAuthUrl
                    };
                }
            }

            return { type: RequestErrorType.DEFAULT };
        }

        /**
         * Validates if Project field is selected and returns selected data.
         *
         * If so, passes the validation and returns the selected Project data.
         * If not, returns Project metadata object with `null` values so it fails eventually on BE
         * - that way the Submit still works (as it is controlled outside this file).
         * @param $container - jQuery object holding the form
         * @returns {{projectKey:string|null, projectName: string|null}}
         */
        function validateAndGetSelectedProject($container) {
            const selectedProjectData = $container.find(selectors.projectPicker).auiSelect2('data');
            if (!selectedProjectData) {
                return {
                    projectKey: 'null',
                    projectName: 'null'
                };
            }

            return selectedProjectData;
        }

        function getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, field, type) {
            const selectedProjectData = validateAndGetSelectedProject(fieldsDiv);
            if (!selectedProjectData) {
                return;
            }
            return {
                type : type,
                parentId: parentSubCalendarId,
                subCalendarId : $('input[name="subCalendarId"]', fieldsDiv).val(),
                name : name,
                description : '',
                color : $('input[name="color"]', fieldsDiv).val(),
                location : 'jira://'.concat(
                    encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv).val() || ""),
                    '?projectKey=',
                    encodeURIComponent(selectedProjectData.projectKey),
                    '&projectName=',
                    encodeURIComponent(selectedProjectData.projectName),
                    '&dateFieldName=',
                    field
                )
            };
        }

        function sendJiraCalendar(requestData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback) {
            function onSubmitStart() {
                if (!$('input[name=\'subCalendarId\']', fieldsDiv)
                    .val()) {
                    Confluence.TeamCalendars.fireEventForAnalytics(
                        'event.create',
                        {
                            eventType: requestData.type,
                            context: callbackHandler.getParameter('calendarContext')
                        }
                    );
                }
            }

            if (callbackHandler.isProcessingSubCalendar()) {
                return false;
            }

            var spinnerDefer = callbackHandler.setSubCalendarSpinnerIconVisible(true);
            onSubmitStart();

            if (callbackHandler.getIncludedCalendars()) {
                requestData.include = callbackHandler.getIncludedCalendars();
            }

            var subCalendarId = requestData.subCalendarId;
            CalUtil.putCalendarContextParams(requestData);

            $.ajax({
                cache: false,
                converters: {
                    'text json': function(jsonObject) {
                        return jsonObject;
                    }
                },
                data: requestData,
                dataFilter: function(data) {
                    var subCalendarsResponseEntity = data ? JSON.parse(data) : null;
                    if (subCalendarsResponseEntity.success) {
                        Confluence.TeamCalendars.mergeSubCalendarObjectsToArray(subCalendarsResponseEntity.payload);
                    }
                    return subCalendarsResponseEntity;
                },
                dataType: 'json',
                error: function(XMLHttpRequest, textStatus, errorThrown) {
                    callbackHandler.showAjaxUpdateError(XMLHttpRequest, textStatus, errorThrown);
                    onErrorCallback();
                },
                success: function(responseEntity) {
                    callbackHandler.setGenericUpdateError();

                    if (responseEntity.success) {
                        Confluence.TeamCalendars.setFieldErrors(fieldsDiv, null);

                        var subCalendarEdited = subCalendarId;
                        if (subCalendarEdited) {
                            callbackHandler.setSubCalendars(subCalendarId, responseEntity.payload);
                            callbackHandler.reloadSubCalendar(subCalendarId);
                        } else {
                            subCalendarId = responseEntity.modifiedSubCalendarId;
                            callbackHandler.setSubCalendars(subCalendarId, responseEntity.payload);
                        }

                        onSuccessCallback();
                        if (!responseEntity.modifiedSubCalendarId && callbackHandler.shouldShowFeatureDiscoveryDialog(subCalendarId)) {
                            callbackHandler.getSubCalendarFeatureDiscoveryDialog(subCalendarId);
                        }
                    } else {
                        Confluence.TeamCalendars.setFieldErrors(fieldsDiv, responseEntity.fieldErrors);
                        onErrorCallback();
                    }

                },
                complete: function() {
                    if (spinnerDefer) {
                        spinnerDefer.resolve();
                    }
                },
                type: 'PUT',
                timeout: Confluence.TeamCalendars.ajaxTimeout,
                url: Confluence.TeamCalendars.getCalendarServiceBaseUrl('/subcalendars.json')
            });

            return false;
        }

        /**
         * Re-instantiates data used across functions/views for each dialog opening / Event type change / Jira link change.
         * @private
         */
        function _initData(fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
            data = {
                subCalendar: JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event),
                callbackHandler: CalendarPlugin.getRenderedMacroCallbackHandler(calendarDiv),
                calendarPlugin: CalendarPlugin,
                eventDialogView: eventEditDialog,
                $dialogContainer: fieldsDiv,
            };

            $(".form-mode", fieldsDiv).removeClass('jira-query-options-loading');
            fieldsDiv.find('.oauth-error').addClass("hidden");
            eventEditDialog.enableOkButton();
            fieldsDiv.find('.query-group').removeClass('hidden');
            fieldsDiv.find('.compatibility-group, .field-group-jira-show').removeClass('hidden');
            $(".date-fields", fieldsDiv).empty();
            setDateFieldsHidden(true, fieldsDiv);
        }

        /**
         * Initializes new flow based on the Backbone structures.
         */
        function _initNewFlow() {
            const dialogMode = !data.subCalendar || !data.subCalendar.id ? DialogMode.CREATE : DialogMode.EDIT;
            var jiraSubCalendar = (data.subCalendar && data.subCalendar.sourceSubCalendar) || data.subCalendar;

            var initFilterOption;
            var selectedOptionValue;
            if (dialogMode === DialogMode.EDIT) {
                const possibleSelectedValues = [jiraSubCalendar.searchFilterId, jiraSubCalendar.jql, {projectKey: jiraSubCalendar.projectKey, projectName: jiraSubCalendar.projectName}];
                const selectedOptionIndex = possibleSelectedValues.findIndex(function(option) {
                    return option;
                });

                initFilterOption = [DatesQueryType.FILTER, DatesQueryType.JQL, DatesQueryType.PROJECT][selectedOptionIndex];
                selectedOptionValue = possibleSelectedValues[selectedOptionIndex];
            }
            const jiraDatesModel = new JiraEventDatesModel({
                mode: dialogMode,
                eventType: data.eventDialogView.getSelectedEventType(),
                queryType: initFilterOption,
                initValue: selectedOptionValue,
                selectedDateFields: (jiraSubCalendar && jiraSubCalendar.dateFieldNames) || [],
                selectedDateDurations: (jiraSubCalendar && jiraSubCalendar.durations) || []
            });

            if (jiraIssueDatesView) {
                jiraIssueDatesView.cleanup();
            }
            jiraIssueDatesView = new JiraIssueDatesView({
                model: jiraDatesModel,
                el: data.$dialogContainer.get(0)
            });
            activeView = jiraIssueDatesView;
        }

        /**
         * Shows or hides the date fields container.
         * @param {boolean} hideDateFields - whether to hide or show the date fields
         * @param fieldsDiv - jQuery container
         */
        function setDateFieldsHidden(hideDateFields, fieldsDiv) {
            const dateFieldsGroup = $('.field-group-jira-show', fieldsDiv);
            if (hideDateFields) {
                dateFieldsGroup.hide();
            } else {
                dateFieldsGroup.show();
            }
        }

        function populateFieldsList(successCallback) {
            const $container = data.$dialogContainer;

            const selectedDateFields = $('.field-group-jira-show div.checkbox', $container)
                .filter(function() {
                    return $('input', this)
                        .is(':checked');
                });
            var dateFieldsUrl = '';

            const selectedQuery = $('.jira-query-options .matrix .radio:checked', $container)
                .val();
            const durationContainer = $('.duration', $container);
            const dateFieldsData = {};

            // Figure out how to query for date fields
            if ('option-project' === selectedQuery) {
                const selectedProjectData = $container.find(selectors.projectPicker).auiSelect2('data');
                if (selectedProjectData) {
                    dateFieldsUrl = '/project/' + encodeURIComponent(selectedProjectData.projectKey);
                }
            } else if ('option-search-filter' === selectedQuery) {
                const searchFilterId = $('.jira-query-options .option-search-filter input[name=\'searchFilter\']', $container)
                    .val();
                if (searchFilterId) {
                    dateFieldsUrl = '/filter/' + encodeURIComponent(searchFilterId);
                }
            } else if ('option-jql' === selectedQuery) {
                dateFieldsData.query = $('.jira-query-options .option-jql textarea', $container)
                    .val() || '';
                if (dateFieldsData.query) {
                    dateFieldsUrl = '/jql';
                }
            }

            durationContainer.removeClass('can-add-duration');

            if (dateFieldsUrl) {
                const dateFieldsSpinnerContainer = $('.date-fields-spinner-container', $container)
                        .removeClass('hidden'),
                    singleDateFieldsListContainer = $('.single-dates .date-fields', $container)
                        .empty(),
                    durationDateFieldsListContainer = $('.date-fields', durationContainer)
                        .empty();

                $.ajax({
                    cache: false,
                    data: dateFieldsData,
                    dataType: 'json',
                    error: function() {
                        dateFieldsSpinnerContainer.addClass('hidden');
                        $('.dateFieldName-error', $container)
                            .text(translations.dateFieldsLoadingError);
                    },
                    success: function(dateFields) {
                        dateFieldsSpinnerContainer.addClass('hidden');

                        const singleDateFieldsList = $('<div/>'),
                            selectedDateFieldKeys = $.map(selectedDateFields, function(selectedDateField) {
                                return $('input', selectedDateField)
                                    .val();
                            }),
                            selectableDateFieldKeys = $.map(dateFields, function(dateField) {
                                return dateField.key;
                            });

                        // Repopulate single date fields
                        function appendDateFieldOption(dateFieldIdx, dateField) {
                            const isChecked = $.inArray(dateField.key, selectedDateFieldKeys) !== -1;
                            var warning;
                            if (!dateField.isSearchable) {
                                warning = translations.dateFieldNotSearchable;
                            }
                            $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDatefield({
                                dateField: dateField,
                                isChecked: isChecked,
                                warning: warning
                            }))
                                .data('mappableAsDuration',
                                    dateField.customfield ||
                                    dateField.key === 'duedate' ||
                                    dateField.key === 'created' ||
                                    dateField.key === 'resolution')
                                .appendTo(singleDateFieldsList);
                        }

                        $.each(dateFields, appendDateFieldOption);

                        // Repopulate durations
                        selectedDateFields.filter(function() {
                            const durationFields = $('input', this)
                                .val();
                            if (durationFields.indexOf('/') !== -1) {
                                const durationFieldTokens = durationFields.split('/', 2);
                                return durationFieldTokens.length === 2 &&
                                    $.inArray(durationFieldTokens[0], selectableDateFieldKeys) !== -1 &&
                                    $.inArray(durationFieldTokens[1], selectableDateFieldKeys) !== -1;
                            }

                            return false;
                        })
                            .appendTo(
                                // Empty the container again because it may have been populated by another instance of this AJAX request
                                // after the container was initially emptied (just before the start of this AJAX request).
                                durationDateFieldsListContainer.empty()
                            );

                        singleDateFieldsList.appendTo(
                            // See comment regarding emptying durationDateFieldsListContainer
                            singleDateFieldsListContainer.empty()
                        );

                        if (singleDateFieldsList.children().length) {
                            setDateFieldsHidden(false, $container);
                            durationContainer.addClass('can-add-duration');
                        } else {
                            durationContainer.removeClass('can-add-duration');
                        }

                        if (successCallback && $.isFunction(successCallback)) {
                            successCallback(dateFields);
                        }
                    },
                    timeout: Confluence.TeamCalendars.ajaxTimeout,
                    url: Confluence.getBaseUrl()
                        .concat(
                            '/rest/calendar-services/1.0/calendar/jira/',
                            encodeURIComponent(JiraEventFieldProcess.getServerSelect($container)
                                .val()),
                            '/date-fields',
                            dateFieldsUrl,
                            '.json')
                });
            } else {
                setDateFieldsHidden(true, $container);
                if (successCallback && $.isFunction(successCallback)) {
                    successCallback();
                }
            }
        }

        function populateForm() {
            const $container = data.$dialogContainer;

            var formMode = $('.form-mode', $container),
                serverSelect = JiraEventFieldProcess.getServerSelect($container),
                projectOption = $('.jira-query-options .option-project', $container),
                projectSelect = $('select[name=\'project\']', projectOption),
                timeZoneIdDisplay = $('.timeZoneIdDisplay', $container),
                helper = Confluence.TeamCalendars.getEditCalendarDialogHelper($container, data.subCalendar, data.callbackHandler);

            Confluence.TeamCalendars.setFieldErrors($container, null);

            if (!data.subCalendar || !data.subCalendar.id) {
                helper.setFormModeCreate();
                helper.setSubCalendarIdField('');
                helper.setNameField('');
                helper.setDescriptionField('');
                helper.setColorField('');
                helper.setSpaceToDefault();
                helper.setTimeZoneIdDisplayField('');
                timeZoneIdDisplay.hide();
            } else {
                helper.setFormModeEdit();
                helper.setSubCalendarIdField(data.subCalendar.id);
                helper.setNameField(data.subCalendar.name);
                helper.setDescriptionField(data.subCalendar.description);
                helper.setColorField(data.subCalendar.color);
                helper.setSpaceKeyField(data.subCalendar.spaceKey);
                helper.setSpaceKeyAutocompleteField(data.subCalendar.spaceName);

                timeZoneIdDisplay.text(data.subCalendar.timeZoneId);
            }

            //Set read / read only
            const textInputs = $('input[name=\'name\'], input[name=\'description\'], input[name=\'spaceKeyAutocomplete\'], textarea[name=\'jql\']', $container);
            const searchFilterInput = $('input[name=\'searchFilter\']', $container);
            if (data.callbackHandler.isCalendarInEditMode()) {
                textInputs.prop('readonly', false);
                serverSelect.prop('disabled', false);
                projectSelect.prop('disabled', false);
                JiraEventFieldProcessCommon.enableSearchFilterSelect(searchFilterInput);
            } else {
                formMode.removeClass()
                    .addClass('form-mode read-only');
                textInputs.prop('readonly', true);
                serverSelect.prop('disabled', true);
                projectSelect.prop('disabled', true);
                JiraEventFieldProcessCommon.disableSearchFilterSelect(searchFilterInput);
            }
        }

        function setupFields() {
            const $container = data.$dialogContainer;

            $('input[name=\'spaceKeyAutocomplete\']', $container)
                .bind('selected.autocomplete-content', function(e, selection) {
                    $('input[name=\'spaceKey\']', $container)
                        .val(selection.content.key);
                    $(this)
                        .val(selection.content.name);
                    $('.spaceKeyAutocomplete-error', $container)
                        .empty();
                });

            $('select[name=\'project\']', $container)
                .change(function() {
                    populateFieldsList(null, $container);
                });

            $('.add-duration', $container)
                .unbind()
                .click(function() {
                    AJS.log('Open Jira Duration Dialog');
                    var addDurationDialog = new AJS.Dialog({
                        'height': 423,
                        'width': 520,
                        'id': 'add-jira-duration-dialog'
                    });

                    addDurationDialog.addHeader(translations.addDurationDialogTitle);
                    addDurationDialog.addPanel(
                        '',
                        Confluence.TeamCalendars.Templates.editJiraSubCalendarDurations(),
                        'calendar-dialog-panel'
                    );

                    var addDurationDialogPanel = addDurationDialog.getCurrentPanel().body,
                        durationFieldsListContainer = $('.field-group-jira-show .duration .date-fields', $container),
                        dateFieldsListContainer = $('.field-group-jira-show .single-dates .date-fields', $container);

                    addDurationDialog.addSubmit(translations.wordAdd, function() {
                        var selectedStartDateFieldOption = $('.duration-start select option:selected', addDurationDialogPanel),
                            selectedEndDateFieldOption = $('.duration-end select option:selected', addDurationDialogPanel),
                            selectedDurationFieldKey = selectedStartDateFieldOption.val() + '/' + selectedEndDateFieldOption.val(),
                            selectedDurationField = $('div.checkbox', durationFieldsListContainer)
                                .filter(function() {
                                    return $('input', this)
                                        .val() === selectedDurationFieldKey;
                                });

                        if (selectedDurationField.length) {
                            // Retick, just in case the user ticked it off when opening up the add durations dialog.
                            $('input', selectedDurationField)
                                .prop('checked', true);
                        } else {
                            var durationFieldId = 'duration-' + selectedDurationFieldKey;

                            $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDateDuration({
                                durationFieldId: durationFieldId,
                                isChecked: true,
                                selectedDurationFieldKey: selectedDurationFieldKey,
                                selectedStartDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.start', selectedStartDateFieldOption.html()),
                                selectedEndDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.end', selectedEndDateFieldOption.html())
                            }))
                                .prependTo(durationFieldsListContainer);
                            durationFieldsListContainer.parent()
                                .addClass('has-selected-duration');
                        }

                        addDurationDialog.remove();
                        data.eventDialogView.show();
                    });

                    addDurationDialog.addCancel(translations.wordCancel, function() {
                        addDurationDialog.remove();
                        data.eventDialogView.show();
                    });

                    var mappableDateFields = $.map(
                            $('div.checkbox', dateFieldsListContainer)
                                .filter(function() {
                                    return $(this)
                                        .data('mappableAsDuration');
                                }),
                            function(aField) {
                                return $('<option/>', {
                                    'value': $('input', aField)
                                        .val(),
                                    'text': $('label', aField)
                                        .text()
                                });
                            }
                        ),
                        startDateFieldSelect = $('.duration-start select', addDurationDialogPanel),
                        endDateFieldSelect = $('.duration-end select', addDurationDialogPanel),
                        setSubmitButtonDisabledState = function(startField, endField) {
                            var okButton = addDurationDialog.popup.element.find('.button-panel-submit-button');

                            if (endField && startField && endField !== startField) {
                                okButton.prop('disabled', false)
                                    .removeClass('ui-state-disabled');
                            } else {
                                okButton.prop('disabled', true)
                                    .addClass('ui-state-disabled');
                            }
                        };

                    // Populate each date field dropdowns for selection
                    $.each(mappableDateFields, function(mappableDateFieldIdx, mappableDateField) {
                        $(mappableDateField)
                            .appendTo(startDateFieldSelect)
                            .clone()
                            .appendTo(endDateFieldSelect);
                    });

                    if (!mappableDateFields.length) {
                        $('.duration-fields', addDurationDialogPanel)
                            .addClass('hidden');
                        $('.add-fields', $('.no-selectable-date-fields', addDurationDialogPanel)
                            .removeClass('hidden'))
                            .html(
                                AJS.I18n.getText(
                                    'calendar3.jira.duration.fields.addfields',
                                    $('.field-group-jira-server select option:checked', $container)
                                        .data('display-url')
                                )
                            );
                    }

                    var adjustSelectableDateFieldOptions = function(selectedDateField, otherDateFieldSelect) {
                        var initialSelectedEndDateField = $('option:selected', otherDateFieldSelect)
                            .val();

                        // Remove all date fields.
                        $('option', otherDateFieldSelect)
                            .filter(function() {
                                return $(this)
                                    .val();
                            })
                            .remove();

                        $.each(mappableDateFields, function(mappableDateFieldIdx, mappableDateField) {
                            if (mappableDateField.val() !== selectedDateField) {
                                mappableDateField.clone()
                                    .appendTo(otherDateFieldSelect);
                            }
                        });

                        $('option', otherDateFieldSelect)
                            .prop('selected', false)
                            .filter(
                                function() {
                                    return $(this)
                                        .val() === initialSelectedEndDateField;
                                }
                            )
                            .prop('selected', true);
                    };

                    startDateFieldSelect.change(function() {
                        var selectedStartDateField = $('option:selected', this)
                            .val();

                        adjustSelectableDateFieldOptions(selectedStartDateField, endDateFieldSelect);
                        setSubmitButtonDisabledState(selectedStartDateField, $('option:selected', endDateFieldSelect)
                            .val());
                    });

                    endDateFieldSelect.change(function() {
                        var selectedEndDateField = $('option:selected', this)
                            .val();

                        adjustSelectableDateFieldOptions(selectedEndDateField, startDateFieldSelect);
                        setSubmitButtonDisabledState(selectedEndDateField, $('option:selected', startDateFieldSelect)
                            .val());
                    });

                    data.eventDialogView.hide();
                    setSubmitButtonDisabledState();
                    addDurationDialog.show();
                    return false;
                });

            $('.subcalendar-cancel', $container)
                .click(function() {
                    data.eventDialogView.remove();
                    return false;
                });

            return $container;
        }

        const JiraEventFieldProcess = {
            getServerSelect: function(fieldsDiv) {
                return $('.tc-jira-server', fieldsDiv);
            },

            getSubCalendarForEvent: function(CalendarPlugin, calendarDiv, event) {
                if (event.eventType === 'jira-calendar' ||
                    event.eventType === 'jira-agile-sprint-calendar' ||
                    event.eventType === 'jira-project-releases-calendar') {
                    //This is not a new event, so get the child subCalendar it is associated with
                    return CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId);
                }

                return null;
            },

            /**
             * Attempts to display "OAuth required" message assuming the response is in JSON format
             * and contains `oAuthUrl` property.
             * @param jqXHR
             * @returns {string|null} - Either OAuth URL if message was successfully displayed, or `null` otherwise.
             */
            showOauthRequiredMessage: function(jqXHR) {
                var responseEntity;
                try {
                    responseEntity = jqXHR.responseText ? JSON.parse(jqXHR.responseText) : null;
                    if(!(responseEntity && responseEntity.oAuthUrl)) {
                        return null;
                    }
                } catch (e) {
                    return null;
                }
                const $container = data.$dialogContainer;
                const eventEditDialog = data.eventDialogView;
                const controlsToToggleVisibleOnOAuthError = $('.compatibility-group, .field-group-jira-show', $container);
                const $queryFilterOptions = $(".query-group", $container);
                $queryFilterOptions.addClass('hidden');
                const oAuthErrorContainer = $('.oauth-error', $container)
                    .empty()
                    .removeClass('hidden');
                controlsToToggleVisibleOnOAuthError.addClass('hidden');

                eventEditDialog.disableOkButton({ 'buttonText': translations.wordOk });
                $(document.createElement('div'))
                    .addClass('oauth-approve aui-message aui-message-hint hint')
                    .html(
                        AJS.format(
                            translations.oauthApprovalMessage,
                            $(document.createElement('span'))
                                .text(translations.oauthLoginApproveText)
                                .html()
                        )
                    )
                    .appendTo(oAuthErrorContainer)
                    .find('.approve-dance')
                    .click(function() {
                        CalUtil.fireEventForAnalytics('oauth.client.request.attempt');
                        AppLinks.authenticateRemoteCredentials(
                            responseEntity.oAuthUrl,
                            function() {
                                CalUtil.fireEventForAnalytics('oauth.client.request.succeed');
                                controlsToToggleVisibleOnOAuthError.removeClass('hidden');
                                oAuthErrorContainer.addClass('hidden');
                                $queryFilterOptions.removeClass('hidden');
                                eventEditDialog.enableOkButton();

                                if (activeView === jiraIssueDatesView) {
                                    // Force selected query option to reload data
                                    activeView.reloadActiveView();
                                }
                            },
                            function() {
                                CalUtil.fireEventForAnalytics('oauth.client.request.failed');
                                alert(translations.oauthApprovalUnsuccessful);
                            }
                        );

                        return false;
                    }).focus();

                return responseEntity.oAuthUrl;
            },

            /**
             * Initialize dialog content for "Jira Agile Sprints" event.
             * Called also on Jira link change within the form.
             */
            populateProjectList : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                _initData(fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                const jiraEventType = eventEditDialog.getSelectedEventType();
                const jiraSubCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event);
                const dialogMode = !jiraSubCalendar ? DialogMode.CREATE : DialogMode.EDIT;
                var selectedOptionValue;
                if (dialogMode === DialogMode.EDIT) {
                    selectedOptionValue = {projectKey: jiraSubCalendar.projectKey, projectName: jiraSubCalendar.projectName};
                }
                const jiraSprintOrReleaseModel = new JiraEventDatesModel({
                    mode: dialogMode,
                    eventType: jiraEventType,
                    queryType: DatesQueryType.PROJECT,
                    initValue: selectedOptionValue,
                });
                jiraSprintsOrReleasesView = new JiraSprintsOrReleasesView({
                    model: jiraSprintOrReleaseModel,
                    el: fieldsDiv.get(0),
                });
                activeView = jiraSprintsOrReleasesView;

                $(".field-group-jira-project .spinner").hide();
                const jiraProjectSelector = $("div.tc-jira-project", fieldsDiv);
                FormStateControl.enableElement(jiraProjectSelector);
            },

            /**
             * Initialize dialog content for "Jira Issue Dates" event.
             * Called also on Jira link change within the form.
             **/
            initJiraDate: function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                _initData(fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                _initNewFlow();

                setupFields();
                populateForm();
            },

            CustomSubmitter: {
                JiraIssues: {
                    submitForm: function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        function getFormDataAsAjaxData() {
                            var dateFieldQueryParam = '',
                                selectedDateFields = $('.field-group-jira-show .date-fields div.checkbox')
                                    .filter(function() {
                                        return $('input:checked', this).length;
                                    });

                            //Build date field query
                            $.each(selectedDateFields, function() {
                                var dateFieldKey = $('input', this)
                                    .val();

                                //handle durations
                                if (dateFieldKey.indexOf('/') !== -1) {
                                    var durationFieldTokens = dateFieldKey.split('/', 2);
                                    if (dateFieldQueryParam) {
                                        dateFieldQueryParam += '&';
                                    }

                                    dateFieldQueryParam += 'duration=' + encodeURIComponent(durationFieldTokens[0] + '/' + durationFieldTokens[1]);
                                } else {  //Single dates
                                    if (dateFieldQueryParam) {
                                        dateFieldQueryParam += '&';
                                    }

                                    dateFieldQueryParam += 'dateFieldName=' + encodeURIComponent(dateFieldKey);
                                }
                            });

                            var data = {
                                    type: 'jira',
                                    parentId: parentSubCalendarId,
                                    subCalendarId: $('input[name=\'subCalendarId\']', fieldsDiv)
                                        .val(),
                                    name: name,
                                    description: '',
                                    color: $('input[name=\'color\']', fieldsDiv)
                                        .val()
                                },
                                location = 'jira://' + encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv)
                                    .val() || '') + '?',
                                selectedQueryOption = $('.jira-query-options .matrix input.radio:checked', fieldsDiv)
                                    .val();

                            if ('option-project' === selectedQueryOption) {
                                const selectedProjectData = validateAndGetSelectedProject(fieldsDiv);
                                if (!selectedProjectData) {
                                    return;
                                }
                                location += 'projectKey=' + encodeURIComponent(selectedProjectData.projectKey)+'&projectName='+encodeURIComponent(selectedProjectData.projectName);
                            } else if ('option-search-filter' === selectedQueryOption) {
                                const searchFilterId = $('input[name=\'searchFilter\']', fieldsDiv).val()
                                location += 'searchFilterId=' + searchFilterId;
                            } else {
                                location += 'jql=' + encodeURIComponent($('textarea[name=\'jql\']', fieldsDiv)
                                    .val());
                            }

                            if (dateFieldQueryParam) {
                                location += '&' + dateFieldQueryParam;
                            }

                            data.location = location;
                            return data;
                        }
                        const formData = getFormDataAsAjaxData();
                        if (formData) {
                            sendJiraCalendar(formData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                        }
                    }
                },
                JiraAgileSprints : {
                    submitForm : function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        const formData = getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, 'sprint', 'jira-agile-sprint');
                        if (formData) {
                            sendJiraCalendar(formData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                        }
                    }
                },
                JiraProjectReleases : {
                    submitForm : function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        const formData = getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, 'versiondue', 'jira-project-releases');
                        if (formData) {
                            sendJiraCalendar(formData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                        }
                    }
                }
            }
        };

        return JiraEventFieldProcess;
    }
);
