/**
 * Module responsible for handling form for "Jira Issue Dates" event type in Team Calendar dialog.
 * It is the old process where all three filter query options (Project, Saved Filter, JQL) are combined
 * and served through single HTTP request.
 *
 * See {@link module:tc/jira-event-field-process-divided}
 * for improved process where query filter options are divided.
 *
 * NOTE: Endpoint used on Jira side is not performant and is causing time outs for enterprise customers.
 */
define("tc/jira-event-field-process-united",
    [
        "jquery",
        "tc/calendar-util",
        "tc/form-state-control",
        'tc/jira-event-field-process-common'
    ],
    function (
        $,
        CalUtil,
        FormStateControl,
        JiraEventFieldProcessCommon
    )
    {
        "use strict";
        function sendJiraCalendar(requestData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback) {
            function onSubmitStart() {
               if (!$("input[name='subCalendarId']", fieldsDiv).val()) {
                   Confluence.TeamCalendars.fireEventForAnalytics(
                       "event.create",
                       {eventType: requestData.type, context: callbackHandler.getParameter("calendarContext")}
                   );
               }
            }

            if (callbackHandler.isProcessingSubCalendar()) {
               return false;
            }

            var spinnerDefer = callbackHandler.setSubCalendarSpinnerIconVisible(true);
            onSubmitStart();

            if (callbackHandler.getIncludedCalendars()) {
               requestData.include = callbackHandler.getIncludedCalendars();
            }

            var subCalendarId = requestData.subCalendarId;
            CalUtil.putCalendarContextParams(requestData);

            $.ajax({
               cache : false,
               converters : {
                   "text json" : function(jsonObject) {
                       return jsonObject;
                   }
               },
               data : requestData,
               dataFilter : function(data) {
                   var subCalendarsResponseEntity = data ? JSON.parse(data) : null;
                   if (subCalendarsResponseEntity.success) {
                       Confluence.TeamCalendars.mergeSubCalendarObjectsToArray(subCalendarsResponseEntity.payload);
                   }
                   return subCalendarsResponseEntity;
               },
               dataType : "json",
               error : function(XMLHttpRequest, textStatus, errorThrown) {
                   callbackHandler.showAjaxUpdateError(XMLHttpRequest, textStatus, errorThrown);
                   onErrorCallback();
               },
               success : function(responseEntity) {
                  callbackHandler.setGenericUpdateError();

                   if (responseEntity.success) {
                       Confluence.TeamCalendars.setFieldErrors(fieldsDiv, null);

                       var subCalendarEdited = subCalendarId;
                       if (subCalendarEdited) {
                           callbackHandler.setSubCalendars(subCalendarId, responseEntity.payload);
                           callbackHandler.reloadSubCalendar(subCalendarId);
                       } else {
                           subCalendarId = responseEntity.modifiedSubCalendarId;
                           callbackHandler.setSubCalendars(subCalendarId, responseEntity.payload);
                       }

                       onSuccessCallback();
                       if (!responseEntity.modifiedSubCalendarId && callbackHandler.shouldShowFeatureDiscoveryDialog(subCalendarId)) {
                           callbackHandler.getSubCalendarFeatureDiscoveryDialog(subCalendarId);
                       }
                   } else {
                       Confluence.TeamCalendars.setFieldErrors(fieldsDiv, responseEntity.fieldErrors);
                       onErrorCallback();
                   }

               },
                complete: function(){
                    if (spinnerDefer) spinnerDefer.resolve();
                },
               type : "PUT",
               timeout: Confluence.TeamCalendars.ajaxTimeout,
               url : Confluence.TeamCalendars.getCalendarServiceBaseUrl("/subcalendars.json")
            });

            return false;
        }

        function getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, field, type) {
            return {
               type : type,
               parentId: parentSubCalendarId,
               subCalendarId : $("input[name='subCalendarId']", fieldsDiv).val(),
               name : name,
               description : "",
               color : $("input[name='color']", fieldsDiv).val(),
               location : "jira://".concat(
                               encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv).val() || ""),
                               "?projectKey=",
                               encodeURIComponent($("select[name='tc-jira-project'] option:selected", fieldsDiv).val()),
                               "&dateFieldName=",
                               field
                           )
            };
        }

        var JiraEventFieldProcess = {
            getServerSelect : function(fieldsDiv) {
                return $(".tc-jira-server", fieldsDiv);
            },

            getJiraOptions : function(onError, onSuccess, server) {
                $.ajax({
                    cache:    false,
                    dataType: "json",
                    error:    function (XMLHttpRequest, textStatus, errorThrown) {
                        onError(XMLHttpRequest, textStatus, errorThrown);
                    },
                    success:  function (queryOptions) {
                        onSuccess(queryOptions);
                        var message = AJS.I18n.getText("calendar3.jira.error.project.nonefound", $(".tc-jira-server option:selected").data("display-url"));
                        $(".option-project .unavailable-message").html(message);
                    },
                    timeout:  Confluence.TeamCalendars.ajaxTimeout,
                    url:      Confluence.getBaseUrl() + "/rest/calendar-services/1.0/calendar/jira/" + encodeURIComponent(server) + "/query/options.json"
                });
            },

            getSubCalendarForEvent : function(CalendarPlugin, calendarDiv, event) {
                if (event.eventType === "jira-calendar" ||
                    event.eventType === "jira-agile-sprint-calendar" ||
                    event.eventType === "jira-project-releases-calendar") {
                        //This is not a new event, so get the child subCalendar it is associated with
                        return CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId);
                }

                return null;
            },

            showOauthRequitedMessage : function(XMLHttpRequest, fieldsDiv, eventEditDialog, controlsToToggleVisibleOnOAuthError, populateForm) {
                var responseEntity = XMLHttpRequest.responseText ? JSON.parse(XMLHttpRequest.responseText) : null;
                var oAuthErrorContainer = $(".oauth-error", fieldsDiv).empty().removeClass("hidden");

                eventEditDialog.disableOkButton({"buttonText" : AJS.I18n.getText("ok") });
                $(document.createElement("div")).addClass("oauth-approve aui-message aui-message-hint hint").html(
                                AJS.format(
                                        AJS.I18n.getText("calendar3.oauth.loginapprovetoseeprojects"),
                                        $(document.createElement("span")).text(AJS.I18n.getText("calendar3.oauth.loginandauthorize")).html()
                                )
                        ).appendTo(oAuthErrorContainer).find(".approve-dance").click(function () {
                            CalUtil.fireEventForAnalytics("oauth.client.request.attempt");
                            AppLinks.authenticateRemoteCredentials(
                                    responseEntity.oAuthUrl,
                                    function () {
                                        CalUtil.fireEventForAnalytics("oauth.client.request.succeed");
                                        controlsToToggleVisibleOnOAuthError.removeClass("hidden");
                                        oAuthErrorContainer.addClass("hidden");
                                        eventEditDialog.enableOkButton();
                                        populateForm();
                                    },
                                    function () {
                                        CalUtil.fireEventForAnalytics("oauth.client.request.failed");
                                        alert(AJS.I18n.getText("calendar3.oauth.trydancingagain"));
                                    }
                            );

                            return false;
                        });
            },

            populateProjectList : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                var jiraProjectSelector = $(".tc-jira-project", fieldsDiv),
                    spinner = $(".field-group-jira-project .spinner"),
                    jiraEventType = eventEditDialog.getSelectedEventType(),
                    server = JiraEventFieldProcess.getServerSelect().val(),

                    updateEventSeriesNameField = function() {
                        var nameInputField = $("input[name='name']", fieldsDiv),
                            subCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event),

                            getName = function(computedName) {
                                if (!nameInputField.data("valueChangedByUser")) {
                                    return computedName;
                                }

                                return nameInputField.val();
                            };

                        if (!subCalendar) {
                            // Only do this when creating a new JIRA event type.
                            if (jiraEventType === "jira-agile-sprint-calendar") {
                                nameInputField.val(getName(AJS.I18n.getText("calendar3.event.field.name.jira.sprints", $("option:selected", jiraProjectSelector).text())));
                            } else if (jiraEventType === "jira-project-releases-calendar") {
                                nameInputField.val(getName(AJS.I18n.getText("calendar3.event.field.name.jira.releases", $("option:selected", jiraProjectSelector).text())));
                            }
                        }
                    };

                jiraProjectSelector.change(updateEventSeriesNameField);
                FormStateControl.disableElement(jiraProjectSelector);
                spinner.show();

                JiraEventFieldProcess.getJiraOptions(
                    function(jqXHR, textStatus, errorThrown) {
                        spinner.hide();

                        if (Confluence.TeamCalendars.isRequireOauth(jqXHR)) {
                            CalUtil.fireEventForAnalytics("oauth.client.fetch.failed");
                            jiraProjectSelector.empty();
                            JiraEventFieldProcess.showOauthRequitedMessage(jqXHR, fieldsDiv, eventEditDialog, jiraProjectSelector, function() {
                                JiraEventFieldProcess.populateProjectList(server);
                            });
                        } else {
                            FormStateControl.enableElement(jiraProjectSelector);
                            CalendarPlugin.showAjaxError(fieldsDiv.parent().parent(), jqXHR, textStatus, errorThrown);
                        }
                    },
                    function(queryOptions) {
                        var projects = queryOptions.projects,
                            projectList = jiraProjectSelector.empty(),
                            subCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event);

                        spinner.hide();
                        FormStateControl.enableElement(jiraProjectSelector);

                        $.each(projects, function (projectIdx, project) {
                            $(document.createElement("option")).val(project.key).text(project.name).appendTo(projectList);
                        });

                        if (subCalendar) {
                            projectList.val(subCalendar.projectKey);
                        } else {
                            updateEventSeriesNameField();
                        }
                    },
                    server
                );
            },

            initJiraDate : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                //Holy crap there is a lot of code here! A brave person should try to refactor it.
                var subCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event),
                    callbackHandler = CalendarPlugin.getRenderedMacroCallbackHandler(calendarDiv),

                    projectSelect = $("select[name='project']", fieldsDiv),
                    updateJiraEventSeriesName = function() {
                        if (!subCalendar) {
                            var nameInputField = $("input[name='name']", fieldsDiv);

                            // Only do this for new JIRA event type
                            nameInputField.val(function() {
                                if (!nameInputField.data("valueChangedByUser")) {
                                    return $("option:selected", projectSelect).val() ? AJS.I18n.getText('calendar3.event.field.name.jira.issues', $("option:selected", projectSelect).text()) : "";
                                }

                                return nameInputField.val();
                            });

                            cleanErrorInForm();
                        }
                    };

                projectSelect
                    .off("change")
                    .on("change", updateJiraEventSeriesName);

                var cleanErrorInForm = function() {
                    //clean error
                    $(".project-error", fieldsDiv).empty();
                    $(".name-error", fieldsDiv).empty();
                    $(".dateFieldName-error", fieldsDiv).empty();
                };

                var setDateFieldsHidden = function(hideDateFields) {
                        var dateFieldsGroup = $(".field-group-jira-show", fieldsDiv);
                        if (hideDateFields) {
                            dateFieldsGroup.css("display", "none");
                        } else {
                            dateFieldsGroup.css("display", "block");
                        }
                    };

                var populateForm = function() {
                    var formMode = $(".form-mode", fieldsDiv),
                        serverSelect = JiraEventFieldProcess.getServerSelect(fieldsDiv),
                        projectOption = $(".jira-query-options .option-project", fieldsDiv),
                        projectSelect = $("select[name='project']", projectOption),
                        timeZoneIdDisplay = $(".timeZoneIdDisplay", fieldsDiv),
                        helper = Confluence.TeamCalendars.getEditCalendarDialogHelper(fieldsDiv, subCalendar, callbackHandler);

                    Confluence.TeamCalendars.setFieldErrors(fieldsDiv, null);

                    $("textarea[name='jql']", fieldsDiv).addClass("with-hint").val(AJS.I18n.getText("calendar3.jira.fields.jql.hint"));

                    if (!subCalendar || !subCalendar.id) {
                        helper.setFormModeCreate();
                        helper.setSubCalendarIdField("");
                        helper.setNameField("");
                        helper.setDescriptionField("");
                        helper.setColorField("");
                        helper.setSpaceToDefault();
                        helper.setTimeZoneIdDisplayField("");
                        timeZoneIdDisplay.hide();

                        populateQueryOptions(null, function () {
                            populateFieldsList(function () {
                                updateJiraEventSeriesName();
                            });
                        });
                    }
                    else
                    {
                        helper.setFormModeEdit();
                        helper.setSubCalendarIdField(subCalendar.id);
                        helper.setNameField(subCalendar.name);
                        helper.setDescriptionField(subCalendar.description);
                        helper.setColorField(subCalendar.color);
                        helper.setSpaceKeyField(subCalendar.spaceKey);
                        helper.setSpaceKeyAutocompleteField(subCalendar.spaceName);

                        populateQueryOptions(null, function () {
                            var jiraQueryOptionsContainer = $(".jira-query-options", fieldsDiv),
                                allPossibleSelectedOptionsClasses = "option-project-selected option-search-filter-selected option-jql-selected",
                                hideDateFields;

                            $(".matrix input.radio", jiraQueryOptionsContainer).prop("checked", false);

                            var jiraSubCalendar = subCalendar.sourceSubCalendar || subCalendar;

                            if (jiraSubCalendar.projectKey) {
                                var isProjectInList = false;

                                //Set the project selector
                                $("option", projectSelect).prop("selected", false).each(function () {
                                    var aProjectOption = $(this);
                                    if (aProjectOption.val() === jiraSubCalendar.projectKey) {
                                        aProjectOption.prop("selected", true);
                                        isProjectInList = true;
                                        return false;
                                    }
                                });

                                //If project not found append to selector
                                if (!isProjectInList) {
                                    projectSelect.append(
                                            $(document.createElement("option")).val(jiraSubCalendar.projectKey).text(
                                                jiraSubCalendar.projectName || jiraSubCalendar.projectKey
                                            ).prop("selected", true)
                                    );
                                }

                                $(".radio-project input.radio", jiraQueryOptionsContainer.removeClass(allPossibleSelectedOptionsClasses).addClass("option-project-selected")).prop("checked", true);
                                hideDateFields = projectOption.data("hiddenDateFields");

                            } else if (jiraSubCalendar.searchFilterId) {
                                //Set search filter
                                var searchFilterOption = $(".option-search-filter", jiraQueryOptionsContainer);

                                const searchFilterInput = $("input[name='searchFilter']", searchFilterOption);
                                JiraEventFieldProcessCommon.setValueOfSearchFilterSelect(searchFilterInput, jiraSubCalendar.searchFilterId);

                                $(".radio-search-filter input.radio", jiraQueryOptionsContainer.removeClass(allPossibleSelectedOptionsClasses).addClass("option-search-filter-selected")).prop("checked", true);
                                hideDateFields = searchFilterOption.data("hideDateFields");

                            } else if (jiraSubCalendar.jql) {
                                //Set JQL
                                var jqlOption = $(".option-jql", jiraQueryOptionsContainer);

                                $("textarea[name='jql']", jqlOption).removeClass("with-hint").val(jiraSubCalendar.jql);
                                $(".radio-jql input.radio", jiraQueryOptionsContainer.removeClass(allPossibleSelectedOptionsClasses).addClass("option-jql-selected")).prop("checked", true);
                            }

                            populateFieldsList(function () {
                                var singleDateFields = $(".field-group-jira-show .single-dates .date-fields div.checkbox", fieldsDiv),
                                    availableDateFields = {},
                                    jiraCalendarDateFields = jiraSubCalendar.dateFieldNames || [];

                                $(".date-fields input", fieldsDiv).prop("checked", false);

                                //Set single date fields
                                singleDateFields.each(function() {
                                    var dateFieldInput = $("input", this),
                                        dateFieldKey = dateFieldInput.val();

                                    if ($.inArray(dateFieldKey, jiraCalendarDateFields) !== -1) {
                                        dateFieldInput.prop("checked", true);
                                    }

                                    availableDateFields[dateFieldKey] = $("label", this).text();
                                });

                                //Set durations
                                $.each(
                                    jiraSubCalendar.durations || [],
                                    function(durationIdx, subCalendarDuration) {
                                        var startDateFieldLabel = availableDateFields[subCalendarDuration.startDateFieldName || ""],
                                            endDateFieldLabel = availableDateFields[subCalendarDuration.endDateFieldName || ""];

                                        if (startDateFieldLabel && endDateFieldLabel) {
                                            // Both fields are valid
                                            var durationFieldsListContainer = $(".field-group-jira-show .duration .date-fields", fieldsDiv),
                                                durationFieldKey = subCalendarDuration.startDateFieldName + "/" + subCalendarDuration.endDateFieldName,
                                                durationFieldId = "duration-" + durationFieldKey;

                                            $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDateDuration({
                                                durationFieldId: durationFieldId,
                                                isChecked: true,
                                                selectedDurationFieldKey: durationFieldKey,
                                                selectedStartDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.start', startDateFieldLabel),
                                                selectedEndDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.end', endDateFieldLabel),
                                            })).appendTo(durationFieldsListContainer);

                                            durationFieldsListContainer.parent().addClass("has-selected-duration");
                                        }
                                    }
                                );

                                if (hideDateFields) {
                                    setDateFieldsHidden(hideDateFields);
                                }
                            });
                        });

                        timeZoneIdDisplay.text(subCalendar.timeZoneId);
                    }

                    //Set read / read only
                    const textInputs = $("input[name='name'], input[name='description'], input[name='spaceKeyAutocomplete'], textarea[name='jql']", fieldsDiv);
                    const searchFilterInput = $("input[name='searchFilter']", fieldsDiv);
                    if (callbackHandler.isCalendarInEditMode()) {
                        textInputs.prop("readonly", false);
                        serverSelect.prop("disabled", false);
                        projectSelect.prop("disabled", false);
                        JiraEventFieldProcessCommon.enableSearchFilterSelect(searchFilterInput);
                    } else {
                        formMode.removeClass().addClass("form-mode read-only");
                        textInputs.prop("readonly", true);
                        serverSelect.prop("disabled", true);
                        projectSelect.prop("disabled", true);
                        JiraEventFieldProcessCommon.disableSearchFilterSelect(searchFilterInput);
                    }
                };

                var setupFields = function() {
                    $("select[name='tc-jira-server']", fieldsDiv).change(function() {
                        var selectedJiraServer = $("select[name='tc-jira-server'] option:selected", fieldsDiv);
                        if (selectedJiraServer.length && selectedJiraServer.val()) {
                            populateQueryOptions(null, function() {
                                populateFieldsList();
                            });
                        }
                    });

                    $("input[name='spaceKeyAutocomplete']", fieldsDiv).bind("selected.autocomplete-content", function(e, selection) {
                        $("input[name='spaceKey']", fieldsDiv).val(selection.content.key);
                        $(this).val(selection.content.name);
                        $(".spaceKeyAutocomplete-error", fieldsDiv).empty();
                    });

                    $(".jira-query-options .query-group input.radio", fieldsDiv)
                        .off("change")
                        .on("change", function() {
                            var value = $(this).val(),
                                jiraQueryOptionsContainer = $(".jira-query-options", fieldsDiv)
                                    .removeClass("option-project-selected option-search-filter-selected option-jql-selected")
                                    .addClass(value + "-selected"),
                                option = $("." + value, jiraQueryOptionsContainer);

                            $("input", option).focus(function() {
                                var input = $(this);
                                if (input.hasClass("with-hint")) {
                                    input.removeClass("with-hint").val("");
                                }
                            });

                            if ("option-jql" === value) {
                                $("textarea", option).trigger("keydown", [ true ]);
                            }

                            populateFieldsList(function() {
                                setDateFieldsHidden(option.data("hideDateFields") || !$(".option-value", option).val());

                                var durationContainer = $(".duration", fieldsDiv);
                                if ($(".date-fields", durationContainer).children().length) {
                                    durationContainer.addClass("has-selected-duration");
                                } else {
                                    durationContainer.removeClass("has-selected-duration");
                                }
                            });
                        });

                    var searchFilterOption = $(".option-search-filter", fieldsDiv);
                    var searchFilterAutoCompleteContainer = $(".search-filter-suggestions", searchFilterOption);

                    $("select[name='project']", fieldsDiv).change(function() {
                        populateFieldsList();
                    });

                    $(".add-duration", fieldsDiv).unbind().click(function() {
                        AJS.log("Open Jira Duration Dialog");
                        var addDurationDialog = new AJS.Dialog({
                            "height": 423,
                            "width" : 520,
                            "id" : "add-jira-duration-dialog"
                        });

                        addDurationDialog.addHeader(AJS.I18n.getText("calendar3.jira.duration.title"));
                        addDurationDialog.addPanel(
                            "",
                            Confluence.TeamCalendars.Templates.editJiraSubCalendarDurations(),
                            "calendar-dialog-panel"
                        );

                        var addDurationDialogPanel = addDurationDialog.getCurrentPanel().body,
                            durationFieldsListContainer = $(".field-group-jira-show .duration .date-fields", fieldsDiv),
                            dateFieldsListContainer = $(".field-group-jira-show .single-dates .date-fields", fieldsDiv);

                        addDurationDialog.addSubmit(AJS.I18n.getText("add.word"), function() {
                            var selectedStartDateFieldOption = $(".duration-start select option:selected", addDurationDialogPanel),
                                selectedEndDateFieldOption = $(".duration-end select option:selected", addDurationDialogPanel),
                                selectedDurationFieldKey = selectedStartDateFieldOption.val() + "/" + selectedEndDateFieldOption.val(),
                                selectedDurationField = $("div.checkbox", durationFieldsListContainer).filter(function() {
                                    return $("input", this).val() === selectedDurationFieldKey;
                                });

                            if (selectedDurationField.length) {
                                // Retick, just in case the user ticked it off when opening up the add durations dialog.
                                $("input", selectedDurationField).prop("checked", true);
                            } else {
                                var durationFieldId = "duration-" + selectedDurationFieldKey;

                                $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDateDuration({
                                    durationFieldId: durationFieldId,
                                    isChecked: true,
                                    selectedDurationFieldKey: selectedDurationFieldKey,
                                    selectedStartDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.start', selectedStartDateFieldOption.html()),
                                    selectedEndDateFieldName: AJS.I18n.getText('calendar3.jira.fields.duration.end', selectedEndDateFieldOption.html())
                                })).prependTo(durationFieldsListContainer);
                                durationFieldsListContainer.parent().addClass("has-selected-duration");
                            }

                            addDurationDialog.remove();
                            eventEditDialog.show();
                        });

                        addDurationDialog.addCancel(AJS.I18n.getText("cancel.name"), function() {
                            addDurationDialog.remove();
                            eventEditDialog.show();
                        });

                        var mappableDateFields = $.map(
                                $("div.checkbox", dateFieldsListContainer).filter(function() {
                                    return $(this).data("mappableAsDuration");
                                }),
                                function(aField) {
                                    return $("<option/>", {
                                        "value" : $("input", aField).val(),
                                        "text" : $("label", aField).text()
                                    });
                                }
                            ),
                            startDateFieldSelect = $(".duration-start select", addDurationDialogPanel),
                            endDateFieldSelect = $(".duration-end select", addDurationDialogPanel),

                            setSubmitButtonDisabledState = function(startField, endField) {
                                var okButton = addDurationDialog.popup.element.find(".button-panel-submit-button");

                                if (endField && startField && endField !== startField) {
                                    okButton.prop("disabled", false).removeClass("ui-state-disabled");
                                } else {
                                    okButton.prop("disabled", true).addClass("ui-state-disabled");
                                }
                            };

                        // Populate each date field dropdowns for selection
                        $.each(mappableDateFields, function(mappableDateFieldIdx, mappableDateField) {
                            $(mappableDateField).appendTo(startDateFieldSelect).clone().appendTo(endDateFieldSelect);
                        });

                        if (!mappableDateFields.length) {
                            $(".duration-fields", addDurationDialogPanel).addClass("hidden");
                            $(".add-fields", $(".no-selectable-date-fields", addDurationDialogPanel).removeClass("hidden")).html(
                                AJS.I18n.getText(
                                    "calendar3.jira.duration.fields.addfields",
                                    $(".field-group-jira-server select option:checked", fieldsDiv).data("display-url")
                                )
                            );
                        }

                        var adjustSelectableDateFieldOptions = function(selectedDateField, otherDateFieldSelect) {
                            var initialSelectedEndDateField = $("option:selected", otherDateFieldSelect).val();

                            // Remove all date fields.
                            $("option", otherDateFieldSelect).filter(function() {
                                return $(this).val();
                            }).remove();

                            $.each(mappableDateFields, function(mappableDateFieldIdx, mappableDateField) {
                                if (mappableDateField.val() !== selectedDateField) {
                                    mappableDateField.clone().appendTo(otherDateFieldSelect);
                                }
                            });

                            $("option", otherDateFieldSelect).prop("selected", false).filter(
                                function() {
                                    return $(this).val() === initialSelectedEndDateField;
                                }
                            ).prop("selected", true);
                        };

                        startDateFieldSelect.change(function() {
                            var selectedStartDateField = $("option:selected", this).val();

                            adjustSelectableDateFieldOptions(selectedStartDateField, endDateFieldSelect);
                            setSubmitButtonDisabledState(selectedStartDateField, $("option:selected", endDateFieldSelect).val());
                        });

                        endDateFieldSelect.change(function() {
                            var selectedEndDateField = $("option:selected", this).val();

                            adjustSelectableDateFieldOptions(selectedEndDateField, startDateFieldSelect);
                            setSubmitButtonDisabledState(selectedEndDateField, $("option:selected", startDateFieldSelect).val());
                        });

                        eventEditDialog.hide();
                        setSubmitButtonDisabledState();
                        addDurationDialog.show();
                        return false;
                    });

                    $(".subcalendar-cancel", fieldsDiv).click(function() {
                        eventEditDialog.remove();
                        return false;
                    });

                    return fieldsDiv;
                };

                var populateQueryOptions = function(onError, onSuccess) {
                    var jiraQueryOptionsContainer = $(".jira-query-options", fieldsDiv);
                    var queryRadioButtons = $(".query-group input.radio", jiraQueryOptionsContainer).prop("disabled", true).prop("checked", false);
                    var jqlNotSupportedControls = $(".field-group .project-select-label", jiraQueryOptionsContainer);
                    var jqlSupportedControls = $(".query-group", jiraQueryOptionsContainer);
                    var controlsToToggleVisibleOnOAuthError = $(".compatibility-group, .field-group-jira-show", jiraQueryOptionsContainer);
                    var formMode = $(".form-mode", fieldsDiv).addClass("jira-query-options-loading");
                    var server = JiraEventFieldProcess.getServerSelect(fieldsDiv).val();
                    var handleGetOptionsError = function(XMLHttpRequest, textStatus, errorThrown) {
                        formMode.removeClass("jira-query-options-loading");
                        jqlSupportedControls.addClass("hidden");
                        jqlNotSupportedControls.removeClass("hidden");
                        controlsToToggleVisibleOnOAuthError.addClass("hidden");

                        if (Confluence.TeamCalendars.isRequireOauth(XMLHttpRequest)) {
                            CalUtil.fireEventForAnalytics("oauth.client.fetch.failed");
                            JiraEventFieldProcess.showOauthRequitedMessage(XMLHttpRequest, fieldsDiv, eventEditDialog, controlsToToggleVisibleOnOAuthError, populateForm);
                        } else {
                            if (onError && $.isFunction(onError)) {
                                onError(XMLHttpRequest, textStatus, errorThrown);
                            }
                        }
                    };
                    var handleGetOptionsSuccess = function(queryOptions) {
                        controlsToToggleVisibleOnOAuthError.removeClass("hidden");
                        formMode.removeClass("jira-query-options-loading");
                        queryRadioButtons.prop("disabled", false);
                        if (!queryOptions.searchRequests) {
                            $(".radio-search-filer input", fieldsDiv).prop("disabled", true);
                        }

                        // Selects the project list again - so we're resetting the previous query mode and defaulting it back to "project".
                        if (!jiraQueryOptionsContainer.hasClass("option-project-selected")) {
                            jiraQueryOptionsContainer.removeClass("option-search-filter-selected option-jql-selected").addClass("option-project-selected");
                        }

                        $(".radio-project input", jiraQueryOptionsContainer).prop("checked", true);
                        if (queryOptions.jqlNotSupported) {
                            jqlSupportedControls.addClass("hidden");
                            jqlNotSupportedControls.removeClass("hidden");
                        } else {
                            jqlNotSupportedControls.addClass("hidden");
                            jqlSupportedControls.removeClass("hidden");

                            var currentTimeMs = new Date().getTime();
                            var newJqlInputid = "jql-" + currentTimeMs;

                            $("textarea[name='jql']", jiraQueryOptionsContainer).before(
                                    $(document.createElement("textarea")).attr({
                                        autocomplete: "off",
                                        cols:         "20",
                                        id:           newJqlInputid,
                                        name:         "jql",
                                        rows:         1,
                                        wrap:         "off",
                                        placeholder:  AJS.I18n.getText("calendar3.jira.fields.jql.hint")
                                    }).addClass("textarea with-hint"
                                            ).focus(function () {
                                                var thisTextArea = $(this);
                                                if (thisTextArea.hasClass("with-hint")) {
                                                    thisTextArea.removeClass("with-hint").val("");
                                                }
                                            }).val("")
                            ).remove();

                            var jqlAutoCompleteUrlRegex = /.+\/jql\/autocompletedata\/suggestions.json\?fieldName=[^&]+&fieldValue=.*/;
                            var jqlAutoCompleteSpinner = $(".jqlAutoCompleteSpinner", jiraQueryOptionsContainer);

                            $(document).ajaxSend(function (event, jqXHR, ajaxOptions) {
                                if (jqlAutoCompleteUrlRegex.test(ajaxOptions.url || "")) {
                                    jqlAutoCompleteSpinner.removeClass("hidden");
                                }

                            }).ajaxComplete(function (event, jqXHR, ajaxOptions) {
                                        if (jqlAutoCompleteUrlRegex.test(ajaxOptions.url || "")) {
                                            jqlAutoCompleteSpinner.addClass("hidden");
                                        }
                                    });

                            $(".option-jql label", jiraQueryOptionsContainer).attr("for", newJqlInputid);

                            var newJqlErrorContainerId = "jql-error-" + currentTimeMs;
                            $(".jql-error", jiraQueryOptionsContainer).before(
                                    $(document.createElement("div")).addClass("error jql-error").attr("id", newJqlErrorContainerId)
                            ).remove();

                            $(".dropdown-ready", jiraQueryOptionsContainer).remove();

                            JIRA.JQLAutoComplete({
                                fieldID:            newJqlInputid,
                                parser:             JIRA.JQLAutoComplete.MyParser(JSON.parse(queryOptions.jqlReservedKeywordsJson || "[]")),
                                queryDelay:         .65,
                                jqlFieldNames:      JSON.parse(queryOptions.visibleFieldNamesJson || "[]"),
                                jqlFunctionNames:   JSON.parse(queryOptions.visibleFunctionNamesJson || "[]"),
                                minQueryLength:     0,
                                allowArrowCarousel: true,
                                autoSelectFirst:    false,
                                errorID:            newJqlErrorContainerId,
                                jiraApplicationId:  JiraEventFieldProcess.getServerSelect().val(),
                                baseUrl:            Confluence.getBaseUrl()
                            });

                            $("textarea[name='jql']", fieldsDiv).keydown(function (jsEvent, validateRegardless) {
                                var jqlInput = $(this);
                                var initialJql = jqlInput.val();

                                if (validateRegardless || jsEvent.keyCode !== 27) {
                                    setTimeout(function () {
                                        var jql = jqlInput.val();
                                        var jqlErrorContainer = $(".jql-error", fieldsDiv);

                                        if (jql && !jqlInput.hasClass("with-hint") && (validateRegardless || jql !== initialJql)) {
                                            $.ajax({
                                                cache:   false,
                                                data:    {
                                                    jql: jql
                                                },
                                                error:   function () {
                                                    jqlErrorContainer.empty().text(AJS.I18n.getText("calendar3.jira.error.jql.cannotvalidateagainstselectedjira"));
                                                },
                                                success: function (jqlValidationResult) {
                                                    var jqlOk = $(".jql-ok", fieldsDiv),
                                                            jqlBad = $(".jql-bad", fieldsDiv),
                                                            durationContainer = $(".duration", fieldsDiv);

                                                    if (jqlValidationResult.valid) {
                                                        jqlOk.removeClass("hidden");
                                                        jqlBad.addClass("hidden");

                                                        populateFieldsList(function () {
                                                            durationContainer.addClass("can-add-duration");
                                                            if ($(".date-fields div.checkbox", durationContainer).length) {
                                                                durationContainer.addClass("has-selected-duration");
                                                            } else {
                                                                durationContainer.removeClass("has-selected-duration");
                                                            }
                                                        });
                                                        jqlInput.focus();
                                                    } else {
                                                        jqlOk.addClass("hidden");
                                                        jqlBad.removeClass("hidden");

                                                        $(".date-fields", fieldsDiv).empty();
                                                        setDateFieldsHidden(true);
                                                        durationContainer.removeClass("can-add-duration");
                                                    }
                                                },
                                                type:    "PUT",
                                                url:     Confluence.getBaseUrl().concat(
                                                                "/rest/calendar-services/1.0/calendar/jira/",
                                                                encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv).val()),
                                                                "/jql/validate.json"
                                                            ),
                                                timeout: Confluence.TeamCalendars.ajaxTimeout
                                            });
                                        }
                                    }, 300);

                                } else {
                                    var suggestionsContainer = $(".option-jql .suggestions", jiraQueryOptionsContainer);
                                    if (suggestionsContainer.is(":visible")) {
                                        suggestionsContainer.hide();
                                        return false;
                                    }
                                }
                            });
                        }

                        const searchFiltersData = queryOptions.searchRequests;
                        const searchFilterOption = $(".option-search-filter", fieldsDiv);
                        const searchFilterErrors = $('.option-search-filter .error', fieldsDiv);
                        const searchFilterInput = $("input[name='searchFilter']", searchFilterOption);

                        JiraEventFieldProcessCommon.setupSearchFilterSelect(searchFilterInput, searchFiltersData, () => {
                            searchFilterErrors.empty();
                            populateFieldsList();
                        });

                        if (searchFiltersData && $.isArray(searchFiltersData) && searchFiltersData.length) {
                            $(".option-content", searchFilterOption.removeData("hideDateFields")).removeClass("disabled");
                        } else {
                            JiraEventFieldProcessCommon.disableSearchFilterSelect(searchFilterInput);
                            $(".option-content", searchFilterOption.data("hideDateFields", true)).addClass("disabled");
                        }

                        var projects = queryOptions.projects;
                        var projectOption = $(".option-project", fieldsDiv);
                        if (projects && $.isArray(projects) && projects.length) {
                            var projectList = $("select", projectOption).empty();

                            //need add select project first line for improvement performance jira project
                            $(document.createElement("option")).val("").text(AJS.I18n.getText("calendar3.project.select")).appendTo(projectList);

                            $.each(projects, function (projectIdx, project) {
                                $(document.createElement("option")).val(project.key).text(project.name).appendTo(projectList);
                            });

                            $(".option-content", projectOption.removeData("hideDateFields")).removeClass("disabled");
                        } else {
                            $(".option-content", projectOption.data("hideDateFields", true)).addClass("disabled");
                        }

                        if (!queryOptions.dateRangeSupported) {
                            $(".duration", fieldsDiv).css("visibility", "hidden");
                        } else {
                            $(".duration", fieldsDiv).css("visibility", "visible");
                        }

                        eventEditDialog.enableOkButton();

                        if (onSuccess && $.isFunction(onSuccess)) {
                            onSuccess(queryOptions);
                        }
                    };
                    JiraEventFieldProcess.getJiraOptions(handleGetOptionsError, handleGetOptionsSuccess, server);
                };

                var populateFieldsList = function(successCallback) {
                        var selectedDateFields = $(".field-group-jira-show div.checkbox", fieldsDiv).filter(function() {
                            return $("input", this).is(":checked");
                        }),
                        selectedQuery = $(".jira-query-options .matrix .radio:checked", fieldsDiv).val(),
                        durationContainer = $(".duration", fieldsDiv),
                        dateFieldsData = {},
                        dateFieldsUrl = "";

                    // Figure out how to query for date fields
                    if ("option-project" === selectedQuery) {
                        var projectKey = encodeURIComponent($(".jira-query-options .option-project select option:selected", fieldsDiv).val());
                        if(projectKey) {
                            dateFieldsUrl = "/project/" + projectKey;
                        }
                    } else if ("option-search-filter" === selectedQuery) {
                        var searchFilterId = $(".jira-query-options .option-search-filter input[name='searchFilter']", fieldsDiv).val();
                        if (searchFilterId) {
                            dateFieldsUrl = "/filter/" + encodeURIComponent(searchFilterId);
                        }
                    } else if ("option-jql" === selectedQuery) {
                        dateFieldsData.query = $(".jira-query-options .option-jql textarea", fieldsDiv).val() || "";
                        if (dateFieldsData.query) {
                            dateFieldsUrl = "/jql";
                        }
                    }

                    durationContainer.removeClass("can-add-duration");

                    if (dateFieldsUrl) {
                        var dateFieldsSpinnerContainer = $(".date-fields-spinner-container", fieldsDiv).removeClass("hidden"),
                            singleDateFieldsListContainer = $(".single-dates .date-fields", fieldsDiv).empty(),
                            durationDateFieldsListContainer = $(".date-fields", durationContainer).empty();

                        $.ajax({
                            cache: false,
                            data : dateFieldsData,
                            dataType: "json",
                            error: function() {
                                dateFieldsSpinnerContainer.addClass("hidden");
                                $(".dateFieldName-error", fieldsDiv).text(AJS.I18n.getText("calendar3.jira.error.datefields.unabletoshow"));
                            },
                            success: function(dateFields) {
                                dateFieldsSpinnerContainer.addClass("hidden");

                                var singleDateFieldsList = $("<div/>"),
                                    selectedDateFieldKeys = $.map(selectedDateFields, function(selectedDateField) {
                                        return $("input", selectedDateField).val();
                                    }),
                                    selectableDateFieldKeys = $.map(dateFields, function(dateField) {
                                        return dateField.key;
                                    });

                                // Repopulate single date fields
                                $.each(dateFields, appendDateFieldOption);

                                function appendDateFieldOption(dateFieldIdx, dateField) {
                                    var isChecked = $.inArray(dateField.key, selectedDateFieldKeys) !== -1;
                                    var warning;
                                    if (!dateField.isSearchable) {
                                        warning = AJS.I18n.getText("calendar3.jira.warning.fields.fieldnotsearchable");
                                    }
                                    $(Confluence.TeamCalendars.Templates.editJiraSubCalendarDatefield({dateField: dateField, isChecked: isChecked, warning: warning}))
                                        .data("mappableAsDuration",
                                            dateField.customfield ||
                                            dateField.key === "duedate" ||
                                            dateField.key === "created" ||
                                            dateField.key === "resolution")
                                        .appendTo(singleDateFieldsList);
                                }

                                // Repopulate durations
                                selectedDateFields.filter(function() {
                                    var durationFields = $("input", this).val();
                                    if (durationFields.indexOf("/") !== -1) {
                                        var durationFieldTokens = durationFields.split("/", 2);
                                        return durationFieldTokens.length === 2 &&
                                            $.inArray(durationFieldTokens[0], selectableDateFieldKeys) !== -1 &&
                                            $.inArray(durationFieldTokens[1], selectableDateFieldKeys) !== -1;
                                    }

                                    return false;
                                }).appendTo(
                                        // Empty the container again because it may have been populated by another instance of this AJAX request
                                        // after the container was initially emptied (just before the start of this AJAX request).
                                        durationDateFieldsListContainer.empty()
                                );

                                singleDateFieldsList.appendTo(
                                    // See comment regarding emptying durationDateFieldsListContainer
                                    singleDateFieldsListContainer.empty()
                                );

                                if (singleDateFieldsList.children().length) {
                                    setDateFieldsHidden(false);
                                    durationContainer.addClass("can-add-duration");
                                } else {
                                    durationContainer.removeClass("can-add-duration");
                                }

                                if (successCallback && $.isFunction(successCallback)) {
                                    successCallback(dateFields);
                                }
                            },
                            timeout : Confluence.TeamCalendars.ajaxTimeout,
                            url : Confluence.getBaseUrl().concat(
                                    "/rest/calendar-services/1.0/calendar/jira/",
                                    encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv).val()),
                                    "/date-fields",
                                    dateFieldsUrl,
                                    ".json")
                        });
                    } else {
                        setDateFieldsHidden(true);
                        if (successCallback && $.isFunction(successCallback)) {
                            successCallback();
                        }
                    }
                };

                setupFields();
                populateForm();
            },

            CustomSubmitter : {
                JiraIssues : {
                    submitForm : function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        function getFormDataAsAjaxData() {
                            var dateFieldQueryParam = "",
                                selectedDateFields = $(".field-group-jira-show .date-fields div.checkbox").filter(function() {
                                    return $("input:checked", this).length;
                                });

                            //Build date field query
                            $.each(selectedDateFields, function() {
                                var dateFieldKey = $("input", this).val();

                                //handle durations
                                if (dateFieldKey.indexOf("/") !== -1) {
                                    var durationFieldTokens = dateFieldKey.split("/", 2);
                                    if (dateFieldQueryParam) {
                                        dateFieldQueryParam += "&";
                                    }

                                    dateFieldQueryParam += "duration=" + encodeURIComponent(durationFieldTokens[0] + "/" + durationFieldTokens[1]);
                                } else {  //Single dates
                                    if (dateFieldQueryParam) {
                                        dateFieldQueryParam += "&";
                                    }

                                    dateFieldQueryParam += "dateFieldName=" + encodeURIComponent(dateFieldKey);
                                }
                            });

                            var data = {
                                    type : "jira",
                                    parentId: parentSubCalendarId,
                                    subCalendarId : $("input[name='subCalendarId']", fieldsDiv).val(),
                                    name : name,
                                    description : "",
                                    color : $("input[name='color']", fieldsDiv).val()
                                },
                                location = "jira://" + encodeURIComponent(JiraEventFieldProcess.getServerSelect(fieldsDiv).val() || "") + "?",
                                selectedQueryOption = $(".jira-query-options .matrix input.radio:checked", fieldsDiv).val();

                            if ("option-project" === selectedQueryOption) {
                                location += "projectKey=" + encodeURIComponent($("select[name='project'] option:selected", fieldsDiv).val());
                            } else if ("option-search-filter" === selectedQueryOption) {
                                location += "searchFilterId=" + $("input[name='searchFilter']", fieldsDiv).val();
                            } else {
                                location += "jql=" + encodeURIComponent($("textarea[name='jql']", fieldsDiv).val());
                            }

                            if (dateFieldQueryParam) {
                                location += "&" + dateFieldQueryParam;
                            }

                            data.location = location;
                            return data;
                        }

                        sendJiraCalendar(getFormDataAsAjaxData(), fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                    }
                },

                JiraAgileSprints : {
                    submitForm : function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        var formData = getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, "sprint", "jira-agile-sprint");
                        sendJiraCalendar(formData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                    }
                },

                JiraProjectReleases : {
                    submitForm : function(fieldsDiv, name, parentSubCalendarId, callbackHandler, onSuccessCallback, onErrorCallback) {
                        var formData = getSimpleJiraFormData(fieldsDiv, name, parentSubCalendarId, "versiondue", "jira-project-releases");
                        sendJiraCalendar(formData, fieldsDiv, callbackHandler, onErrorCallback, onSuccessCallback);
                    }
                }
            }
        };

        return JiraEventFieldProcess;
    }
);
