define("tc/repeat-model",
    [
        "jquery",
        "tc/rrule",
        "tc-backbone",
        "tc/calendar-util"
    ],
    function (
        $,
        TCRRule,
        Backbone,
        CalUtil
    ) {
        "use strict";

        /**
         * Backbone model which represent the whole RRULE
         */
        var repeatModel = Backbone.Model.extend({
            defaults: {
                isUnsupportedRrule: false,
                isDisable: false,
                freq: null,
                dtstart: null,
                interval: null,
                wkst: null,
                count: null,
                until: null,
                untilDateOnly: null,
                bysetpos: null,
                bymonth: null,
                bymonthday: null,
                byyearday: null,
                byweekno: null,
                byweekday: null,
                bynweekday: null,
                byhour: null,
                byminute: null,
                bysecond: null,
                byeaster: null
            },
            /**
             * Initialize model from RRRULE string (Ideally that value is from server)
             */
            initialize: function(params) {
                this.rruleStr = params.rruleStr ? params.rruleStr : "";
                // an current event object, we could find start date and end date here
                this.event = params.event;
                this._rrule = TCRRule.fromString(this.rruleStr);

                this.set(
                    {
                        isDisable: params.isDisable,
                        isUnsupportedRrule: false
                    });
                // set all rrule attribute
                this.set(this._rrule.origOptions);
            },
            /**
             * Get back iCalendar RRULE from this model
             */
            getRRuleAsString: function () {
                // all view need to update model according to it current state
                this.trigger("rrule-string", {model: this});
                if (this.attributes.freq === null || this.attributes.freq === -1) {
                    return "";
                }

                if (this.attributes.isDisable) {
                    // if repeat-model is disable then we won't generate new rule just return old one
                    return this.rruleStr;
                }

                // CONFSERVER-61041: Explicitly exclude UNTIL from RRULE. We manage UNTIL on the back-end to ensure correct time.
                var newRule = new TCRRule({
                    freq: this.attributes.freq,
                    dtstart: this.attributes.dtstart,
                    interval: this.attributes.interval,
                    wkst: this.attributes.wkst,
                    count: this.attributes.count,
                    bysetpos: this.attributes.bysetpos,
                    bymonth: this.attributes.bymonth,
                    bymonthday: this.attributes.bymonthday,
                    byyearday: this.attributes.byyearday,
                    byweekno: this.attributes.byweekno,
                    byweekday: this.attributes.byweekday,
                    bynweekday: this.attributes.bynweekday,
                    byhour: this.attributes.byhour,
                    byminute: this.attributes.byminute,
                    bysecond: this.attributes.bysecond,
                    byeaster: this.attributes.byeaster
                });
                // the newer version of rrule lib added RRule: prefix to their toString method, which is why we need to trim it.
                var rruleStr = newRule.toString().replace("RRULE:", "").trim();
                console.log("Computed RRULE: " + rruleStr);
                return rruleStr;
            },
            trigger: function() {
                console.debug('Event', arguments);
                Backbone.Model.prototype.trigger.apply(this, arguments);
            }
        });

        return repeatModel;
    });
