define("tc/repeat-on",
    [
        "jquery",
        "tc/rrule",
        "tc-backbone",
        "tc/calendar-util",
        "underscore",
        "tc/repeat-base-child-component"
    ],
    function (
        $,
        TCRRule,
        Backbone,
        CalUtil,
        _,
        BaseChildComponentView
    ) {
        "use strict";

        var repeatOnView = BaseChildComponentView.extend({
            _WEEKDAY: [
                TCRRule.MO, TCRRule.TU, TCRRule.WE, TCRRule.TH, TCRRule.FR, TCRRule.SA, TCRRule.SU
            ],
            _templates: {
                "WEEKLY" :Confluence.TeamCalendars.Templates.Fields.repeatOnWeekly,
                "MONTHLY": Confluence.TeamCalendars.Templates.Fields.repeatOnMonthly
            },
            _notShowInFrequency: ["", "DAILY", "YEARLY"],
            _monthlyFreq: "MONTHLY",
            _weeklyFreq: "WEEKLY",
            _supportingFrequencies: ["MONTHLY", "WEEKLY"],
            options: null,
            events: {
                "change .repeat-on-container input[type='checkbox']": "updateModel"
            },
            /**
             * Initialize repeatOnView
             */
            initialize: function (options) {
                this.setJQueryEl(options.$el);

                this.options = options;
                this.initializeTemplate(this.options);

                if (!this.validateInput()){
                    this.model.set({isUnsupportedRrule: true});
                    return;
                }

                this.model.on('change:isDisable', this.render, this);
                this.model.on('change:frequencySelection', this.initializeTemplateAndRender, this);
                this.model.on('rrule-string', this.updateModel, this);
            },
            initializeTemplateAndRender: function() {
                this.initializeTemplate(this.options);
                this.render();
            },
            initializeTemplate: function(options) {
                var modelAttributes = this.model.attributes;
                if (this.hideIfNotMatch &&
                    this.hideIfNotMatch(this._notShowInFrequency, modelAttributes.frequencySelection, this.$el)) {
                    return;
                }

                // change template base on frequency
                var template = this._templates[modelAttributes.frequencySelection];

                this.$el.html($(template()).html());

                var component = this;
                var componentElement = this.$el;

                // because we replace el so it is safe to add UI event listener
                if (this._weeklyFreq === modelAttributes.frequencySelection) {

                    // event listening on UI component
                    componentElement.on("change", "input[type='checkbox']", function() {
                        var element = $(this);
                        element.bind(component.updateModel(), this);
                    });

                    // try to select default value for weekday
                    if (!modelAttributes.byweekday) {
                        modelAttributes.byweekday = [this.model.event ? this._dateToWeekday(this.model.event.start) : TCRRule.MO];

                        console.debug("The default value of weekday is ===========> " + modelAttributes.byweekday);
                    }
                }

                if (this._monthlyFreq === modelAttributes.frequencySelection){
                    if ($.fn.auiSelect2) {
                        componentElement.find("select.select").each(function (index, dropdownElement) {
                            $(dropdownElement).auiSelect2({
                                minimumResultsForSearch: -1,
                            });
                        })
                    }

                    componentElement.on("change", "select.select", function() {
                        var element = $(this);
                        element.bind(component.updateModel(), this);
                    });

                    componentElement.on("change", "input.radio", function() {
                        var element = $(this);
                        element.bind(component.updateModel(), this);
                    });

                    // if we don't have bymonthday then just set default value for byweekday as today
                    if (!this.model.rruleStr || this.model.rruleStr === "") {
                        // only set default value if we don't have rrule string input
                        if (!modelAttributes.bymonthday && this.model.event) {
                            if (!modelAttributes.byweekday) {
                                modelAttributes.byweekday = [this._dateToWeekday(this.model.event.start)];
                                console.debug("The default value of weekday is ===========> " + modelAttributes.byweekday);
                            }
                        }
                    }
                }
            },
            validateInput: function () {
                var modelAttributes = this.model.attributes;
                if (this._supportingFrequencies.indexOf(modelAttributes.frequencySelection) === -1 && modelAttributes.byweekday) {
                    // it makes no sense if we have byweekday in unsupported frequencySelection
                    console.debug("Validate failed on repeat on because of invalid frequencySelection");
                    return false;
                }

                if (modelAttributes.byweekday && modelAttributes.bymonthday){
                    console.debug("Validate failed on repeat on because of 'byweekday' and 'bymonthday could not exist at the same time'");
                    return false;
                }

                if (this._monthlyFreq === modelAttributes.frequencySelection) {
                    var modelAttributes = this.model.attributes;
                    if (!modelAttributes.bymonthday && modelAttributes.byweekday){
                        var pos = modelAttributes.byweekday[0].n;
                        var supportedPos = [1, 2, 3, 4, -1];
                        if (supportedPos.indexOf(pos) === -1) {
                            console.debug("Validate failed on repeat on because of invalid pos in 'byweekday' on monthly: " + pos);
                            return false;
                        }
                    }
                }

                if (this._weeklyFreq === modelAttributes.frequencySelection) {
                    if (!modelAttributes.byweekday) {
                        return false;
                    }

                    var validationResult = true;
                    $.each(modelAttributes.byweekday, function(index, item) {
                        var pos = item.n;
                        if (pos > 1) {
                            console.debug("Validate failed on repeat on because of invalid pos " + pos + " in 'byweekday' on weekly: " + item );
                            validationResult = false;
                            return;
                        }
                    });

                    if (!validationResult) return validationResult;
                }

                return true;
            },
            /**
             *  For updating model
             */
            updateModel: function() {
                var modelAttributes = this.model.attributes;
                if (this.hideIfNotMatch &&
                    this.hideIfNotMatch(this._notShowInFrequency, modelAttributes.frequencySelection, this.$el)) {
                    // if not visible then those attributes should be empty
                    this.model.set({byweekday: null, bymonthday: null});
                    return;
                }

                if (modelAttributes.isDisable){
                    console.debug("Won't update model for repeat on component because model is disable state");
                    return;
                }

                if ("WEEKLY" === modelAttributes.frequencySelection) {
                    this._updateModelOnRepeatOnWeekly();
                    return;
                }

                if ("MONTHLY" === modelAttributes.frequencySelection) {
                    this._updateModelOnRepeatOnMonthly();
                }
                console.debug("Update model for repeat on component" + JSON.stringify({
                    innerHtml : this.$el.html(),
                    modelAttrs: modelAttributes
                }));
                this.render();
            },
            _dateToWeekday: function(date) {
                if (!date) {
                    return  TCRRule.MO;
                }

                switch (date.getDay()) {
                    case 1:
                        return TCRRule.MO;
                    case 2:
                        return TCRRule.TU;
                    case 3:
                        return TCRRule.WE;
                    case 4:
                        return TCRRule.TH;
                    case 5:
                        return TCRRule.FR;
                    case 6:
                        return TCRRule.SA;
                    case 0:
                        return TCRRule.SU;
                    default:
                        return TCRRule.MO;
                }
            },
            /**
             * This will update 'byweekday' property only
             *
             * @private
             */
            _updateModelOnRepeatOnWeekly: function() {
                var byweekday = [];
                var component = this;
                var componentElement = this.$el;
                componentElement.find("input[type='checkbox']").each(function(index, element) {
                    if (element.checked) {
                        byweekday.push(component._WEEKDAY[$(element).val()]);
                    }
                });
                this.model.set({byweekday: byweekday, bymonthday: null});
            },
            /**
             * This will update :
             *  - Monthly only : 'byweekday' property
             *  - By day : 'bymonthday'
             *
             * @private
             */
            _updateModelOnRepeatOnMonthly: function() {
                var bypos = null;
                var byweekday = null;
                if (this.$el.find("#monthlyRepeatOnSameDay").is(":checked")){
                    bypos = this.$el.find("#monthlyRepeatOnSameDayByMonthDay").val();
                    byweekday = this.$el.find("#monthlyRepeatOnSameDayByWeekDay").val();
                }

                if ("day" === byweekday) {
                    this.model.set({bymonthday: bypos, byweekday: null});
                }else{
                    this.model.set(
                        {
                            bymonthday: null,
                            byweekday: this._WEEKDAY[byweekday] ? this._WEEKDAY[byweekday].nth(bypos) : null
                        });
                }
            },
            /**
             * Rendering repeat on
             */
            render: function (){
                var modelAttributes = this.model.attributes;
                var componentElement = this.$el;
                if (this.hideIfNotMatch &&
                    this.hideIfNotMatch(this._notShowInFrequency, modelAttributes.frequencySelection, this.$el)){
                    return;
                }

                if ("WEEKLY" === modelAttributes.frequencySelection) {
                   this._renderRepeatOnWeekly(componentElement);
                }

                if ("MONTHLY" === modelAttributes.frequencySelection) {
                    this._renderRepeatOnMonthly(componentElement);
                }

                // Disable after fill up data
                if (modelAttributes.isDisable){
                    this._disable(componentElement);
                    console.debug("Disable repeat on because model is disable state");
                }

                console.debug("Render repeat on component: " + JSON.stringify({
                    innerHtml : this.$el.html(),
                    modelAttrs: modelAttributes
                }));
            },
            _disable: function(componentElement) {
                var modelAttributes = this.model.attributes;
                if ("WEEKLY" === modelAttributes.frequencySelection) {
                    componentElement.find("input[type='checkbox']").prop("disabled", true);
                }

                if ("MONTHLY" === modelAttributes.frequencySelection) {
                    componentElement.find("#monthlyRepeatOnSameDate").prop("disabled", true);
                    componentElement.find("#monthlyRepeatOnSameDay").prop("disabled", true);

                    if ($.fn.auiSelect2) {
                        componentElement.find("#monthlyRepeatOnSameDayByMonthDay").auiSelect2("disable");
                        componentElement.find("#monthlyRepeatOnSameDayByWeekDay").auiSelect2("disable");
                    } else {
                        componentElement.find("#monthlyRepeatOnSameDayByMonthDay").prop("disabled", true);
                        componentElement.find("#monthlyRepeatOnSameDayByWeekDay").prop("disabled", true);
                    }
                }
            },
            _renderRepeatOnMonthly: function(componentElement) {
                var modelAttributes = this.model.attributes;
                var container = componentElement.find("#monthlyRepeatOnSameDayContainer");

                container.removeClass("hidden");
                //Enable it first
                componentElement.find("#monthlyRepeatOnSameDate").prop("disabled", false);
                componentElement.find("#monthlyRepeatOnSameDay").prop("disabled", false);

                if ($.fn.auiSelect2) {
                    componentElement.find("#monthlyRepeatOnSameDayByMonthDay").auiSelect2("enable");
                    componentElement.find("#monthlyRepeatOnSameDayByWeekDay").auiSelect2("enable");
                } else {
                    componentElement.find("#monthlyRepeatOnSameDayByMonthDay").prop("disabled", false);
                    componentElement.find("#monthlyRepeatOnSameDayByWeekDay").prop("disabled", false);
                }

                // fill in data
                if (!modelAttributes.byweekday && !modelAttributes.bymonthday) {
                    componentElement.find("#monthlyRepeatOnSameDate").prop("checked", true);
                    componentElement.find("#monthlyRepeatOnSameDay").prop("checked", false);
                    container.addClass("hidden");
                } else {
                    componentElement.find("#monthlyRepeatOnSameDate").prop("checked", false);
                    componentElement.find("#monthlyRepeatOnSameDay").prop("checked", true);
                    // display setting options
                    container.removeClass("hidden");

                    var setValues = function (byMonthDay, byWeekday) {
                        if (!byMonthDay) {
                            //set default value for byMonthDay
                            byMonthDay = 1;
                        }
                        if ($.fn.auiSelect2) {
                            container.find("#monthlyRepeatOnSameDayByMonthDay").auiSelect2('val', byMonthDay);
                            container.find("#monthlyRepeatOnSameDayByWeekDay").auiSelect2('val', byWeekday);
                        } else {
                            container.find("#monthlyRepeatOnSameDayByMonthDay").val(byMonthDay);
                            container.find("#monthlyRepeatOnSameDayByWeekDay").val(byWeekday);
                        }
                    };

                    if (modelAttributes.byweekday && modelAttributes.bymonthday){
                        // TODO unsupported
                        return;
                    } else {
                        if (modelAttributes.bymonthday) {
                            var pos = modelAttributes.bymonthday;
                            setValues(pos, "day");
                        } else {
                            var byweekday =  modelAttributes.byweekday[0] || modelAttributes.byweekday;
                            var pos = byweekday.n;
                            setValues(pos, byweekday.weekday);
                        }
                    }
                }
            },
            _renderRepeatOnWeekly: function(componentElement) {
                var modelAttributes = this.model.attributes;

                //Enable it first
                componentElement.find("input[type='checkbox']").prop("disabled", false);

                if (modelAttributes.byweekday) {
                    modelAttributes.byweekday.forEach(function (weekday) {
                        var item = componentElement.find("input[value='" + weekday.weekday +  "']");
                        $(item).prop('checked', true);
                    });
                }
            }
        });

        return repeatOnView;
    });
