/**
 * Introduction tour for Team Calendar
 */
define("tc/myCalendarTour",
    [
        "jquery",
        "underscore",
        "tc/localStorage"
    ],
    function(
        $,
        _,
        LocalStorage) {
        var identifier = "team.calendar";
        var externalClickNamespace = identifier + ".tour";
        var hasBoundOnExternalClick = false;
        var isPausing = false;
        var isStoppingTour = true;
        var currentLegIndexKey = externalClickNamespace + ".currentLegIndex";
        var currentAttachedCalendarKey = externalClickNamespace + ".currentAttachedCalendar";
        var tourFinishKey = externalClickNamespace + ".done";
        var selectorStringTemplate = "div[data-sub-calendar-id='#calendarID']";

        var getSelectorString = function(context) {
            var calendarID = context.calendarID;
            if (!calendarID) {
                return null;
            }

            return selectorStringTemplate.replace("#calendarID", calendarID);
        };

        var menuHelper = {
            selectMenuItem: function(context, menuItemSelector) {
                var selectorString = getSelectorString(context);
                if (!selectorString) {
                    return;
                }

                var targetString = selectorString + menuItemSelector;
                $(targetString).addClass("aui-dropdown2-active");
            },
            unSelectMenuItem: function(context, menuItemSelector) {
                var selectorString = getSelectorString(context);
                if (!selectorString) {
                    return;
                }

                var targetString = selectorString + menuItemSelector;
                $(targetString).removeClass("aui-dropdown2-active");
            }
        };
        // actions could be apply before or after leg
        var actionList = {
            openMenu: function(context) {
                // open menu
                var selectorString = getSelectorString(context);

                if ($(selectorString + " .subcalendar-dropdown-menu.parent-subCalendar.aui-dropdown2-active").length > 0) {
                    // if menu is already open then we won;t do anything
                    return;
                }

                setTimeout(function() {
                    $(selectorString + " .subcalendar-dropdown-menu .subcalendar-dropdown-menu-button").trigger("aui-button-invoke");
                }, 0);
            },
            closeMenu: function(context) {
                // close menu if it is open
                var selectorString = getSelectorString(context);
                if ($(selectorString + " .subcalendar-dropdown-menu .aui-dropdown2-active").length > 0) {
                    // click again to close
                    isStoppingTour = false;
                    setTimeout(function() {
                        $(selectorString + " .subcalendar-dropdown-menu .subcalendar-dropdown-menu-button").trigger("aui-button-invoke");
                    }, 0);
                }
            },
            hightLightEditMenuItem: function(context) {
                return menuHelper.selectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-edit");
            },
            unHightLightEditMenuItem: function(context) {
                return menuHelper.unSelectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-edit");
            },
            hightLightRestrictionMenuItem: function(context) {
                return menuHelper.selectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-restrict");
            },
            unHightLightRestrictionMenuItem: function(context) {
                return menuHelper.unSelectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-restrict");
            },
            hightLightShareMenuItem: function(context) {
                return menuHelper.selectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-share");
            },
            unHightLightShareMenuItem: function(context) {
                return menuHelper.unSelectMenuItem(context, " .parent-subCalendar aui-item-link.subcalendar-share");
            }
        };

        var bindHideEvents = function(tourbus) {
            /**
             * Catch click events on the body to see if the click target occurs outside of this popup
             * If it does, the popup will be hidden
             */
            bindHideOnExternalClick(tourbus);
            bindHideOnEscPressed(tourbus);
            bindWindowResize(tourbus);
        };

        var unbindHideEvents = function(tourbus) {
            unbindHideOnExternalClick(tourbus);
            unbindHideOnEscPressed(tourbus);
            unbindWindowResize();
        };

        var bindHideOnEscPressed = function(tourbus) {

            $(document).on("keydown", onKeydown);
        };

        var unbindHideOnEscPressed = function(tourbus) {
            $(document).off("keydown", onKeydown);
        };

        var onKeydown = function(e) {
            if (e.keyCode === 27) {
                isPausing = true;
                bus.stop();
            }
        };

        var bindHideOnExternalClick = function(tourbus) {
            var that = this;
            if (!hasBoundOnExternalClick) {
                $("body").on("click." + externalClickNamespace, function(e) {
                    var $target = $(e.target);
                    // just prevent infinity loop when force close dropdown menu
                    if (!isStoppingTour) {
                        isStoppingTour = true;
                        return;
                    }

                    // hide the popup if the target of the event is not in the dialog
                    if ($target.closest('.tourbus-leg-inner').length === 0) {
                        isPausing = true;
                        tourbus.stop();
                    }
                });
                hasBoundOnExternalClick = true;
            }
        };

        var unbindHideOnExternalClick = function(tourbus) {
            if (hasBoundOnExternalClick) {
                $("body").off("click." + externalClickNamespace);
            }
            hasBoundOnExternalClick = false;
        };

        var bindWindowResize = function(tourbus) {
            $(window).bind("resize." + externalClickNamespace, function(e) {
                tourbus.repositionLegs();
            });
        };

        var unbindWindowResize = function() {
            $(window).unbind("resize." + externalClickNamespace);
        };

        var attachTourTo = function(calendarID) {
            // we only attach to the first one that we found
            var selectorString = getSelectorString({ "calendarID": calendarID });
            if ($(selectorString).is(selectorString)) {
                _.each($("ol.tourbus-legs li.tc-tour-dynamic"), function(tourItem) {
                    var $tourItem = $(tourItem);
                    var targetId = $tourItem.data("el");
                    // update the actual targetId base on calendar
                    $tourItem.attr("data-el", selectorString + " " + targetId);
                });

                return true;
            }

            return false;
        };

        var executeActions = function(calendarID, actionsString) {
            if (!calendarID || !actionsString) {
                AJS.log("Team Calendar tour - could not execute action because invalid calendarID or actionsString");
                return;
            }

            _.each(actionsString.split(" "), function(actionID) {
                var action = actionList[actionID];
                if (action) {
                    AJS.log("Team Calendar tour - leg actions - " + actionID);
                    action({ "calendarID": calendarID });
                    AJS.log("Team Calendar tour - leg actions - " + actionID + " done");
                }
            });


        };

        var markDoneTheTour = function() {
            AJS.log("Team Calendar tour - mark as DONE");
            LocalStorage.setItem(tourFinishKey, "done");
        };

        var isAlreadyDoneTour = function() {
            return LocalStorage.getItem(tourFinishKey) === "done";
        };

        var tour = undefined;
        var bus = undefined;
        return {
            // this only available if we already start the tour
            tcStorage: LocalStorage,
            tourBus: bus,
            start: function(calendarID) {
                // we don't allow to start if there's license error message announced because calendars are readonly
                if ($(".aui-message.error.license-invalid").length > 0) {
                    AJS.log("Team Calendar tour - is available when license is valid");
                    return;
                }

                // we don't allow to start if user already finish the tour
                if (isAlreadyDoneTour()) {
                    AJS.log("Team Calendar tour - is already done");
                    return;
                }

                // should not show tour on page
                if ($(".plugin-calendar-container.page").length > 0) {
                    AJS.log("Team Calendar tour - is only available on MyCalendar");
                    return
                }

                if (!calendarID) {
                    // go and get attached calendar Id from our storage
                    calendarID = LocalStorage.getItem(currentAttachedCalendarKey);
                } else {
                    LocalStorage.setItem(currentLegIndexKey, 0);
                    LocalStorage.setItem(currentAttachedCalendarKey, calendarID);
                }

                // get last leg index from localStorage
                var lastLegIndex = LocalStorage.getItem(currentLegIndexKey);
                if (!lastLegIndex) {
                    lastLegIndex = 0;
                }

                // try to find the calendar to attach the tour
                if (!attachTourTo(calendarID)) {
                    AJS.log("Team Calendar tour - could not attached to target elements");
                    return;
                }

                // clean up the previous tour if it exist
                if (bus) {
                    // close the previous bus
                    isPausing = true;
                    bus.stop();
                    bus.destroy();
                    isPausing = false;
                }
                var tourSelector = '#my-calendar-tour'
                tour = $(tourSelector).tourbus({
                    selector: tourSelector,
                    startAt: lastLegIndex,
                    onDepart: function(tourbus) {
                        AJS.log("Team Calendar tour - depart");
                        bindHideEvents(tourbus);
                    },
                    onStop: function(tourbus) {
                        AJS.log("Team Calendar tour - stop");
                        unbindHideEvents(tourbus);

                        actionList.closeMenu({ "calendarID": calendarID });

                        if (!isPausing) {
                            markDoneTheTour();
                            LocalStorage.setItem(currentLegIndexKey, 0);
                            Confluence.TeamCalendars.fireEventForAnalytics("tour.finish", { stepNumber: -1 });
                            isPausing = false;
                        }
                    },
                    onLegStart: function(leg, tourbus) {
                        AJS.log("Team Calendar tour - leg start");
                        // will store current leg index for go back case
                        LocalStorage.setItem(currentLegIndexKey, tourbus.currentLegIndex);

                        // update template items
                        var $order = leg.$el.find(".order");
                        var currentIndex = tourbus.currentLegIndex;
                        var orderHtml = $order.html();
                        orderHtml = orderHtml.replace("#legIndex", ++currentIndex);
                        orderHtml = orderHtml.replace("#totalLegs", tourbus.totalLegs);
                        $order.html(orderHtml)

                        // execute generic actions before leg start
                        AJS.log("Team Calendar tour - leg - executing before actions");
                        var actionsString = leg.$original.data("before-leg-actions");
                        executeActions(calendarID, actionsString);
                        AJS.log("Team Calendar tour - leg start - before actions executed");

                        leg.reposition();

                        var property = {
                            stepNumber: tourbus.currentLegIndex
                        };
                        Confluence.TeamCalendars.fireEventForAnalytics("tour.step", property);
                    },
                    onLegEnd: function(leg, tourbus) {
                        AJS.log("Team Calendar tour - leg end");

                        // execute generic actions after leg execute
                        AJS.log("Team Calendar tour - leg - executing after actions");
                        var actionsString = leg.$original.data("after-leg-actions");
                        executeActions(calendarID, actionsString);
                        AJS.log("Team Calendar tour - leg end - after actions executed");
                    },
                    leg: {
                        zindex: 2400,
                        width: 270,
                        margin: 15
                    }
                });
                bus = tour.data('tourbus');

                // only depart if the tour is not running state
                if (!bus.running) bus.depart();
            },
            stop: function() {
                if (bus) {
                    markDoneTheTour();
                    bus.stop();
                }
            },
            softStop: function() {
                if (bus) {
                    markDoneTheTour();
                    bus.hideLeg();
                }
            },
            destroy: function() {
                if (bus) bus.destroy();
            },
            // only available after start the tour
            showLeg: function(index) {
                if (bus) bus.showLeg(index);
            },
            // only available after start the tour
            hideLeg: function() {
                if (bus) {
                    bus.hideLeg();

                    var property = {
                        stepNumber: bus.currentLegIndex
                    };
                    //  Confluence.TeamCalendars.fireEventForAnalytics("tour.paused", property);
                }
            },
            isShowCustomEventTypeFrom: function(calendarId) {
                if (isAlreadyDoneTour()) {
                    return false;
                }

                var currentAttachedCalID = LocalStorage.getItem(currentAttachedCalendarKey);
                if (currentAttachedCalID === "" || !currentAttachedCalID) {
                    return false;
                }

                var currentTourIndex = LocalStorage.getItem(currentLegIndexKey);
                if (currentTourIndex !== "0") {
                    return false;
                }

                if (currentAttachedCalID !== calendarId) {
                    return false;
                }

                return true;
            },
            reposition: function() {
                if (bus && !isAlreadyDoneTour()) {
                    bus.repositionLegs();
                }
            }
        };
    }
);
