define("tc/non-space-calendar-entry-view",
    [
        "jquery",
        "underscore",
        "tc-backbone",
        "ajs",
        "tc/calendar-util"
    ],
    function (
        $,
        _,
        Backbone,
        AJS,
        CalUtil
    ) {

        "use strict";

        var NonSpaceCalendarEntryView = Backbone.View.extend({

            initialize: function (options) {
                this.template = Confluence.TeamCalendars.Templates.nonSpaceCalendarEntry;
                this.subCalendar = options.subCalendar;

                this.CalendarPlugin = options.CalendarPlugin;
                this._calendarDiv = options._calendarDiv;
                this.cache = options.cache;
            },

            render: function () {
                this.$el.html(this.template({subCalendar: this.subCalendar}));

                this.spaceKeyAutocompleteField = this.$("input.autocomplete-space");
                this.spaceKeyField = this.$("input.space-key");

                this.spaceKeyAutocompleteField.keypress(_.bind(function (e) {
                    if (e.which !== 13) {
                        this.spaceKeyField.val(e.which);
                    }
                }, this));

                this.spaceKeyAutocompleteField.bind("selected.autocomplete-content", _.bind(function (e, selection) {
                    this.spaceKeyField.val(selection.content.key);
                    this.spaceKeyAutocompleteField.val(selection.content.name);
                }, this));

                this.successIcon = this.$(".success-icon");
                this.errorIcon = this.$(".error-icon");
                this.spinnerIcon = this.$(".spinner");
                this.errorMessageDiv = this.$(".error-message");

                return this.$el;
            },

            getData: function () {
                return {
                    calendarId: this.subCalendar.id,
                    spaceKey: this.spaceKeyField.val()
                }
            },

            isUpdateSuccessfully: function () {
                return this.spaceKeyAutocompleteField.attr("updated") === "true";
            },

            updateSubCalendar: function () {
                if (this.isUpdateSuccessfully()) {
                    return;
                }

                var data = this.getData();
                if (!data.spaceKey) {
                    return;
                }

                this.spinnerIcon.show();

                return $.ajax({
                    cache: false,
                    data: JSON.stringify(data),
                    contentType: "application/json",

                    dataFilter: function (data) {
                        var subCalendarsResponseEntity = data ? JSON.parse(data) : null;
                        if (subCalendarsResponseEntity.success) {
                            CalUtil.mergeSubCalendarObjectsToArray(subCalendarsResponseEntity.payload);
                        }
                        return subCalendarsResponseEntity;
                    },

                    dataType: "json",

                    converters: {
                        "text json": function (jsonObject) {
                            return jsonObject;
                        }
                    },

                    error: _.bind(function (XMLHttpRequest, textStatus, errorThrown) {
                        var errorMessage = JSON.parse(XMLHttpRequest.responseText)[0] || AJS.I18n.getText("calendar3.error.unknown");
                        this._showResult(false, errorMessage);

                        CalUtil.fireEventForAnalytics("non.space.calendars.upgrade.one.calendar.failed");
                    }, this),

                    success: _.bind(function (responseEntity) {
                        if (responseEntity.success) {

                            this._showResult(true, "");
                            this._hide();

                            this.CalendarPlugin.setSubCalendar(data.calendarId, this._calendarDiv, responseEntity.payload);

                            CalUtil.fireEventForAnalytics("non.space.calendars.upgrade.one.calendar.successful");
                        } else {
                            var errorMessage = (responseEntity.fieldErrors && responseEntity.fieldErrors.length) ?
                                    responseEntity.fieldErrors[0].errorMessages[0] : AJS.I18n.getText("calendar3.error.unknown");

                            this._showResult(false, errorMessage);

                            CalUtil.fireEventForAnalytics("non.space.calendars.upgrade.one.calendar.failed");
                        }
                    }, this),

                    type: "POST",
                    timeout: CalUtil.ajaxTimeout,
                    url: Confluence.TeamCalendars.getCalendarServiceBaseUrl("/subcalendar/space")
                });
            },

            _hide: function () {
                this.$el.animate({
                    "height": 0,
                    "opacity": 0
                }, 500, function () {
                    $(this).remove();
                });
            },

            _showResult: function (success, errorMessage) {
                this.spinnerIcon.hide();

                this.successIcon.toggleClass("hidden", !success);
                this.errorIcon.toggleClass("hidden", success);
                this.errorMessageDiv.toggleClass("hidden", success);


                if (success) {
                    this.spaceKeyAutocompleteField.prop("disabled", true);
                    this.spaceKeyAutocompleteField.attr("updated", "true");
                }

                if (errorMessage) {
                    this.errorMessageDiv.text(errorMessage);
                }
            }
        });

        return NonSpaceCalendarEntryView;
    }
);