define("tc/subcalendar-menu-view",
    [
        "jquery",
        "underscore",
        "tc-backbone",
        "tc/myCalendarTour",
        "tc/calendar-util",
        "tc/util",
        "tc/calendar-caldav-subscribe-model",
        "tc/subcalendar-subscribe-dialog",
        "tc/calendar-caldav-subscribe-service"
    ],
    function(
        $,
        _,
        Backbone,
        CalendarTour,
        CalUtil,
        Util,
        CalendarSubscribeModel,
        CalendarSubscribeDialog,
        CalendarSubscribeService
    ) {

        "use strict";

        var SubCalendarMenuView = Backbone.View.extend({
            initialize: function(options) {
                this.cache = options.cache;
                this.CalendarPlugin = options.CalendarPlugin;
                this._calendarDiv = options._calendarDiv;

                this.subCalendar = options.subCalendar;
                this.parentSubCalendar = options.parentSubCalendar;

                this.$subCalendarEntryView = options.subCalendarEntryView;

                this.template = options.template;
                this.subcalendarDisplayer = options.subcalendarDisplayer;
            },

            events: {
                "click .subcalendar-addevent": "_onAddEventClick",

                "click .subcalendar-watch": "_onWatchClick",
                "click .subcalendar-unwatch": "_onUnWatchClick",

                "click .subcalendar-edit, .subcalendar-view": "_onEditClick",

                "click .subcalendar-restrict": "_onRestrictClick",

                "click .subcalendar-remove": "_onRemoveClick",

                "click .subcalendar-share": "_onShareClick",

                "click .subcalendar-subscribe": "_onSubscribeClick",

                "click .subcalendar-icalimport": "_onImportClick",

                "click .subcalendar-reload": "_onRefreshClick",

                "click .subcalendar-setting-reminder": "_onReminderClick",

                "click .custom-event-type-edit, .setting-reminder-setting": "_onEditEventTypeClick",

                "click .color-cell": "_onColorCellClick",

                "click .subcalendar-display-hide, .subcalendar-display-show": "_onDisplayClick"
            },

            render: function() {
                this.$el.html(this.template({
                    "subCalendar": this.subCalendar,
                    "autoConvertSupported": this.CalendarPlugin.isAutoConvertSupported(this._calendarDiv),
                    "subCalendarClasses": this.CalendarPlugin.getParameter(this._calendarDiv, "subCalendarColorClass"),
                    "parentSubCalendar": this.parentSubCalendar
                }));

                this._initExportLink();
                this._initDetailLink();

                return this.$el;
            },

            _initExportLink: function() {
                this.$(".subcalendar-icalexport").attr("href", this.CalendarPlugin.getCalendarServiceBaseUrl(this._calendarDiv, "/export/subcalendar/" + encodeURIComponent(this.subCalendar.id + ".ics") + "?os_authType=basic&isSubscribe=false"));
            },

            _initDetailLink: function() {
                this.$(".subcalendar-page").attr(
                    "href",
                    AJS.contextPath() + "/display/" + encodeURIComponent(this.subCalendar.spaceKey)
                    + "/calendar/" + encodeURIComponent(this.subCalendar.subscriptionId || this.subCalendar.id) + "?calendarName=" + encodeURIComponent(this.subCalendar.name)
                );
            },

            // === Menu items click handlers ===
            _onAddEventClick: function(event) {
                Confluence.TeamCalendars.Dialogs.getEditEventDialog({
                    start: new Date(),
                    localizedStartTime: this.CalendarPlugin.getDefaultStartTime(),
                    localizedEndTime: this.CalendarPlugin.getDefaultEndTime(),
                    allDay: (this._calendarDiv.data(this.CalendarPlugin.PREF_LAST_ALL_DAY_USED) || "true") === "true",
                    subCalendarId: this.subCalendar.id
                },
                    this.subCalendar,
                    this.CalendarPlugin,
                    this._calendarDiv
                ).show();

                event.preventDefault();
            },

            _onWatchClick: function(event) {
                setSubCalendarWatch(this.CalendarPlugin, this._calendarDiv, this.subCalendar, true, _.bind(function() {
                    this.$subCalendarEntryView.addClass("watched").removeClass("not-watched");
                }, this));

                event.preventDefault();
            },

            _onUnWatchClick: function(event) {
                setSubCalendarWatch(this.CalendarPlugin, this._calendarDiv, this.subCalendar, false, _.bind(function() {
                    this.$subCalendarEntryView.removeClass("watched").addClass("not-watched");
                }, this));

                event.preventDefault();
            },

            _onEditClick: function(event) {
                this.CalendarPlugin.showSubCalendarEdit(this._calendarDiv, this.subCalendar);
                //process behaviour when event type panel active
                $("#edit-calendar-dialog").find(".button-panel-button.submit").removeClass("hidden");

                event.preventDefault();
            },

            _onRestrictClick: function(event) {
                this.CalendarPlugin.getSubCalendarRestrictionsDialog(this._calendarDiv, this.subCalendar);

                event.preventDefault();
            },

            _onRemoveClick: function(event) {
                if (this.CalendarPlugin.isSubCalendarDeletable(this._calendarDiv, this.subCalendar)) {
                    if (CalUtil.isJiraSubCalendar(this.subCalendar)) {
                        this.CalendarPlugin.deleteSubCalendar(this._calendarDiv, this.subCalendar);
                    } else {
                        this.CalendarPlugin.removeSubCalendarDialog(this._calendarDiv, this.subCalendar);
                    }
                } else {
                    this.CalendarPlugin.unsubscribeFromSubcalendar(this._calendarDiv, this.subCalendar);
                }
                event.preventDefault();
            },

            _onShareClick: function(event) {
                this.CalendarPlugin.getSubCalendarShareDialog(this._calendarDiv, this.subCalendar);

                event.preventDefault();
            },

            _onSubscribeClick: function(event) {
                var CalendarPlugin = this.CalendarPlugin;
                var calendarDiv = this._calendarDiv;
                var subCalendars = CalendarPlugin.getSubCalendars(calendarDiv);
                var service = new CalendarSubscribeService(CalendarPlugin, calendarDiv);
                var model = new CalendarSubscribeModel({
                    subCalendars: subCalendars,
                    selectedSubCalId: this.subCalendar.id,
                    baseUrl: CalUtil.getParameter(calendarDiv, "baseUrl"),
                    isPrivateUrlsEnabled: CalendarPlugin.getParameter(calendarDiv, "privateUrlsEnabled"),
                    privateUrlProvider: _.bind(service.getPrivateUrlProvider, service)
                });
                var subscribeDialogView = new CalendarSubscribeDialog({
                    calendarPlugin: CalendarPlugin,
                    calendarDiv: calendarDiv,
                    model: model
                });
                subscribeDialogView.render();
                event.preventDefault();
            },

            _onImportClick: function(event) {
                var callbackHandler = this.CalendarPlugin.getRenderedMacroCallbackHandler(this._calendarDiv);
                var subCalendarImportDialog = Confluence.TeamCalendars.Dialogs.getSubCalendarImportDialog(callbackHandler, this.subCalendar);
                Util.showDialogInUpdatedHeight(subCalendarImportDialog);
                $("input[name='name']", subCalendarImportDialog.getCurrentPanel()).focus();
                $("#inline-dialog-more").removeClass("active");
                $("#inline-dialog-inline-dialog-more").hide();
                Confluence.TeamCalendars.fireEventForAnalytics("add.calendar.click.import." + Confluence.TeamCalendars.getCalendarContext());

                event.preventDefault();
            },

            _onRefreshClick: function(event) {
                if (!this.CalendarPlugin.isSubCalendarEventsUpdatable(this._calendarDiv, this.subCalendar) || isSubCalendarOrAnyOfItsChildrenJiraSubCalendar(this.subCalendar) || CalUtil.isSubscriptionSubCalendar(this.subCalendar)) {
                    var that = this;

                    $.ajax({
                        cache: false,
                        data: {
                            subCalendarId: this.subCalendar.id
                        },
                        dataType: "json",
                        error: function(XMLHttpRequest, textStatus, errorThrown) {
                            that.CalendarPlugin.showAjaxError(that._calendarDiv, XMLHttpRequest, textStatus, errorThrown, that.CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_LIST_EVENTS + that.CalendarPlugin.getStringAsHex(that.subCalendar.id));
                        },
                        success: function(responseEntity) {
                            that.CalendarPlugin.setGenericErrors(that._calendarDiv, null, that.CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_LIST_EVENTS + that.CalendarPlugin.getStringAsHex(that.subCalendar.id));
                            if (responseEntity.success) {
                                that.CalendarPlugin.reloadSubCalendar(that._calendarDiv, that.subCalendar.id);
                            }
                        },
                        type: "PUT",
                        timeout: this.CalendarPlugin.ajaxTimeout,
                        url: this.CalendarPlugin.getCalendarServiceBaseUrl(this._calendarDiv, "/subcalendars/admin/refresh.json")
                    });
                } else {
                    // Just reload if the sub-calendar is not sourced externally - the cached data need not to be flushed for the reload
                    // because the plugin will keep it updated
                    this.CalendarPlugin.reloadSubCalendar(this._calendarDiv, this.subCalendar.id);
                }

                event.preventDefault();
            },

            _onReminderClick: function(event) {
                setSubCalendarReminderMe(this.CalendarPlugin, this._calendarDiv, this.subCalendar, this.subCalendar.id, _.bind(function() {
                    this.$subCalendarEntryView.find(".reminder-me").toggleClass("hidden");
                    this.$subCalendarEntryView.find(".unreminder-me").toggleClass("hidden");
                }, this), this.$subCalendarEntryView.find(".reminder-me.hidden").length ? false : true);

                event.preventDefault();
            },

            _onEditEventTypeClick: function(event) {
                var eventTypeName = this.$subCalendarEntryView.data("customEventTypeId");

                if (!eventTypeName) {
                    eventTypeName = CalUtil.isInternalSubscriptionSubCalendar(this.subCalendar) ? this.subCalendar.subscriptionType : this.subCalendar.type;
                }

                this.CalendarPlugin.showCustomEventEditForm(this._calendarDiv, this.parentSubCalendar, eventTypeName);

                event.preventDefault();
            },

            _onColorCellClick: function(event) {
                if (!this.CalendarPlugin.isProcessingSubCalendar(this._calendarDiv)) {
                    this.CalendarPlugin.setProcessingSubCalendar(this._calendarDiv, true);

                    var colourCell = $(event.target);


                    var childSubCalendarToUpdate = this.subCalendar;
                    var that = this;
                    setSubCalendarColour(
                        this.CalendarPlugin,
                        this._calendarDiv,
                        childSubCalendarToUpdate,
                        colourCell.data("colour"),
                        function(responseEntity) {
                            if (responseEntity.success) {
                                that.cache.removeCalendar(childSubCalendarToUpdate.id);
                                that.CalendarPlugin.setGenericErrors(that._calendarDiv, null, that.CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_UPDATE);
                                that.CalendarPlugin.setSubCalendars(that._calendarDiv, responseEntity.payload);
                                that.CalendarPlugin.removeSubCalendarEventSource(that._calendarDiv, childSubCalendarToUpdate.id);
                                that.CalendarPlugin.updateAvailableSubCalendarsInSubCalendarPanel(that._calendarDiv);

                                that.$subCalendarEntryView.removeClass().addClass("aui-dd-parent " + colourCell.data("colour"));

                            } else {
                                // Ideally, we should allow the edit to go through, even if the user hasn't authenticated with JIRA,
                                // but because it is an existing bug, and so far, no one has complained about it, it might not be
                                // worth the effort.
                                //
                                // A warning like the below would prompt the user to do the right (tm) thing, and it does fix
                                // https://jira.atlassian.com/browse/TEAMCAL-1537
                                //
                                that.CalendarPlugin.setGenericErrors(that._calendarDiv, AJS.I18n.getText("calendar3.error.edit.jira.colour.notpermitted"), that.CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_UPDATE);
                            }

                            setNotProcessingSubCalendar(that.CalendarPlugin, that._calendarDiv);
                        },
                        setNotProcessingSubCalendar(that.CalendarPlugin, that._calendarDiv)
                    );
                }

                event.preventDefault();
            },

            _onDisplayClick: function(event) {
                this.subcalendarDisplayer.toggleVisibility();
                event.preventDefault();
            }
        });

        // private methods
        var setSubCalendarWatch = function(CalendarPlugin, calendarDiv, subCalendar, watch, successCallback) {
            $.ajax({
                cache: false,
                data: {
                    subCalendarId: subCalendar.id
                },
                dataType: "json",
                error: function(XMLHttpRequest, textStatus, errorThrown) {
                    CalendarPlugin.showAjaxError(calendarDiv, XMLHttpRequest, textStatus, errorThrown, CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_WATCH + CalendarPlugin.getStringAsHex(subCalendar.id));
                },
                success: function() {
                    if (successCallback) {
                        successCallback();
                    }
                },
                type: watch ? "PUT" : "DELETE",
                timeout: CalendarPlugin.ajaxTimeout,
                url: CalendarPlugin.getCalendarServiceBaseUrl(calendarDiv, "/preferences/subcalendars/watch.json")
            });
        };

        var isSubCalendarOrAnyOfItsChildrenJiraSubCalendar = function(subCalendar) {
            return subCalendar && (
                CalUtil.isJiraSubCalendar(subCalendar) ||
                $.grep(subCalendar.childSubCalendars || [], function(childSubCalendar) {
                    return isSubCalendarOrAnyOfItsChildrenJiraSubCalendar(childSubCalendar);
                }).length
            );
        };

        var setSubCalendarReminderMe = function(CalendarPlugin, calendarDiv, subCalendar, subCalendarChildrenId, successCallback, isReminder) {
            $.ajax({
                cache: false,
                data: {
                    subCalendarId: subCalendarChildrenId,
                    isReminder: isReminder
                },
                dataType: "json",
                error: function(XMLHttpRequest, textStatus, errorThrown) {
                    CalendarPlugin.showAjaxError(calendarDiv, XMLHttpRequest, textStatus, errorThrown, CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_WATCH + CalendarPlugin.getStringAsHex(subCalendar.id));
                },
                success: function() {
                    if (successCallback) {
                        successCallback();
                    }
                },
                type: "PUT",
                timeout: CalendarPlugin.ajaxTimeout,
                url: CalendarPlugin.getCalendarServiceBaseUrl(calendarDiv, "/preferences/subcalendars/setreminder.json")
            });
        };

        var setNotProcessingSubCalendar = function(CalendarPlugin, calendarDiv) {
            CalendarPlugin.setProcessingSubCalendar(calendarDiv, false);
        };

        var setSubCalendarColour = function(CalendarPlugin, calendarDiv, subCalendar, colour, successCallback, errorCallback) {
            var spinnerDefer = CalendarPlugin.setSubCalendarSpinnerIconVisible(true);

            $.ajax({
                cache: false,
                converters: {
                    "text json": function(jsonObject) {
                        return jsonObject;
                    }
                },
                data: (function() {
                    var ajaxData = {
                        subCalendarId: subCalendar.id,
                        color: colour
                    };
                    if (CalendarPlugin.getParameter(calendarDiv, "include")) {
                        ajaxData.include = CalendarPlugin.getParameter(calendarDiv, "include");
                    }

                    CalUtil.putCalendarContextParams(ajaxData);

                    return ajaxData;
                })(),
                dataType: "json",
                dataFilter: function(data) {
                    var subCalendarsResponseEntity = data ? JSON.parse(data) : null;
                    if (subCalendarsResponseEntity.success) {
                        CalendarPlugin.mergeSubCalendarObjectsToArray(subCalendarsResponseEntity.payload);
                    }
                    return subCalendarsResponseEntity;
                },
                error: function(XMLHttpRequest, textStatus, errorThrown) {
                    CalendarPlugin.showAjaxError(calendarDiv, XMLHttpRequest, textStatus, errorThrown, CalendarPlugin.ERROR_CLASS_SUB_CALENDAR_UPDATE);

                    if (errorCallback) {
                        errorCallback(XMLHttpRequest, textStatus, errorThrown);
                    }
                },
                success: function(responseEntity) {
                    if (successCallback) {
                        successCallback(responseEntity);
                    }

                },
                complete: function() {
                    if (spinnerDefer) spinnerDefer.resolve();
                },
                type: "PUT",
                timeout: CalendarPlugin.ajaxTimeout,
                url: CalendarPlugin.getCalendarServiceBaseUrl(calendarDiv, "/subcalendars/color.json")
            });
        };

        return SubCalendarMenuView;
    }
);
