//This code acts as an adapter for the CHAP timeline (see timeline.js) so it can work with fullcalendar.js and calendar-plugin.js.
(function($){
    Confluence.TeamCalendars.getTimelineAdapter = function(_element, _calendarPlugin, _calendarDiv, _fullCalendarView) {
        var calendarPlugin,
            calendarDiv,
            element,
            fullCalendarView,
            millisecondsInAWeek,
            today,
            visStart, //The start of the visible range
            visEnd, //The end of the visible range
            loadedRangeStart, //The start of the range that has been loaded
            loadedRangeEnd, //The end of the range that has been loaded
            events,
            timelineEvents, //The same events as above, but transformed into a format the timeline library understands
            eventDetailsDialog = null,
            timeline, //The CHAP timeline object
            //If lastModifiedSubCalendar is not null then we have modified a calendar in the timeline view and asked
            //CalendarPlugin to refresh the view. This will cause the calendar to be removed from the cache, and a
            //refresh to be preformed without the calendar, and then another refresh with the calendar. To avoid
            //flickering we ignore the first refresh in this case (see renderEvents()).
            lastModifiedSubCalendar = null,
            DEFAULT_TIMELINE_HEIGHT = 500,
            TIMELINE_INCREMENT = 0.2,
            lastSelectedEvent;

        //Initialisation
        calendarPlugin = _calendarPlugin;
        calendarDiv = _calendarDiv;
        element = _element;
        fullCalendarView = _fullCalendarView;
        millisecondsInAWeek = 1000 * 60 * 60 * 24 * 7;
        today = new Date();
        visStart = new Date(today.getTime() - millisecondsInAWeek * 2);
        visEnd = new Date(today.getTime() + millisecondsInAWeek * 2);
        fullCalendarView.visStart = visStart;
        fullCalendarView.visEnd = visEnd;
        loadedRangeStart = visStart;
        loadedRangeEnd = visEnd;

        function drawVisualization(data) {
            var height,
                options,
                oldScrollTop,
                oldHeight,

                saveScrollPos = function() {
                    if (timeline) {
                        oldScrollTop = timeline.dom.frame.scrollTop;
                        oldHeight = $( timeline.dom.content).height();
                    }
                },

                restoreScrollPos = function() {
                    var heightdiff = $(timeline.dom.content).height() - oldHeight;

                    if (heightdiff > 0) {
                        timeline.dom.frame.scrollTop = oldScrollTop + heightdiff;
                    } else {
                        timeline.dom.frame.scrollTop = oldScrollTop;
                    }
                },

                restoreSelection = function() {
                    if (lastSelectedEvent) {
                        var tmpTimelineEventElem = $("<div/>");

                        $.each(data, function(dataIdx, theData) {
                            var dataEventId = tmpTimelineEventElem.empty().html(theData.content).find("[data-event-id]").data("event-id");
                            if (dataEventId === lastSelectedEvent.id)
                                timeline.setSelection([ { "row" : dataIdx }], true);
                        });

                        lastSelectedEvent = null;
                    }
                },

                styleLabelEvent = function() {
                    var listItemEvent = $(timeline.dom.content).find(".timeline-event-content");
                    $.each(listItemEvent, function(dataIdx, theData) {
                        var parentItem = $(theData).parent();
                        if(parentItem !== undefined && parentItem.length > 0) {
                            var positionLeftItemEvent = $(parentItem).position().left;
                            var widthTimeline = $(timeline.dom.content).width();
                            if($(parentItem).width() >= widthTimeline) {
                                if(positionLeftItemEvent > 0) {
                                    $(theData).css("padding-left", (widthTimeline - positionLeftItemEvent)/2);
                                } else {
                                    if(widthTimeline >= ($(parentItem).width() + positionLeftItemEvent)) {
                                        $(theData).css("padding-left",($(parentItem).width() + positionLeftItemEvent)/2 - positionLeftItemEvent);
                                    } else {
                                        $(theData).css("padding-left", widthTimeline/2 - positionLeftItemEvent);
                                    }
                                }
                            } else {
                                var paddingLeft = ($(parentItem).width() + positionLeftItemEvent)/2 - positionLeftItemEvent;
                                if((widthTimeline >= ($(parentItem).width() + positionLeftItemEvent)) && paddingLeft >= 0 && (paddingLeft < $(parentItem).width())) {
                                    $(theData).css("padding-left", paddingLeft);
                                } else {
                                    $(theData).css("padding-left", 0);
                                }
                            }
                        }
                    });
                };

            height = calendarPlugin.getTimelineHeight(calendarDiv) || DEFAULT_TIMELINE_HEIGHT;

            var maxMonthToDisplayTimelineCalendar = parseInt(calendarPlugin.getMaxMonthToDisplayTimelineCalendar(calendarDiv)) || 6; // default 6 month maximum zoom

            //Configuration options for CHAP timeline.
            //See http://almende.github.com/chap-links-library/js/timeline/doc/#Configuration_Options
            options = {
                width:  "99%", //setting 'width 100%' causes the right border to be cutoff in myCalendar view
                height: "auto",
                viewHeight: height + "px", //Size of the containing frame
                minHeight: height,
                editable: true,
                enableKeys: true,
                axisOnTop: false,
                showNavigation: true,
                style: "box",
                start: visStart,
                end: visEnd,
                intervalMin: 1000 * 60 * 60 * 6, //6 hour minimum zoom
                intervalMax: millisecondsInAWeek * 4 * maxMonthToDisplayTimelineCalendar,
                MONTHS: calendarPlugin.getMonthNames(),
                DAYS: calendarPlugin.getDayNames()
            };

            saveScrollPos();

            timeline = new links.Timeline(element[0]);

            //Add event listeners
            links.events.addListener(timeline, 'select', onSelect);
            links.events.addListener(timeline, 'change', onChange);
            links.events.addListener(timeline, 'rangechange', onRangeChange);
            links.events.addListener(timeline, 'rangechanged', onRangeChanged);
            links.events.addListener(timeline, 'add', onAdd);
            links.events.addListener(timeline, 'moved', onMove);

            timeline.draw(data, options);

            restoreScrollPos();
            restoreSelection();
            styleLabelEvent();
            onRangeChange();
        }

        function onRangeChange() {
            hideEventDetailsDialog();
        }

        //Callback function for the select item
        function onSelect(event) {
            var calEvent = getSelectedEvent(),
                target = $(".timeline-event-box.timeline-event-selected"); //Gets selected box (one day event)

            if (!calEvent) {
                return;
            }

            if (!target.length) {
                target = $(".timeline-event-range.timeline-event-selected"); //Gets selected range
            }

            if (!isEditable(calEvent)) {
                timeline.setSelection([]); //Unselect the item so it can't be dragged and dropped.
            }

            if (calEvent.disableResizing) {
                $(".timeline-event-range-drag-left, .timeline-event-range-drag-right").addClass("hidden");
            } else {
                $(".timeline-event-range-drag-left, .timeline-event-range-drag-right").removeClass("hidden");
            }

            eventDetailsDialog = calendarPlugin.getEventDetailsDialog(calendarDiv, calEvent, target);
            //Setting the z-index higher is necessary to prevent any of the timeline event lines for going
            //on top of the dialog.
            eventDetailsDialog[0].style.zIndex = 1000;
            eventDetailsDialog.show();

            if (target.position().left < 0)
            {
                //Tragically the AUI inline dialog does not handle this case well at all. We push a function on to the
                //back of the execution stack to reposition the dialog at the left hand side of the timeline container.
                setTimeout(function() {
                    var left = target.parent().offset().left;
                    eventDetailsDialog.offset({left : left});
                }, 0);
            }
        }

        //Callback function for the change item
        function onChange() {
            var calEvent = getSelectedEvent(),
                timelineEvent = getSelectedTimelineEvent(),
                transformedEvent = transformEventForServer(calEvent, timelineEvent);

            lastSelectedEvent = calEvent;

            if (!isEditable(calEvent)) {
                //In theory this should not happen, but do this just in case.
                timeline.cancelChange();
            } else {
                var spinnerDefer = calendarPlugin.setSubCalendarSpinnerIconVisible(calendarDiv, true);
                calendarPlugin.updateEvent(
                        calendarDiv,
                        transformedEvent,
                        function(XMLHttpRequest, textStatus, errorThrown) {
                            if (spinnerDefer) spinnerDefer.resolve();
                            calendarPlugin.showAjaxError(
                                    calendarDiv,
                                    XMLHttpRequest,
                                    textStatus,
                                    errorThrown,
                                    calendarPlugin.ERROR_CLASS_EVENT_UPDATE);
                        },
                        function(responseEntity) {
                            if (spinnerDefer) spinnerDefer.resolve();
                            if (responseEntity.success) {
                                lastModifiedSubCalendar = calEvent.subCalendarId;
                                calendarPlugin.reloadSubCalendar(calendarDiv, calEvent.subCalendarId);
                            } else {
                                Confluence.TeamCalendars.setGenericErrors(
                                        calendarDiv,
                                        formatFieldErrors(responseEntity.fieldErrors),
                                        calendarPlugin.ERROR_CLASS_EVENT_UPDATE);
                            }
                        }
                )
            }

            hideEventDetailsDialog();
        }

        function onRangeChanged() {
            var range = timeline.getVisibleChartRange();
            updateVisibleRange(range.start.getTime(), range.end.getTime());
        }

        function updateDateDisplay(visStart, visEnd) {
            if (!visStart && !visEnd) {
                var visibleRange = timeline.getVisibleChartRange();
                visStart = new Date(visibleRange.start);
                visEnd = new Date(visibleRange.end);
            }

            Confluence.TeamCalendars.updateDateDisplay(
                calendarDiv,
                _fullCalendarView.formatDate(visStart, "dd MMM")
                + " &mdash; "
                + _fullCalendarView.formatDate(visEnd, "dd MMM, yyyy")
            );
        }

        function updateVisibleRange(rangeStart , rangeEnd) {
            var margin = (rangeEnd - rangeStart) / 2;

            visStart = new Date(rangeStart);
            visEnd = new Date(rangeEnd);

            if (loadedRangeStart.getTime() > rangeStart || loadedRangeEnd.getTime() < rangeEnd) {
                loadedRangeStart.setTime(rangeStart - margin);
                loadedRangeEnd.setTime(rangeEnd + margin);
                fullCalendarView.visStart = loadedRangeStart;
                fullCalendarView.visEnd = loadedRangeEnd;
                fullCalendarView.update();
            }

            updateDateDisplay(visStart, visEnd);
        }

        //Callback for adding an event
        function onAdd(properties) {
            Confluence.TeamCalendars.Dialogs.getEditEventDialog({
                    start : properties.start,
                    localizedStartTime : calendarPlugin.getDefaultStartTime(),
                    localizedEndTime : calendarPlugin.getDefaultEndTime(),
                    allDay : true
                },
                null,
                calendarPlugin,
                calendarDiv).show();
        }

        function onMove() {
            hideEventDetailsDialog();
        }

        //Helper functions
        function isEditable(event) {
            return calendarPlugin.isEventEditable(calendarDiv, event);
        }

        function getSelectedRow() {
            var row = undefined;
            var sel = timeline.getSelection();
            if (sel.length) {
                if (sel[0].row != undefined) {
                    row = sel[0].row;
                }
            }
            return row;
        }

        function getSelectedEvent() {
            return events[getSelectedRow()];
        }

        function getSelectedTimelineEvent() {
            return timelineEvents[getSelectedRow()];
        }

        function hideEventDetailsDialog() {
            eventDetailsDialog && eventDetailsDialog.hide();
        }

        //In theory this should never happen, but lets format the field error messages into something readable
        //just in case. This should save us some time with debugging.
        function formatFieldErrors(fieldErrors) {

            return $.map(fieldErrors, function(fieldError, i) {
                return fieldError.field + " : " + fieldError.errorMessages.toString()
            });
        }

        //Massage our events into the format that CAHP timeline uses.
        function transformToTimelineDataFormat(events) {
            function calculateEndDate(event) {
                var end;

                if (!event.allDay) {
                    return event.end;
                }

                //TEAMCAL-1035 - This causes the event to be rendered as a box, and not a duration. We want to do this
                //for all 'all day' events eventually, but it needs some design work. For now we do this only for Jira
                //versions.
                if (event.className[0] === "jira-version" || (event.className[0] === "jira-issue"  && event.localizedStartDate === event.localizedEndDate)) {
                    return null;
                }

                //So that all day events will be rendered properly in the timeline view
                end = event.end || new Date(event.start.getTime());
                end = new Date(end.getTime());
                end.setHours(end.getHours() + 23);
                end.setMinutes(59);
                end.setSeconds(59);
                return end;
            }

            function calculateEventContent(event) {
                var className = event.className[0];

                function calculateTitle() {
                    if ($.inArray("jira-version", event.className) !== -1 || $.inArray("jira-issue", event.className) !== -1)
                        return event.title;

                    if ($.inArray("other", event.className) !== -1 || event.eventType === "custom") {
                        if (event.shortTitle) {
                            return event.shortTitle + ": " + event.title;
                        } else {
                            if (event.invitees && event.invitees.length) {
                                return event.invitees[0].displayName + ": " + event.title;
                            } else {
                                return event.title;
                            }
                        }
                    } else {
                        return event.shortTitle || event.title;
                    }
                }

                switch (className) {
                    case "birthdays" :
                        return Confluence.TeamCalendars.Templates.timelineEventContentWithEventTypeIcon({
                            eventId: event.id,
                            className: className,
                            title: calculateTitle()
                        });

                        break;
                    case "travel":
                    case "leaves":
                    case "other":
                        if (event.invitees && event.invitees.length) {
                            if (event.invitees.length > 1) {
                                return Confluence.TeamCalendars.Templates.timelineEventContent({
                                    eventId: event.id,
                                    iconClass: "people",
                                    title: calculateTitle()
                                });
                            } else {
                                return Confluence.TeamCalendars.Templates.timelineEventContentSingleInvitee({
                                    eventId: event.id,
                                    iconUrl: event.iconUrl,
                                    title:calculateTitle()
                                });
                            }
                        } else {
                            return Confluence.TeamCalendars.Templates.defaultTimelineEventContent({
                                eventId: event.id,
                                title: event.title
                            });
                        }
                    break;

                    case "jira-issue" :
                    case "jira-version" :
                    case "greenhopper-sprint" :
                        return Confluence.TeamCalendars.Templates.timelineEventContent({
                            eventId: event.id,
                            iconClass: className,
                            title: calculateTitle()
                        });
                    break;
                }

                return Confluence.TeamCalendars.Templates.defaultTimelineEventContent({
                    eventId: event.id,
                    title: event.title
                });
            }

            return $.map(events, function(event) {
                return {
                    start : event.start,
                    end : calculateEndDate(event),
                    content : calculateEventContent(event),
                    className : event.colorScheme
                };
            });
        }

        //Transform the events into the format that calednar-plugin will expect, which is really just the format
        //that the /calendar/events.json resource expects
        //event is a parameter in "our" format
        //timelineEvent is an event in the format used by CHAP timeline
        function transformEventForServer(event, timelineEvent) {
            var transformedEvent = {},
                eventSubCalendar = calendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId);

            if (
                    event.className[0] == "jira-version" || (event.className[0] === "jira-issue"  && event.localizedStartDate === event.localizedEndDate) // Made this first condition because it seems that timelineEvent.end for JIRA version is null
                    || (event.allDay
                        && timelineEvent.end.getHours() == 23
                        && timelineEvent.end.getMinutes() == 59
                        )
                ) { //TEAMCAL-1035
                //If we get here assume we are still an all day event
                transformedEvent.allDayEvent = true;
                if (timelineEvent.end) {
                    timelineEvent.end.setHours(0);
                    timelineEvent.end.setMinutes(0);
                    timelineEvent.end.setSeconds(0);
                }
            } else {
                transformedEvent.allDayEvent = false;
            }

            transformedEvent.subCalendarId = eventSubCalendar.parentId;
            transformedEvent.uid = event._id;
            transformedEvent.originalStartDate = event.originalStart || "";
            transformedEvent.what = event.title;
            transformedEvent.url = event.workingUrl; // TEAMCAL-993
            transformedEvent.start = require("confluence/date-time").formatPlainDateTime(timelineEvent.start);
            transformedEvent.end = require("confluence/date-time").formatPlainDateTime(timelineEvent.end || timelineEvent.start);
            transformedEvent.where = event.where;
            transformedEvent.description = event.description;
            transformedEvent.freq = event.recur ? event.recur.freq : "";
            transformedEvent.byday = event.recur ? event.recur.byDay : "";
            transformedEvent.interval = event.recur ? event.recur.interval : "";
            transformedEvent.until = event.recur ? event.recur.localizedUntil || "" : "";
            transformedEvent.recurrenceId = event.recurId || "";
            transformedEvent.editAllInRecurrenceSeries = "false";
            transformedEvent.dragAndDropUpdate = true;
            transformedEvent.eventType = event.eventType;
            transformedEvent.customEventTypeId = event.customEventTypeId;

            if (eventSubCalendar.eventInviteesSupported) {
                transformedEvent.person = $.map(event.invitees || [], function(invitee) {
                    return invitee.id;
                });
            }

            return transformedEvent;
        }

        //Public interface to the timeline adapter
        return {
            renderEvents : function(_events) {
                var subCalIds = $.map(_events, function(event) { return event.subCalendarId });

                if (lastModifiedSubCalendar && $.inArray(lastModifiedSubCalendar, subCalIds) === -1) {
                    lastModifiedSubCalendar = null;
                    return;
                }

                events = _events;
                timelineEvents = transformToTimelineDataFormat(_events);
                drawVisualization(timelineEvents);
                updateDateDisplay();
            },

            setDate : function(date) {
                var viewSize = visEnd.getTime() - visStart.getTime();

                visStart = new Date(date.getTime() - (viewSize / 2));
                visEnd = new Date(date.getTime() + (viewSize / 2));
                updateVisibleRange(visStart.getTime(), visEnd.getTime());
            },

            next : function() {
                timeline.move(TIMELINE_INCREMENT);
                onRangeChanged();
            },

            prev : function() {
                timeline.move(-TIMELINE_INCREMENT);
                onRangeChanged();
            },

            gotoToday : function() {
                timeline.setVisibleChartRangeNow();
                onRangeChanged();
            }
        };
    };
})(AJS.$);
