/**
 * @module confluence-quick-edit/scroll-util
 */
define('confluence-quick-edit/scroll-util', [
    'window',
    'document',
    'jquery'
], function(
    window,
    document,
    $
) {
    'use strict';

    return {
        scrollToElement: function() {
            if (!this.scrollWindowToElement()) {
                // try and scroll the closest overflow set parent (e.g. Documentation Theme)
                this.scrollOverflowContainerToElement();
            }
            return this;
        },

        /**
         * The default theme has scrollbars on the window and therefore this
         * method can make sure the supplied element is visible. However other themes don't scroll the
         * window so this method will return true if it successfully scrolls and false if it is unable
         * to move the window.
         */
        scrollWindowToElement: function() {
            var $element = this;
            function getScrollY() {
                if ('pageYOffset' in window) { // all browsers, except IE before version 9
                    return window.pageYOffset;
                } // Internet Explorer before version 9
                // we ignore zoom factor which was only an issue before IE8
                return document.documentElement.scrollTop;
            }

            var scrollY = getScrollY();

            var windowHeight;
            if (typeof (window.innerWidth) === 'number') {
                windowHeight = window.innerHeight;
            } else if (document.documentElement && document.documentElement.clientWidth) {
                // IE 6+ in 'standards compliant mode'
                windowHeight = document.documentElement.clientHeight;
            } else {
                // something old and creaky - just try to make sure the element will be visible and return true so we consider this successful
                $element[0].scrollIntoView(false);
                return true;
            }

            var elementVerticalPosition = $element.offset().top;
            var elementHeight = $element.height();

            var extra = 40; // the calculation seems to be off by 40 pixels - I don't know why (perhaps padding on $element?)

            if ((elementVerticalPosition + elementHeight + extra) > scrollY + windowHeight) {
                $element[0].scrollIntoView(false); // align to the bottom of the viewport
                window.scrollBy(0, extra);
                return scrollY !== getScrollY(); // did we successfully scroll the window?
            }
            // no scrolling was necessary
            return true;
        },

        /**
         * Find the closest parent with the CSS property overflow set to either 'scroll' or 'auto' and
         * scroll this to show the element.
         *
         * @return true if successfully found a parent to scroll.
         */
        scrollOverflowContainerToElement: function() {
            var $element = this;
            var $parent = null;

            $element.parents().each(function(index) {
                var overflow = $(this).css('overflow');
                if (overflow === 'auto' || overflow === 'scroll') {
                    $parent = $(this);
                    return false;
                }
            });

            if (!$parent) {
                return false;
            }

            var height = $parent.height();

            var extra = 40; // the calculation seems to be off by 40 pixels - I don't know why (perhaps padding on $element?)
            var elementVerticalPosition = $element.offset().top;
            var elementHeight = $element.height();
            var differential = height - (elementVerticalPosition + elementHeight + extra);

            if (differential < 0) {
                $parent[0].scrollTop = $parent[0].scrollTop - differential;
            }

            return true;
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-quick-edit/scroll-util', function(ScrollUtil) {
    'use strict';

    var $ = require('jquery');

    $.fn.extend(ScrollUtil);
});
