define('confluence/softwareproject/software-project-dialog/panel-view', [
    'jquery',
    'underscore',
    'ajs',
    'backbone',
    'confluence/softwareproject/dialog/abstract-panel-view',
    'confluence/softwareproject/util/config',
    'confluence/softwareproject/util/service'
],
function(
    $,
    _,
    AJS,
    Backbone,
    AbstractPanelView,
    config,
    service
) {
    'use strict';

    var SoftwareProjectPanelView = AbstractPanelView.extend({
        events: {
            'change .jira-projects': '_onSelectProjectChanged'
        },

        initialize: function() {
            AbstractPanelView.prototype.initialize.apply(this, arguments);

            this.panelTitle = '';
            this.panelId = 'jira-sprint-panel';

            // a map of template functions
            this.template = Confluence.SpaceBlueprints.SoftwareProject.Dialog;
        },

        render: function(options) {
            AbstractPanelView.prototype.render.apply(this, arguments);

            var template = this.template.dialogForm({});
            this.$el.html(template);

            this.servers = [];
            this.primaryServer = null;

            // update default values of formData
            _.extend(this.formData.defaults, {
                selectedProject: null
            });

            this.listenTo(this.formData, 'change:selectedServer', function() {
                this.reset();

                if (this.formData.get('isValid')) {
                    this.projectsPromise = this._fetchProjectData();
                }
            });

            _.extend(this.view, {
                $projects: this.$('.jira-projects'),
                $errorMessage: this.$('.error-messages'),
                $createButton: this.dialogView.$el.find('.create-dialog-create-button'),
                $spaceName: this.$("input[name=name]"),
                $spaceKey: this.$("input[name=spaceKey]")
            });

            this.reset();
            this._initSelect2Fields();
            this.initServerField();
        },

        _initSelect2Fields: function() {
            var self = this;

            var fillProjectDataBound = this._fillProjectData.bind(this);

            var format = function(project) {
                var text = project.name || project.text;
                var projectIconUrl = project.avatarUrls !== undefined ? project.avatarUrls['16x16'] : '';

                if (projectIconUrl) {
                    return self.template.selectOptionWithImage({
                        text: text,
                        projectIconUrl: projectIconUrl
                    })
                }

                return project.text;
            };

            // setup board select
            this.setupSelect2({
                $el: this.view.$projects,
                placeholderText: AJS.I18n.getText('jira.project.placeholder'),
                isRequired: true,
                overrideSelect2Ops: {
                    query: function(query) {
                        fillProjectDataBound(query.term, query.callback);
                    },
                    formatResult: format,
                    formatSelection: format
                }
            });
        },

        reset: function() {
            AbstractPanelView.prototype.reset.apply(this, arguments);

            this.resetSelect2Options(this.view.$projects);
        },

        /**
         * Fetch project data.
         * @private
         */
        _fetchProjectData: function() {
            var serverId = this.formData.get('selectedServer').id;

            return service.loadProjectsData(serverId)
                .done(function(projects) {
                    // format data to adapt select2 requirement
                    _.each(projects, function(project) {
                        project.text = project.name;
                    });

                    return projects;
                }.bind(this))
                .fail(function(xhr, errorString, errorStatus) {
                    this.handleAjaxRequestError(this.view.$projects, errorStatus);
                }.bind(this));
        },

        /**
         * Filter and load project data select box.
         * @private
         */
        _fillProjectData: function(queryTerm, callback) {

            this.toggleSelect2Loading(this.view.$projects, true, true);

            this.projectsPromise.done(function(projects) {
                this.toggleSelect2Loading(this.view.$projects, false, true);

                projects = _.filter(projects, function(project) {
                    queryTerm = $.trim(queryTerm.toLowerCase());
                    var projectName = project.name !== undefined ? $.trim(project.name.toLowerCase()): '';

                    return (projectName.indexOf(queryTerm) >= 0);
                });

                var data = {
                    results: projects,
                    text: 'name'
                };

                callback(data);
            }.bind(this));
        },

        _onSelectProjectChanged: function() {
            var selectedProject = this.view.$projects.select2('data');

            if (!selectedProject.name) {
                selectedProject.name = selectedProject.text;
            }

            this.formData.set('selectedProject', selectedProject);
            this.view.$spaceName
                    .val(selectedProject.name)
                    .trigger('focus');

            // wait for spaceKey is filled data first and then trigger events to start validation
            _.delay(function() {
                this.view.$spaceName.trigger('change');
            }.bind(this));
        },

        getUserInputData: function() {
            var userInputData = null;

            if (this.validate()) {
                // this.formData has all data users input.
                userInputData = {
                    serverId: this.formData.get('selectedServer').id,
                    projectId: this.formData.get('selectedProject').id,
                    projectName: this.formData.get('selectedProject').name
                };
            }

            return userInputData;
        },

        /**
         * Do some validations before submiting the dialog.
         * @returns {boolean}
         */
        validate: function() {
            var valid = true;

            valid = valid && this.validateRequiredFields(this.view.$servers, AJS.I18n.getText('confluence.software.jira.project.server.required'));
            valid = valid && this.validateRequiredFields(this.view.$projects, AJS.I18n.getText('confluence.software.jira.project.required'));
            return valid;
        }
    });

    return SoftwareProjectPanelView;
});
