// The `confluence/date-time` module is an extended runtime AMD wrapper for
// the `@atlassian/date-time` package. More details can be found at:
// https://www.npmjs.com/package/@atlassian/date-time
define('confluence-inline-tasks/dateutil', [
    'jquery',
    'ajs',
    'confluence/date-time',
    'confluence/position',
    'confluence-inline-tasks/util',
    'confluence-inline-tasks/date-picker1',
], function ($, AJS, dateTime, Position, InlineTasksUtil, DonnotDelete) {
    const userLocale = AJS.Meta.get('user-locale').replace('_', '-');
    const DateUtil = {
        isDateStringValid: function (dateString) {
            return !isNaN(new Date(dateString).getTime());
        },
        parseDate: dateTime.parse,
        getDateTime: function (date) {
            return dateTime.formatPlainDate(date);
        },
        getLozengeTime: function (date, locale = userLocale) {
            return dateTime.formatDate(date, locale);
        },
        getInputDate: function (date, locale = userLocale) {
            return dateTime.formatDateTimeByOptions(
                {
                    year: 'numeric',
                    month: '2-digit',
                    day: '2-digit',
                    hour: undefined,
                    minute: undefined,
                    second: undefined,
                },
                date,
                locale
            );
        },
        getPatternByLocale: function (locale = userLocale) {
            return dateTime.getDatePattern(locale);
        },

        /**
         * Convert a date object to HTML <time> tag
         * <time> tag have 2 attributes
         * - datetime: have data format as 'YYYY-MM-DD'
         * - contenteditable=false
         * @param {Date} date
         * @param {string} className class name of HTML tag
         * @returns {string} html string to represent date object
         */
        convertDateIntoHtml: function (date, className = '') {
            const tagName = '<time>';
            var $html = $('<div>');
            var $htmlInner = '';
            var dateText = DateUtil.getLozengeTime(date);
            $htmlInner = $(tagName, {
                datetime: DateUtil.getDateTime(date),
            });
            InlineTasksUtil.addContentEditableIfApplicable($htmlInner);
            className && $htmlInner.attr('class', className);
            $htmlInner.text(dateText);
            $html.append($htmlInner);

            return $html.html();
        },

        /**
         * Insert a date object into editor at current cursor
         * @param {Date} date
         * @param {string} tagName
         * @param {string} className
         * @param {string} pattern
         * @param [contentAfter]
         * @returns {*}
         */
        insertDateIntoCurrentCursor: function (date, className, contentAfter) {
            var tinymce = require('tinymce');
            var NodeUtils = require('confluence-editor/utils/tinymce-node-utils');
            var ed = tinymce.activeEditor;
            var $today = DateUtil.convertDateIntoHtml(date, className);
            var node = $($today, ed.getDoc())[0];
            var newTimeNode = NodeUtils.replaceSelection(node);

            //CONFDEV-22225: add white space between the lozenge and text caret,
            //White space should be '&nbsp;', ' ' does not work stable when inserting and removing many times
            if (contentAfter) {
                ed.execCommand('mceInsertContent', false, contentAfter, {
                    skip_undo: true,
                });
            }

            // Trigger an analytics event when the date lozenge is added.
            var contextValue = $(ed.selection.getRng(true).startContainer).closest('li[data-inline-task-id]').length
                ? 'task'
                : 'page';
            var data = { context: contextValue, trigger: 'character' };
            AJS.trigger('analyticsEvent', {
                name: 'confluence-spaces.date.added',
                data: data,
            });

            return newTimeNode;
        },

        /**
         * Get Date object by predicting partial date string
         * @param {string} partial
         * @param {date} date to predict user intent date input
         * @param {string} locale user locale
         * @returns {Date | null} date object for pre-populating date picker
         */
        getDateByPartialString: function (partialString, date = new Date(), locale = userLocale) {
            if (partialString === '') {
                return date;
            }
            const partials = partialString.match(/[\d]+/g);
            if (!partials) {
                return null;
            }
            let year;
            let month;
            let day;
            const dateCloned = new Date(date);
            const formatter = new Intl.DateTimeFormat(locale, {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
            });

            const parts = formatter.formatToParts(dateCloned);
            parts
                .filter((part) => part.type !== 'literal')
                .forEach((part, index) => {
                    if (part.type === 'year') {
                        year = dateCloned.getFullYear().toString();
                        if (partials[index]) {
                            year = year.substring(0, year.length - partials[index].length) + partials[index];
                        }
                    } else if (part.type === 'month') {
                        month = (parseInt(partials[index]) || parseInt(part.value)) - 1;
                    } else if (part.type === 'day') {
                        day = partials[index] || part.value;
                    }
                });

            dateCloned.setFullYear(year, month, day);

            const dateString = `${year}-${(month + 1).toString().padStart(2, 0)}-${day.toString().padStart(2, 0)}`;
            if (DateUtil.isDateStringValid(dateString)) {
                return dateCloned;
            }
            return null;
        },

        datepicker: {
            POSITION_ABOVE: true,
            POSITION_BELOW: false,
            POSITION_RIGHT: true,
            POSITION_LEFT: false,

            dropDownStillFitsVertically: function (preferredVerticalPosition, spaceAvailable, heightRequired) {
                if (preferredVerticalPosition === this.POSITION_ABOVE) {
                    return spaceAvailable.above >= heightRequired;
                }
                return spaceAvailable.below >= heightRequired;
            },

            dropDownStillFitsHorizontally: function (preferredHorizontalPosition, spaceAvailable, widthRequired) {
                if (preferredHorizontalPosition === this.POSITION_RIGHT) {
                    return spaceAvailable.right >= widthRequired;
                }
                return spaceAvailable.left >= widthRequired;
            },

            getPreferredHorizontalPosition: function (spaceAvailable, widthRequired) {
                if (spaceAvailable.right >= widthRequired) {
                    return this.POSITION_RIGHT;
                } else if (spaceAvailable.left >= widthRequired) {
                    return this.POSITION_LEFT;
                }

                return spaceAvailable.right > spaceAvailable.left ? this.POSITION_RIGHT : this.POSITION_LEFT;
            },

            getPreferredVerticalPosition: function (spaceAvailable, heightRequired) {
                if (spaceAvailable.below >= heightRequired) {
                    return this.POSITION_BELOW;
                } else if (spaceAvailable.above >= heightRequired) {
                    return this.POSITION_ABOVE;
                }
                // Not enough space - so use biggest space
                return spaceAvailable.below > spaceAvailable.above ? this.POSITION_BELOW : this.POSITION_ABOVE;
            },

            calculateDatepickerPosition: function ($attachTo) {
                var thiz = this;
                return function (popup, targetPosition) {
                    var verticalSpaceAvailable;
                    var horizontalSpaceAvailable;
                    var top;
                    var left;
                    var offset;
                    var gapForArrowY = 16;
                    var gapForArrowX = 0;
                    var ddHeight;
                    var ddWidth;
                    var heightRequired;
                    var widthRequired;
                    var AtlassianEditor = require('confluence-editor/editor/atlassian-editor');
                    var AtlassianEditorContent = require('confluence-editor/editor/atlassian-editor-content');
                    var iframe = AtlassianEditor.Rte.getEditorFrame();

                    if (Position.spaceLeftRight === undefined) {
                        Position.spaceLeftRight = function (containerElement, element) {
                            var elementPos = element.offset().left;
                            return {
                                left: elementPos,
                                right: $(containerElement).width() - elementPos,
                            };
                        };
                    }

                    verticalSpaceAvailable = Position.spaceAboveBelow(iframe, $attachTo);
                    horizontalSpaceAvailable = Position.spaceLeftRight(iframe, $attachTo);

                    offset = AtlassianEditorContent.offset($attachTo);

                    ddHeight = popup.outerHeight(true);
                    ddWidth = popup.outerWidth(true);
                    heightRequired = ddHeight + gapForArrowY;
                    widthRequired = ddWidth + gapForArrowX;

                    var preferredVerticalPosition = thiz.POSITION_BELOW;
                    var preferredHorizontalPosition = thiz.POSITION_RIGHT;

                    if (preferredVerticalPosition) {
                        if (
                            !thiz.dropDownStillFitsVertically(
                                preferredVerticalPosition,
                                verticalSpaceAvailable,
                                heightRequired
                            )
                        ) {
                            preferredVerticalPosition = null;
                        }
                    }

                    if (!preferredVerticalPosition) {
                        preferredVerticalPosition = thiz.getPreferredVerticalPosition(
                            verticalSpaceAvailable,
                            heightRequired
                        );
                    }

                    if (preferredHorizontalPosition) {
                        if (
                            !thiz.dropDownStillFitsHorizontally(
                                preferredHorizontalPosition,
                                horizontalSpaceAvailable,
                                heightRequired
                            )
                        ) {
                            preferredHorizontalPosition = null;
                        }
                    }
                    preferredHorizontalPosition = thiz.getPreferredHorizontalPosition(
                        horizontalSpaceAvailable,
                        widthRequired
                    );

                    if (preferredVerticalPosition === thiz.POSITION_ABOVE) {
                        top = offset.top - ddHeight - gapForArrowY;
                    } else {
                        top = offset.top + $attachTo.height() + gapForArrowY;
                    }

                    if (preferredHorizontalPosition === thiz.POSITION_RIGHT) {
                        left = offset.left - gapForArrowX;
                    } else {
                        left = offset.left + $attachTo.outerWidth() - ddWidth;
                    }

                    var popupCss = {
                        top: top,
                        left: left,
                    };

                    var arrowCssLeft =
                        preferredHorizontalPosition === thiz.POSITION_RIGHT
                            ? $attachTo.outerWidth() / 2
                            : ddWidth - $attachTo.outerWidth() / 2;

                    // CONFDEV-23097 - If that arrow is outside of the AUI inline dialog due to the autocomplete
                    // line spilling over onto the next line, force it to be on the left of the dialog.
                    if (arrowCssLeft < 0 || arrowCssLeft > ddWidth) {
                        arrowCssLeft = ddWidth / 10;
                    }

                    var arrowCss = {
                        left: arrowCssLeft,
                        top: preferredVerticalPosition === thiz.POSITION_ABOVE ? ddHeight : -gapForArrowY / 2,
                    };

                    return {
                        displayAbove: preferredVerticalPosition,
                        popupCss: popupCss,
                        arrowCss: arrowCss,
                    };
                };
            },

            /**
             * Depending whether a date is set, will return different hint text for date picker
             * @param {boolean} isSetDueDate
             * @returns {String}
             */
            getHintTextOfDatePicker: function (isSetDueDate) {
                return isSetDueDate
                    ? AJS.I18n.getText('inline-tasks.editor.datepicker.hint.selectduedate')
                    : AJS.I18n.getText('inline-tasks.editor.datepicker.hint.selectdate');
            },

            /**
             * Create a date-picker object corresponding with DOM node
             * @param options has following options
             * - $attachTo: is a jQuery object which hidden input is created automatically and appended before it
             * - $positionTo: is a jQuery object which date-picker will use to re-position to follow
             * - startDate: is a date object will be set when date-picker appears first time
             * - onSelect: is a callback function when a date is selected in date-picker
             * - isSetDueDate: will decide how hint copy appears at top of date-picker
             * @returns {{die: die, show: show, hide: hide, getContainer: getContainer, setDate: setDate, placeDatePicker: placeDatePicker}}
             */
            create: function (options) {
                var $attachTo = options.$attachTo;
                var $positionTo = options.$positionTo;
                var startDate = options.startDate;
                var onSelect = options.onSelect;
                var isSetDueDate = options.isSetDueDate;

                var $datePickerControl = $('<input>')
                    .attr('type', 'date')
                    .addClass('aui-date-picker')
                    .css('display', 'none');
                var picker = $datePickerControl.inlineTasksDatePicker1({
                    overrideBrowserDefault: true,
                    dateFormat: 'yy-mm-dd',
                    languageCode: AJS.Meta.get('user-locale') && AJS.Meta.get('user-locale').split('_')[0],
                    position: this.calculateDatepickerPosition($positionTo),
                    onSelect: function (date) {
                        onSelect(date);
                        killDatePicker();
                    },
                    hint: this.getHintTextOfDatePicker(isSetDueDate),
                });

                var killDatePicker = function () {
                    picker && picker.destroyPolyfill && picker.destroyPolyfill();
                    $datePickerControl && $datePickerControl.remove();
                    //remove unused dialog from DOM tree
                    $('.aui-datepicker-dialog').remove();

                    picker = null;
                    $datePickerControl = null;
                };

                picker.show();
                picker.setDate(startDate);

                //for testable
                $('.aui-datepicker-dialog .aui-datepicker-hint').addClass(isSetDueDate ? 'set-duedate' : 'set-date');

                $attachTo.before($datePickerControl);

                return {
                    die: function () {
                        killDatePicker();
                    },
                    show: function () {
                        picker.show();
                    },
                    hide: function () {
                        picker.hide();
                    },
                    getContainer: function () {
                        return $datePickerControl;
                    },
                    setDate: function (dateText) {
                        AJS.debug('Set date: ' + dateText);
                        picker.setDate(DateUtil.getDateByPartialString(dateText));
                    },
                    placeDatePicker: function () {
                        var currentHash = AJS.InlineDialog.current;
                        var popup = currentHash && currentHash.popup;
                        popup && popup.refresh();
                    },
                };
            },
        },
    };

    return DateUtil;
});

require('confluence/module-exporter').exportModuleAsGlobal(
    'confluence-inline-tasks/dateutil',
    'Confluence.InlineTasks.DateUtil'
);
