(function ($) {
    'use strict';

    var Confluence = window.Confluence || {};

    // keeping it camelCase for consistency within confluence files for easier search and replace
    var synchronyId = 'confluence.date-autocomplete-manager.plugin';

    Confluence.Editor = Confluence.Editor || {};
    Confluence.Editor.AutoComplete = Confluence.Editor.AutoComplete || {};

    var KEY = Confluence.InlineTasks.Util.KEY;
    //cache namespace for shorter
    var DateUtil = Confluence.InlineTasks.DateUtil;
    var AutoComplete = Confluence.Editor.AutoComplete;

    AutoComplete.SETTING_DEFAULT = {
        leadingChar: '', //leading character inserting before autocomplete
        isDataValid: function (text) {
            return false;
        }, //function to check a string is valid or not,
        backWords: 0,
        onBeforeDie: function () {},
        onAfterStart: function () {},
        onScroll: function () {},
    };

    /**
     * This class is simplified for tinyMce-autocomplete-manager.js
     * Because tinyMce-autocomplete-manager.js tightly go along with AJS.inputDrivenDropdown
     */
    AutoComplete.Manager = function (options) {
        this.ed = AJS.Rte.getEditor();
        this.settings = $.extend({}, AutoComplete.SETTING_DEFAULT, options);
        this.log = AJS.debug;

        //each instance of AutoComplete.Manager have a 'control' property.
        //at the same time, in editor should have only active control across all instances of AutoComplete.Manager
        //Therefore, you can consider 'control' as static property of AutoComplete.Manager
        this.control = null;
    };

    let insertedFromMenuAt = null;
    AutoComplete.Manager.prototype = {
        /**
         * Init Autocomplete Control object and bind some default events for it
         * @returns {Boolean} true when creating Confluence.Editor.Autocompleter.Control object successfully,
         * false if it cannot.
         */
        start: function (customOption) {
            var settings = $.extend({}, this.settings, customOption);
            this.control = Confluence.Editor.Autocompleter.Control(this.ed, settings);

            if (!this.control) {
                return false;
            }

            insertedFromMenuAt = customOption.insertedFromMenu ? new Date().getTime() : null;

            AJS.trigger('synchrony.stop', { id: synchronyId });
            this.log('startAutoComplete - Started');

            this.attachEventsToControl();
            settings.onAfterStart &&
                settings.onAfterStart({
                    date: customOption.date,
                    isTriggerFromOrphan: customOption.isTriggerFromOrphan ? true : false,
                });

            return true;
        },

        attachEventsToControl: function () {
            var that = this;
            var autoCompleteControl = that.control;
            var log = that.log;
            var settings = that.settings;

            //inject onBeforeDie into this.control.die
            if (typeof this.settings.onBeforeDie === 'function') {
                var oldDie = this.control.die;
                var isDead = false;
                this.control.die = function () {
                    // CONFSRVDEV-11333 - in some scenarios (inconsistently based on race conditions) this function
                    // is called more then once - we need to block this possibility
                    if (isDead) {
                        return;
                    }
                    isDead = true;
                    that.settings.onBeforeDie.apply(that.control, arguments);

                    // If the date was inserted when the autocomplete
                    // control was created, preserve this text on death.
                    if (that.dateInserted()) {
                        oldDie.apply(that.control, [true]);
                        AJS.trigger('analyticsEvent', {
                            name: 'confluence-spaces.tasks.daterecognition.killed',
                        });
                    } else {
                        oldDie.apply(that.control, arguments);
                    }
                };
            }

            var onScroll = _.bind(_.throttle(that.settings.onScroll, 40), this);

            var eventDefaultForAutoCompleteControl = {
                onBeforeKey: function (e, text) {
                    //if user press anything, except ESCAPE & ENTER
                    //it means that user tries to edit it
                    if (e.keyCode !== KEY.ESCAPE && e.keyCode !== KEY.ENTER) {
                        that.control && that.control.getContainer().addClass('isDirty');
                    }

                    //Suppress 3 keys: DOWN, UP and Enter when keydown
                    if (e.keyCode === KEY.DOWN || e.keyCode === KEY.UP || e.keyCode === KEY.ENTER) {
                        tinymce.dom.Event.cancel(e);
                        return false;
                    }

                    // User has key-downed backspace but text is *already* blank - close autocomplete.
                    if (e.keyCode === KEY.ESCAPE || e.keyCode === KEY.TAB || (e.keyCode === KEY.BACKSPACE && !text)) {
                        log('autoCompleteControl.onBeforeKey - killing autoCompleteControl');
                        this.die(e.keyCode === KEY.BACKSPACE);
                        return false;
                    }

                    return true;
                },

                onKeyPress: function (e, text) {
                    var ch = e.which;

                    if (e.keyCode === KEY.ENTER) {
                        tinymce.dom.Event.cancel(e);
                        return false;
                    }

                    var character = String.fromCharCode(ch); // charCode back to '@'
                    if (this.settings.endChars.includes(character)) {
                        log('autoCompleteControl.onKeyPress - caught autocomplete-closing char - character');
                        autoCompleteControl.die();
                    }

                    return true;
                },

                onAfterKey: function (e, text) {
                    //prevent modifying trigger text
                    var $triggerSpan = $('#autocomplete-trigger', that.control.getContainer());
                    var triggerText = $triggerSpan.text();
                    if ($triggerSpan.length > 0 && triggerText !== that.settings.leadingChar) {
                        log('after', 'dying because of: trigger text is modified');
                        that.reset();
                        return false;
                    }

                    // if the autocomplete was activated from menu using Enter,
                    // we might get and Enter keyup afterwards (keydown on manu -> change focus -> keyup here )
                    // we should ignore it, limiting the risk by limiting this to 0.5 sec and only once
                    // after each autocomplete is open
                    if (
                        insertedFromMenuAt &&
                        e.code === 'Enter' &&
                        e.type === 'keyup' &&
                        e.timeStamp - insertedFromMenuAt < 500
                    ) {
                        insertedFromMenuAt = null;
                        return;
                    }

                    if (e.keyCode === KEY.ENTER) {
                        if (settings.isDataValid(text)) {
                            autoCompleteControl.update(text);
                        } else {
                            log('autoCompleteControl.onAfterKey - closing autocomplete due to invalid data - ' + text);
                            autoCompleteControl.die(false); //save trigger chars
                        }

                        //CONFDEV-23090: should return false to cancel the Enter key
                        return false;
                    }

                    if (that.dateInserted() && text.length > 10 && !DateUtil.getDateByPartialString(text)) {
                        that.reset();
                        return false;
                    }

                    return true;
                },

                onDeath: function () {
                    that.reset();
                },

                onScroll: function () {
                    if (!that.isAlive()) {
                        return;
                    }
                    onScroll();
                },
            };

            $.extend(autoCompleteControl, eventDefaultForAutoCompleteControl);
        },

        isAlive: function () {
            return this.control && !this.control.dying;
        },

        /**
         * Reset or destroy Autocomplete Control object
         */
        reset: function () {
            if (!this.control) {
                return;
            }

            this.control.die();
            this.control = null;

            AJS.trigger('synchrony.start', { id: synchronyId });
        },

        dateInserted: function (dateInserted) {
            if (dateInserted === true) {
                return this.control.getContainer().addClass('hasDateInserted');
            } else if (dateInserted === false) {
                return this.control.getContainer().removeClass('hasDateInserted');
            } else {
                return this.control.getContainer().hasClass('hasDateInserted');
            }
        },

        dirty: function (makeDirty) {
            if (makeDirty === true) {
                return this.control.getContainer().addClass('isDirty');
            } else {
                return this.control.getContainer().hasClass('isDirty');
            }
        },

        reattach: function () {
            var orphan = Confluence.Editor.Autocompleter.Control.removeOrphanedControl();
            if (orphan && orphan.leadingChar === this.settings.leadingChar) {
                this.reset();
                return this.start({
                    backWords: 1,
                    isTriggerFromOrphan: true,
                });
            }
            return false;
        },
    };
})(AJS.$);
