define('confluence-inline-tasks/inline-tasks-status', [
    'jquery',
    'ajs',
    'underscore',
    'confluence/message-controller',
    'confluence/dark-features',
], function ($, AJS, _, MessageController, DarkFeatures) {
    'use strict';

    var debounceIsOn = false;
    var taskListQueue = [];
    var pageBatches = new Map();
    var updateRunning = false;
    var pageUnloading = false;
    var flushTimeMillis = 1000;
    var debounceFn = _.debounce(() => {
        debounceIsOn = false;
        batchTaskUpdate();
    }, flushTimeMillis);

    return {
        onTaskClick: function (e) {
            var $el = $(e.target);
            $el.toggleClass('checked');
            syncInlineTaskListItemA11YAttributes($el);
            var status = $el.hasClass('checked') ? 'CHECKED' : 'UNCHECKED';
            var taskId = $el.data('inline-task-id');
            var contentId = $el.closest('ul').attr('data-inline-tasks-content-id') || AJS.params.pageId;

            if (taskListQueue.indexOf(taskId) === -1) {
                taskListQueue.push(taskId);
            }
            AJS.trigger('inline-tasks.status-update.start', {
                status: status,
                taskId: taskId,
                contentId: contentId,
                taskListQueue: taskListQueue,
            });

            if (DarkFeatures.isEnabled('frontend.batch.tasks.disabled')) {
                console.log(
                    'frontend.batch.tasks.disabled flag is true. Sending single task change requests. It can impact performance!'
                );
                $el.prop('disabled', true);
                var $parent = $el.closest('tr');
                $parent.attr('aria-disabled', true);
                var url = AJS.contextPath() + '/rest/inlinetasks/1/task/' + contentId + '/' + taskId + '/';
                singleTaskUpdate(url, status, $el, taskId, contentId);
            } else {
                addTask(contentId, taskId, status, $el);
                if (!debounceIsOn) {
                    debounceIsOn = true;
                    batchTaskUpdate();
                }
                debounceFn();
            }
        },

        /* Determines the location of the element - returns a string
          specifying where it is in a task, task report, my tasks page or just the page.
          Used for analytics only. */
        findContext: function ($el) {
            return findContextImpl($el);
        },

        beforeUnload: function () {
            pageUnloading = true;
            if (pageBatches.size > 0) {
                console.log(`Sending remaining inline task batches. Number of batches: ${pageBatches.size}`);
            }
            // send remaining batches all at once
            while (batchTaskUpdate()) {}
            updateRunning = false;
            debounceIsOn = false;
        },
    };

    function addTask(contentId, taskId, status) {
        if (!pageBatches.has(contentId)) {
            pageBatches.set(contentId, {});
        }
        pageBatches.get(contentId)[taskId] = status;
    }

    function takeBatch() {
        for (var [contentId, batch] of pageBatches) {
            pageBatches.delete(contentId);
            return [contentId, batch];
        }
        return [undefined, undefined];
    }

    function batchTaskUpdate() {
        if (updateRunning && !pageUnloading) {
            return false;
        }
        var [contentId, batch] = takeBatch();
        if (batch === undefined) {
            return false;
        }
        console.log(`Sending batch of tasks for content id ${contentId}: ${JSON.stringify(batch)}`);
        updateRunning = true;
        var url = AJS.contextPath() + '/rest/inlinetasks/1/task/' + contentId + '/';
        $.ajax({
            type: 'POST',
            url: url,
            data: JSON.stringify({
                taskStatuses: batch,
                trigger: 'VIEW_PAGE',
            }),
            dataType: 'json',
            contentType: 'application/json',
            timeout: 30000,
            error: function (jqXHR, statusText, error) {
                if (pageUnloading || statusText === 'timeout') {
                    // CONFDEV-10030 - it's likely that the request still completes successfully despite a timeout,
                    // or when the user navigates away from the page. So we suppress the error in both of these cases.
                    return;
                }
                AJS.logError(
                    'Inline Task updates for ' +
                        contentId +
                        ' were not persisted because of ' +
                        error +
                        ' (status: ' +
                        statusText +
                        ')'
                );

                for (var taskIdToRevert of Object.keys(batch)) {
                    var taskId = Number(taskIdToRevert);
                    var $el = $(
                        "[data-inline-tasks-content-id='" + contentId + "']>[data-inline-task-id='" + taskId + "']"
                    );
                    $el.toggleClass('checked');
                    syncInlineTaskListItemA11YAttributes($el);
                }
                MessageController.showError(
                    MessageController.parseError(jqXHR, AJS.I18n.getText('inline-tasks.notice.unavailable')),
                    MessageController.Location.FLAG
                );
            },
            success: function () {
                for (var taskIdString of Object.keys(batch)) {
                    var taskId = Number(taskIdString);
                    var $el = $(
                        "[data-inline-tasks-content-id='" + contentId + "']>[data-inline-task-id='" + taskId + "']"
                    );
                    var data = {
                        dueDate: findElementInTask($el, 'time').attr('datetime'),
                        completionDate: formatDate(new Date()),
                        mode: 'view',
                        assigneeUsername: findElementInTask($el, '.user-mention').attr('data-username'),
                        context: findContextImpl($el),
                    };

                    if (batch[taskId] === 'CHECKED') {
                        AJS.trigger('analyticsEvent', {
                            name: 'confluence-spaces.tasks.completed',
                            data: data,
                        });
                    }
                }
            },
        }).always(function () {
            updateRunning = false;
            for (var taskIdString of Object.keys(batch)) {
                var taskId = Number(taskIdString);
                var $el = $(
                    '[data-inline-tasks-content-id="' + contentId + '"]>[data-inline-task-id=\'' + taskId + "']"
                );

                var $parent = $el.closest('tr');
                $parent.attr('aria-disabled', false);
                taskListQueue.splice(taskListQueue.indexOf(taskId), 1);
                AJS.trigger('inline-tasks.status-update.complete', {
                    status: batch[taskId],
                    taskId: taskId,
                    contentId: contentId,
                    taskListQueue: taskListQueue,
                });
            }
            if (!debounceIsOn) {
                batchTaskUpdate();
            }
        });
        return true;
    }

    function findContextImpl($el) {
        var context = 'page';

        if ($el.closest('table.tasks-report').length) {
            context = 'report';
        } else if ($el.closest('#task-container').length) {
            context = 'mytasks';
        } else if ($el.closest('ul.inline-task-list').length) {
            context = 'task';
        }

        return context;
    }

    /* Given a task li element, searches using a particular selector for the first element
      that matches that selector. Will not return matching selectors in nested tasks. Returns
      an empty jquery object if no matches are found. */
    function findElementInTask($task, selector) {
        var thisId = $task.attr('data-inline-task-id');
        var selectedItem = $task.find(selector).first();
        if (selectedItem.closest('li').attr('data-inline-task-id') === thisId) {
            return selectedItem;
        } else {
            return $();
        }
    }

    function formatDate(date) {
        var year = '' + date.getFullYear();
        var month = '' + (date.getMonth() + 1);
        var day = '' + date.getDate();

        if (month.length < 2) {
            month = '0' + month;
        }

        if (day.length < 2) {
            day = '0' + day;
        }

        return [year, month, day].join('-');
    }

    function syncInlineTaskListItemA11YAttributes($el) {
        $el.attr('aria-checked', $el.hasClass('checked') ? 'true' : 'false');
    }

    function singleTaskUpdate(url, status, $el, taskId, contentId) {
        $.ajax({
            type: 'POST',
            url: url,
            data: JSON.stringify({
                status: status,
                trigger: 'VIEW_PAGE',
            }),
            dataType: 'json',
            contentType: 'application/json',
            timeout: 30000,
            error: function (jqXHR, statusText, error) {
                if (pageUnloading || statusText === 'timeout') {
                    // CONFDEV-10030 - it's likely that the request still completes successfully despite a timeout,
                    // or when the user navigates away from the page. So we suppress the error in both of these cases.
                    return;
                }
                AJS.logError(
                    'Inline Task #' +
                        taskId +
                        ' was not persisted to ' +
                        status +
                        ' because of ' +
                        error +
                        ' (status: ' +
                        statusText +
                        ')'
                );
                var MessageController = require('confluence/message-controller');
                $el.toggleClass('checked');
                syncInlineTaskListItemA11YAttributes($el);
                MessageController.showError(
                    MessageController.parseError(jqXHR, AJS.I18n.getText('inline-tasks.notice.unavailable')),
                    MessageController.Location.FLAG
                );
            },
            success: function () {
                var data = {
                    dueDate: findElementInTask($el, 'time').attr('datetime'),
                    completionDate: formatDate(new Date()),
                    mode: 'view',
                    assigneeUsername: findElementInTask($el, '.user-mention').attr('data-username'),
                    context: findContextImpl($el),
                };

                if (status === 'CHECKED') {
                    AJS.trigger('analyticsEvent', {
                        name: 'confluence-spaces.tasks.completed',
                        data: data,
                    });
                }
            },
        }).always(function () {
            $el.prop('disabled', false);
            var $parent = $el.closest('tr');
            $parent.attr('aria-disabled', false);
            taskListQueue.splice(taskListQueue.indexOf(taskId), 1);
            AJS.trigger('inline-tasks.status-update.complete', {
                status: status,
                taskId: taskId,
                contentId: contentId,
                taskListQueue: taskListQueue,
            });
        });
    }
});
