(function ($) {
    var inlineTasksStatus = require('confluence-inline-tasks/inline-tasks-status');

    /* This is very dependant on the images used and the padding, etc */
    function isInCheckArea(e) {
        var offsetX;
        var offsetY;

        // offsetX and offsetY are non-standard
        if (e.offsetX === undefined) {
            var offsetLeft = 0;
            var offsetTop = 0;
            var el = e.target;
            var container;

            do {
                if (el.scrollTop !== 0 || el.scrollLeft !== 0) {
                    container = el;
                }
                offsetLeft += el.offsetLeft;
                offsetTop += el.offsetTop;

                el = el.offsetParent;
            } while (el && el !== el.offsetParent); // el.offsetParent shouldn't ever return self, but it doesn't
            // cost us much to have a second guard against infinite looping

            offsetX = e.pageX + (container ? container.scrollLeft : 0) - offsetLeft;
            offsetY = e.pageY + (container ? container.scrollTop : 0) - offsetTop;
        } else {
            offsetX = e.offsetX;
            offsetY = e.offsetY;
        }

        return offsetX >= 3 && offsetX <= 14 && offsetY >= 3 && offsetY <= 14;
    }

    /**
     * Determines if the check area, an <li> element with role="checkbox",
     * is focused when triggered with Space button in the Windows Narrator.
     * This is necessary because the Narrator triggers a JavaScript click event instead of
     * a keypress while pressing the space button, unlike other screen readers like VoiceOver.
     * The method ensures the element:
     *  1. Matches the :focus-visible pseudo-class (indicating the focus is visible).
     *  2. Has offsetX and offsetY equal to 0, confirming that the event was not triggered by a user's mouse click
     *     and to avoid click outside of the check area while element focused.
     */
    function isCheckAreaFocused(e) {
        return e.target.matches(':focus-visible') && !e.offsetX && !e.offsetY;
    }

    // Ignores the bubbled event on any parent inline tasks
    function isActualTarget(e) {
        return e.currentTarget === e.target;
    }

    $(window).bind('beforeunload', function () {
        inlineTasksStatus.beforeUnload();
    });

    define('confluence-inline-tasks/adapter', function () {
        'use strict';
        return {
            clickInlineTaskUIItem: function (taskId, contentId) {
                var $el = $(
                    'ul[data-inline-tasks-content-id="' + contentId + '"] li[data-inline-task-id="' + taskId + '"]'
                );
                if ($el.length) {
                    inlineTasksStatus.onTaskClick({ target: $el[0] });
                }
            },
        };
    });

    $(document).delegate('ul.inline-task-list > li[data-inline-task-id]', {
        click: function (e) {
            if (isActualTarget(e) && (isInCheckArea(e) || isCheckAreaFocused(e))) {
                inlineTasksStatus.onTaskClick(e);
            }
        },
        keypress: function (e) {
            if (e.key === ' ' && isActualTarget(e)) {
                //space bar
                e.preventDefault();
                inlineTasksStatus.onTaskClick(e);
            }
        },
        mousemove: function (e) {
            if (isActualTarget(e)) {
                if (isInCheckArea(e)) {
                    $(e.target).addClass('hover');
                } else {
                    $(e.target).removeClass('hover');
                }
            }
        },
        mouseout: function (e) {
            if (isActualTarget(e)) {
                $(e.target).removeClass('hover');
            }
        },
        mousedown: function (e) {
            if (isActualTarget(e) && isInCheckArea(e)) {
                $(e.target).addClass('task-active');
            }
        },
        mouseup: function (e) {
            if (isActualTarget(e) && isInCheckArea(e)) {
                $(e.target).removeClass('task-active');
            }
        },
    });

    $(document).tooltip({
        title: function () {
            return AJS.I18n.getText('inline-tasks.date.duesoon');
        },
        live: 'ul.inline-task-list li:not(.checked) time.date-upcoming',
    });

    $(document).tooltip({
        title: function () {
            return AJS.I18n.getText('inline-tasks.date.overdue');
        },
        live: 'ul.inline-task-list li:not(.checked) time.date-past',
    });

    $(document).tooltip({
        title: function () {
            return AJS.I18n.getText(
                'com.atlassian.confluence.plugins.confluence-inline-tasks.tasks-report.no.complete.date.tooltip'
            );
        },
        live: 'span.emptycompletedate',
    });

    $(document).on('click', 'time', function () {
        var $this = $(this);

        // Fire an analytics event when a date lozenge is clicked.
        var data = {
            date: $this.attr('datetime'),
            mode: 'view',
            context: inlineTasksStatus.findContext($this),
        };
        AJS.trigger('analyticsEvent', {
            name: 'confluence-spaces.date.clicked',
            data: data,
        });
    });

    AJS.toInit(function () {
        addInlineTaskListA11YAttributes();
        if (AJS.Meta.get('access-mode') === 'READ_ONLY') {
            $('[data-inline-task-id]').addClass('disabled');
            $('[data-inline-task-id]').prop('disabled', true);
        }
    });

    function addInlineTaskListA11YAttributes() {
        $('.inline-task-list').attr('role', 'group');
        $('.inline-task-list [data-inline-task-id]:not(.checked)')
            .attr('role', 'checkbox')
            .attr('aria-checked', 'false')
            .attr('tabindex', 0);
        $('.inline-task-list [data-inline-task-id].checked')
            .attr('role', 'checkbox')
            .attr('aria-checked', 'true')
            .attr('tabindex', 0);
    }
})(AJS.$);
