/**
 * @module confluence-editor/files/file-dialog/file-dialog-view
 */
define('confluence-editor/files/file-dialog/file-dialog-view', [
    'backbone',
    'ajs',
    'confluence/dialog',
    'confluence/templates',
    'underscore',
    'jquery',
    'document',
    'window'
], function(
    Backbone,
    AJS,
    confluenceDialog,
    Templates,
    _,
    $,
    document,
    window
) {
    'use strict';

    /**
     * Invokes all functions in the callbacks array with all values in the items array.
     * @param {Function[]} callbacks an array of functions
     * @param {Array} items an array of values to pass to each callback function
     */
    function run(callbacks, items) {
        items.forEach(function(item) {
            callbacks.forEach(function(cb) {
                cb(item);
            });
        });
    }

    /**
     * Due to an unfortunate reality of changeable loading orders and using flat arrays
     * as a callback registration system, we need to turn a simple array in to a more
     * observable one, so that whenever something gets added to it, we can react accordingly.
     * @param {Array} arrayLike something that implements a push and forEach method.
     * @return {Array} the same object with a #subscribe and #unsubscribe method.
     *         functions passed to #subscribe will be immediately run for each item
     *         in the array, and whenever a new item is added to the array.
     *         All subscribers are removed when #unsubscribe is called.
     */
    function hackishlyObservableArray(arrayLike) {
        var oldPush;
        var subscribers;
        if (!arrayLike || (arrayLike.subscribe && arrayLike.unsubscribe)) {
            return arrayLike;
        }
        oldPush = arrayLike.push;
        subscribers = [];

        Object.assign(arrayLike, {
            push: function() {
                var args = Array.prototype.slice.apply(arguments);
                oldPush.apply(arrayLike, args);
                run(subscribers, args);
            },
            subscribe: function(sub) {
                subscribers.push(sub);
                run([sub], arrayLike);
            },
            unsubscribe: function() {
                subscribers.length = 0;
            }
        });

        return arrayLike;
    }

    return Backbone.View.extend({

        dialogId: 'insert-image-dialog',
        ESC_KEY_CODE: 27,
        width: 840, // width of dialog
        height: 530, // height of dialog

        /**
         * Opens the image dialog. If the options include an imageProperties object, the image represented by the
         * properties will be edited. If not, an insert dialog is shown.
         *
         * @param {Object} options for opening the dialog.
         * @param {Function} [options.submitCallback] called when the dialog form is submitted
         * @param {Function} [options.cancelCallback] called when the dialog is cancelled
         * @param {Array<Function>} [options.beforeShowListeners] a queue of callbacks to be invoked whenever the dialog is shown.
         *        Functions added to this array will be immediately invoked if the dialog is already visible.
         *        Note the parameter name is historical and does not match the intended behaviour.
         * @param {Array<{createPanel: function(context)}>} [options.panelComponents] an array of objects that implement
         *        a createPanel function. These will be invoked when the dialog is created.
         */
        initialize: function(options) {
            this.urlExternalImg = '';
            this.selectItems = [];

            this.submitCallback = options.submitCallback;
            this.cancelCallback = options.cancelCallback;
            this.beforeShowListeners = hackishlyObservableArray(options.beforeShowListeners || []);
            this.panelComponents = options.panelComponents;
        },

        render: function() {
            this._createDialog();
            this.clearSelection();

            // Applies key binding to a particular image container, i.e. for fancybox navigation.
            $(document).on('keydown.insert-image', _.bind(this._onNavigationByKey, this));
            return this;
        },

        /**
         * The main task of this method is to create object dialog from JS.Dialog
         * @returns {AJS.Dialog}
         * @private
         */
        _createDialog: function() {
            var dialog = new (confluenceDialog.confluenceDialog || confluenceDialog)(this.width, this.height, this.dialogId);
            var textDialogTitle = AJS.I18n.getText('file.browser.insert.title');
            var textSubmit = AJS.I18n.getText('image.browser.insert.button');

            this.dialog = dialog;

            dialog.addHeader(textDialogTitle);
            dialog.addButton(textSubmit, _.bind(this._submitDialog, this), 'insert');

            // Close Button
            dialog.addCancel(AJS.I18n.getText('close.name'), _.bind(this._killDialog, this));

            this.el = dialog.popup.element;
            this.$el = $(this.el);
            this.baseElement = this.el;// alias for el
            this.$el.attr('data-tab-default', '0');

            this.$insertButton = this.$el.find('.dialog-button-panel .insert');

            // CONFDEV-12853: Add help link via prepend() instead of append() to prevent FF display issue
            $('#' + this.dialogId + ' .dialog-components .dialog-title').prepend(Templates.File.helpLink());

            this.$el.find('.dialog-button-panel')
                .append($('<div></div>').addClass('dialog-tip').html(AJS.I18n.getText('insert.image.did.you.know')));

            this._createPanels();

            // Ensure all listeners are executed now and until the dialog is closed.
            this.beforeShowListeners.subscribe(function(fn) {
                if (typeof fn === 'function') {
                    fn();
                }
            });
            AJS.debug(this.beforeShowListeners.length + ' beforeShow listeners registered.');

            // Handle pressing ESC to close dialog
            // Because AJS.popup handles ESC keys and do not fire "hide.dialog" event for AJS.Dialog,
            // so we need a another treatment for ESC to remove dialog.
            $(document).on('hideLayer', _.bind(function(e, layerType, popup) {
                if (layerType !== 'popup' || popup !== dialog.popup) {
                    return;
                }
                popup.remove();
                this.teardown();
                if (typeof this.cancelCallback === 'function') {
                    this.cancelCallback();
                }
            }, this));

            AJS.bind('remove.dialog', _.bind(function(e, data) {
                if (data.dialog.id === this.dialog.id) {
                    this.teardown();
                }
            }, this));

            dialog.show();

            return dialog;
        },

        /**
         * Initialize collection of panels from "panelComponents"
         * @private
         */
        _createPanels: function() {
            var that = this;

            // Construct panels
            $.each(this.panelComponents, function() {
                if (this && typeof this.createPanel === 'function') {
                    this.createPanel(that);
                }
            });
        },

        teardown: function() {
            $(document).unbind('.insert-image');
            // remove some orphan tooltips
            $('.tipsy').remove();
            this.undelegateEvents();
            this.beforeShowListeners.unsubscribe();
        },

        /**
         * Close dialog when click cancel button
         * @private
         */
        _killDialog: function() {
            this.dialog.remove();
            this.clearSelection();

            if (typeof this.cancelCallback === 'function') {
                this.cancelCallback();
            }
        },

        /**
         * Close dialog when click submit button
         * @private
         */
        _submitDialog: function() {
            var placeholderRequest = {
                url: this.urlExternalImg,
                // For pages and blogs this is their own pageId. For comments, pageId is the page they are on.
                // For drafts it is contentId.
                contentId: AJS.Meta.get('attachment-source-content-id'),
                selectItems: this.selectItems
            };

            this.dialog.remove();

            if (typeof this.submitCallback === 'function') {
                this.submitCallback(placeholderRequest);
            }
        },

        /**
         * Add file item mode to select files collection
         * @param items if items is object, add file items to select files array.
         * if items is string, set the value to "urlExternalImg" property
         */
        setSelectItems: function(items) {
            if (typeof items === 'string') {
                this.urlExternalImg = items;
                this.selectItems = [];
            } else {
                this.urlExternalImg = '';
                this.selectItems = items;
            }

            // update insert button state in UI
            var allow = this.selectItems.length > 0 || this.urlExternalImg;
            if (this.$insertButton) {
                this.$insertButton.prop('disabled', !allow);
                this.$insertButton.attr('aria-disabled', !allow);
            }
        },

        clearSelection: function() {
            this.setSelectItems([]);
        },

        /**
         * Inserts the selected content.
         */
        insert: function() {
            if (this.$insertButton
                    && (!this.$insertButton.is(':disabled') || !this.$insertButton.attr('aria-disabled'))) {
                this.$insertButton.click();
            }
        },

        /**
         * @param {string} title  panel title
         * @param {string|Object} reference jQuery object or selector for the contents of the Panel
         * @param {string} [className] HTML class name
         * @param {string} [panelButtonId] The unique id for the panel's button.
         * @return {string} the panel id
         */
        addPanel: function(title, reference, className, panelButtonId) {
            var nextPanelId = this.dialog.getPage(0).panel.length;
            this.dialog.addPanel(title, reference, className, panelButtonId);
            return nextPanelId;
        },

        /**
         * Selects the panel by id.
         * @param {string} panelId
         */
        getPanel: function(panelId) {
            return this.dialog.getPanel(panelId);
        },

        /**
         * Handle navigation by arrow keys to choose/select file.
         * @param {Event} e event object
         */
        _onNavigationByKey: function(e) {
            // dialog is hidden when Preview is showing
            if ($('#' + this.dialog.id).is(':hidden')) {
                return;
            }

            var $bodyPanel = this.dialog.getCurrentPanel().body;
            var $listFiles = $bodyPanel.find('.attached-file');

            // empty item or input/select element is focusing, do nothing
            if ($listFiles.length === 0
                    || $(document.activeElement).is('input[type=text], select, button')) {
                return;
            }

            var VK = window.tinymce.VK;

            var findNextIndex = function(currentIndex, keyCode, totalItems) {
                var keyToDelta = {};
                keyToDelta[VK.LEFT] = -1;
                keyToDelta[VK.RIGHT] = 1;
                keyToDelta[VK.UP] = -4;
                keyToDelta[VK.DOWN] = 4;

                var delta = keyToDelta[keyCode] ? keyToDelta[keyCode] : 0;
                var nextIndex = (currentIndex + delta);
                if (nextIndex < 0 || nextIndex >= totalItems) {
                    return currentIndex;
                }
                return nextIndex;
            };

            var moveSelection = function(keyCode) {
                var $selected;
                var $next;

                // find the current one
                $selected = $listFiles.filter('.current');

                // if no current one, find first selected one
                if (!$selected.length) {
                    $selected = $listFiles.filter('.selected');
                }

                // if no selected one, find first one
                if (!$selected.length) {
                    $next = $listFiles.first();
                } else {
                    var currentIndex = $listFiles.index($selected);
                    var nextIndex = findNextIndex(currentIndex, keyCode, $listFiles.length);
                    $next = $listFiles.eq(nextIndex);
                }

                $next.parent().find('li.attached-file.current').removeClass('current');
                $next.addClass('current');

                if ($next.length) {
                    $bodyPanel.find('.scroll-wrapper').simpleScrollTo($next);
                }

                e.stopPropagation();
                return false;
            };

            switch (e.which) {
            case VK.LEFT:
            case VK.UP:
            case VK.RIGHT:
            case VK.DOWN:
                return moveSelection(e.which);

            case VK.SPACEBAR:
                $bodyPanel.find('li.attached-file.current').click();
                e.stopPropagation();
                return false;

            case VK.ENTER:
                this.insert();
                e.stopPropagation();
                return false;
            }
        }

    });
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-editor/files/file-dialog/file-dialog-view', 'Confluence.Editor.FileDialog.FileDialogView');
