/* globals define, AJS */
define("confluence-user-profile/avatar-picker/avatar-picker-panel", [
        "jquery",
        "underscore",
        "ajs",
        "confluence-user-profile/avatar-picker/image-upload-and-crop"
    ],
function(
    $,
    _,
    AJS,
    ImageUploadAndCrop
) {
    "use strict";

    function AvatarPickerPanel(opts) {
        if (!ImageUploadAndCrop.isSupported()) {
            AJS.messages.error({
                body: "This browser doesn't support AvatarPickerPanel."
            });
        }
        return this.init(opts);
    }

    AvatarPickerPanel.prototype.defaults = {
        panel: null,
        saveButton: null,
        cancelButton: null,
        onCrop: function() {},
        onExpanded: function() {}
    };

    AvatarPickerPanel.prototype.init = function(opts) {
        _.bindAll(this, "initPanelContent", "_onImageUpload", "_onImageUploadError", "_toggleSaveButtonEnabled",
            "_onUpdateDetails", "_onImageClear");
        this.options = $.extend(true, {}, this.defaults, opts);
        this._$panel = this.options.panel;
        this._$saveButton = this.options.saveButton;
        this.initPanelContent();
        this._toggleSaveButtonEnabled(false);
        this.imageUploadAndCrop = new ImageUploadAndCrop(this._$panel.find(".profile-pic-upload-and-crop-container"), {
            HiDPIMultiplier: 1, //The mask is already 2x the max size we need
            onCrop: this._onUpdateDetails,
            onImageUpload: this._onImageUpload,
            onImageUploadError: this._onImageUploadError,
            onImageClear: this._onImageClear,
            fallbackUploadOptions: {
                uploadURL: AJS.Meta.get('context-path') + '/rest/user-profile/1.0/avatar/uploadFallback',
                uploadFieldName: "upload-logo-input",
                responseHandler: function(iframeBody, uploadPromise){
                    var $iframeBody = $(iframeBody),
                        $jsonResponseField = $iframeBody.find("#json-response");

                    if ($jsonResponseField.length) {
                        var jsonResponse;

                        try {
                            jsonResponse = JSON.parse($jsonResponseField.html());
                        } catch(e) {
                            uploadPromise.reject();
                        }

                        if (jsonResponse && jsonResponse.src) {
                            uploadPromise.resolve(AJS.Meta.get('context-path') + jsonResponse.src);
                        } else {
                            uploadPromise.reject(jsonResponse && jsonResponse.errorMessages && jsonResponse.errorMessages[0]);
                        }
                    } else {
                        // See if we can parse a meaningful error out of the response.
                        // Firstly look for the main text on the 500 error page, then strip out nested exceptions which tend to make for unfriendly messages.
                        // If it can"t find the h2, it will just reject with a blank string
                        var error = $iframeBody.find(".error-image + h2").text();

                        error = error
                            .replace(/; nested exception.*$/, ".") //remove nested exceptions
                            .replace(/(\d+) bytes/, function(match, size){
                                return TextUtil.formatSizeInBytes(size);
                            }); //convert any values in bytes to the most appropriate unit

                        uploadPromise.reject(error);
                    }
                },
                cancelTrigger: this._$saveButton.add(this.options.cancelButton)
            }
        });

        this._$saveButton.click(function() {
            this.imageUploadAndCrop.crop();
        }.bind(this));

        return this;
    };

    AvatarPickerPanel.prototype.initPanelContent = function() {
        this._$panel.append(Confluence.Templates.User.Avatar.Picker.imageUploadAndCrop({
                description: AJS.I18n.getText("confluence.user.profile.avatar.picker.instructions"),
                fallbackDescription: AJS.I18n.getText("confluence.user.profile.avatar.picker.instructions")
            }));

        this._$message = this._$panel.find(".image-upload-and-crop-message");
    };

    AvatarPickerPanel.prototype._onUpdateDetails = function(croppedDataURI) {
        this.options.onCrop(croppedDataURI);
    };

    AvatarPickerPanel.prototype._onImageUpload = function() {
        this._toggleSaveButtonEnabled(true);
        this.hideMessage();
    };

    AvatarPickerPanel.prototype._onImageUploadError = function() {
        this._toggleSaveButtonEnabled(false);
    };

    AvatarPickerPanel.prototype._onImageClear = function() {
        this._toggleSaveButtonEnabled(false);
        this.hideMessage();
    }

    AvatarPickerPanel.prototype._toggleSaveButtonEnabled = function(opt_enable) {
        if (opt_enable === null) {
            opt_enable = this._$saveButton.prop("disabled");
        }

        if (opt_enable) {
            this._$saveButton.prop("disabled", false );
        } else {
            this._$saveButton.prop("disabled", true);
        }
    };

    AvatarPickerPanel.prototype.setMessage = function(message) {
        this._$message.html(aui.message.error({
            content: message,
            closeable: false
        })).show();
        this._toggleExpanded(true);
    };

    AvatarPickerPanel.prototype.hideMessage = function() {
        this._$message.hide();
        this._toggleExpanded(false);
    };

    AvatarPickerPanel.prototype.getImageCropper = function() {
        return this.imageUploadAndCrop;
    };

    AvatarPickerPanel.prototype._toggleExpanded = function(toggle) {
        this.options.onExpanded(toggle);
    };

    return AvatarPickerPanel;
});
