require([
    'jquery',
    'ajs',
    'aui/dialog2',
    'confluence/analytics-support',
    'confluence-user-profile/avatar-picker/avatar-picker-dialog',
],
/**
 * @tainted AJS.I18n
 * @tainted AJS.Meta
 */
function(
    $,
    AJS,
    Dialog2,
    Analytics,
    AvatarPickerDialog
) {
    'use strict';

    $(function () {
        let avatarPickerDialog, avatarInitialsDialog;
        const userKey = AJS.Meta.get('remote-user-key');
        const DEFAULT_AVATAR_PATH = '/images/icons/profilepics/default.png';
        const DEFAULT_AVATAR_ICON_PATH = '/images/icons/profilepics/default.svg';
        const DEFAULT_USER_LOGO_PATH = '/images/icons/profilepics/add_profile_pic.svg';
        const confirmDialogId = '#confirm-user-profile-avatar-remove-dialog';
        const $uploadImageAvatarButton = $('#profile-avatar-button');
        const $uploadInitialsAvatarButton = $('#profile-avatar-initials-button');
        const $removeAvatarButton = $('#profile-avatar-remove-photo-button');
        const $confirmRemoveAvatarButton = $('#confirm-remove-avatar-dialog-confirm');
        const PICKER = 'picker';
        const PHOTO = 'photo';
        const INITIALS = 'initials';
        const DELETE_EVENT_NAME = 'confluence.user.profile.avatar.delete';
        const UPLOAD_EVENT_NAME = 'confluence.user.profile.avatar.upload';
        const SAVE_EVENT_NAME = 'confluence.user.profile.avatar.save';

        const updatePreviewImages = function (newAvatarPath) {
            let avatarPath = newAvatarPath || DEFAULT_AVATAR_PATH;
            let avatarIconPath = newAvatarPath || DEFAULT_AVATAR_ICON_PATH;
            let userLogoPath = newAvatarPath || DEFAULT_USER_LOGO_PATH;

            $('.user-avatar-preview').attr('src', AJS.Meta.get('context-path') + avatarPath); // update main preview image
            $('#title-heading img').attr('src', AJS.Meta.get('context-path') + userLogoPath); // update icon in top left
            $('#user-menu-link img').attr('src', AJS.Meta.get('context-path') + avatarIconPath); // update icon in top right
        };

        const toggleRemoveAvatarButton = function (shouldShow) {
            if (shouldShow) {
                toggleConfirmButtonDisabled(false);
                $removeAvatarButton.show();
            } else {
                $removeAvatarButton.hide();
            }
        };

        const toggleConfirmButtonDisabled = function (shouldDisable) {
            if (shouldDisable) {
                $confirmRemoveAvatarButton.prop('disabled', true);
            } else {
                $confirmRemoveAvatarButton.prop('disabled',false);
            }
        };

        const publishAnalyticsEvent = function (eventName, isSuccess, avatarType) {
            let payload = {};
            if (eventName === DELETE_EVENT_NAME) {
                payload = { success: isSuccess };
            } else if (eventName === UPLOAD_EVENT_NAME) {
                payload = { avatarType: avatarType };
            } else if (eventName === SAVE_EVENT_NAME){
                payload = { success: isSuccess, avatarType: avatarType };
            }

            Analytics.publish(eventName, payload);
        }

        function showDialog(dialogType, e) {
            const isPickerDialog = dialogType === PICKER;
            let dialog = isPickerDialog ? avatarPickerDialog : avatarInitialsDialog;

            if (!dialog) {
                dialog = new AvatarPickerDialog({
                    dialogType: dialogType,
                    dialogOptions: {
                        id: `personal-avatar-${dialogType}-dialog`,
                        width: isPickerDialog ? 400 : 350,
                        height: isPickerDialog ? 550 : 440,
                        closeOnOutsideClick: false,
                    },
                    onCrop: function (croppedDataURI) {
                        $.ajax({
                            type: 'POST',
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({ avatarDataURI: croppedDataURI }),
                            url: AJS.Meta.get('context-path') + '/rest/user-profile/1.0/' + userKey + '/avatar/upload',
                            success: function (data) {
                                updatePreviewImages(data.avatarPath);
                                toggleRemoveAvatarButton(true);
                                dialog.hide();
                                publishAnalyticsEvent(SAVE_EVENT_NAME, true, isPickerDialog ? PHOTO : INITIALS);
                            },
                            error: function () {
                                if (isPickerDialog) {
                                    dialog.setMessage(
                                        AJS.I18n.getText('user.avatar.picker.upload.error')
                                    );
                                }
                                publishAnalyticsEvent(SAVE_EVENT_NAME, false, isPickerDialog ? PHOTO : INITIALS);
                            },
                        });
                    },
                });

                if (isPickerDialog) {
                    avatarPickerDialog = dialog;
                } else {
                    avatarInitialsDialog = dialog;
                }
            }

            dialog.show();
            publishAnalyticsEvent(UPLOAD_EVENT_NAME, true, isPickerDialog ? PHOTO : INITIALS);
            e.preventDefault();
            return false;
        }

        const deleteProfilePhoto = function () {
            toggleConfirmButtonDisabled(true);

            $.ajax({
                type: 'DELETE',
                dataType: 'json',
                contentType: 'application/json',
                data: {},
                url: AJS.Meta.get('context-path') + '/rest/user-profile/1.0/' + userKey + '/avatar/',
                success: function () {
                    updatePreviewImages();
                    toggleRemoveAvatarButton(false);
                    Dialog2(confirmDialogId).hide();

                    AJS.flag({
                        type: 'success',
                        title: AJS.I18n.getText('user.profile.picture.delete.flag.title'),
                        body: AJS.I18n.getText('user.profile.picture.delete.flag.description'),
                        close: 'auto',
                    });
                    publishAnalyticsEvent(DELETE_EVENT_NAME, true);
                },
                error: function () {
                    toggleConfirmButtonDisabled(false);
                    AJS.flag({
                        type: 'error',
                        title: AJS.I18n.getText('user.profile.picture.delete.flag.error.title'),
                        body: AJS.I18n.getText('user.profile.picture.delete.flag.error.description'),
                        close: 'auto',
                    });
                    publishAnalyticsEvent(DELETE_EVENT_NAME, false);
                },
            });

        };

        $uploadImageAvatarButton.click(function (e) {
            showDialog(PICKER, e);
            e.preventDefault();
        });

        $uploadInitialsAvatarButton.on('click keydown', function (e) {
            // Only process Enter and space key for keydown event
            if (e.type === 'keydown' && e.key !== 'Enter' && e.key !== ' ')  {
                return;
            }

            // Prevent the meatball button from retaining focus when dialog is opened.
            $("#profile-avatar-actions-menu").trigger("aui-button-invoke");
            e.stopPropagation();

            showDialog(INITIALS, e);
        });

        $removeAvatarButton.on('click keydown', function (e) {
            // Only process Enter and space key for keydown event
            if (e.type === 'keydown' && e.key !== 'Enter' && e.key !== ' ') {
                return;
            }

            Dialog2(confirmDialogId).show();
        });

        $confirmRemoveAvatarButton.on('click', function () {
            deleteProfilePhoto();
        });

        $("#confirm-remove-avatar-dialog-cancel").on("click", function () {
            Dialog2(confirmDialogId).hide();
        });
    });
});
