package com.atlassian.jira_soapclient;


import _soapclient.JiraSoapService;
import _soapclient.JiraSoapServiceService;
import _soapclient.JiraSoapServiceServiceLocator;
import com.atlassian.jira.rpc.soap.beans.RemoteGroup;
import com.atlassian.jira.rpc.soap.beans.RemoteUser;
import com.atlassian.jira.rpc.exception.RemoteValidationException;
import org.apache.log4j.Category;
import org.apache.xmlrpc.XmlRpcClient;

import java.util.Hashtable;
import java.util.Vector;
import java.util.Random;
import java.util.Properties;
import java.net.URL;
import java.io.FileInputStream;
import java.io.File;
import java.io.InputStream;

/**
 * A set of User tools.
 */
public class UserTools
{
    private static Category LOG = Category.getInstance(UserTools.class);

    // connect to confluence using xml rpc
    private XmlRpcClient confluence = null;
    private String confluenceToken = null;

    // connect to jira using soap.
    private JiraSoapService jira = null;
    private String jiraToken = null;

    public static void main(String[] args) throws Exception
    {
        UserTools migration = new UserTools();
        if (migration.initialise())
        {
            migration.migrate();
        }
    }

    private void migrate() throws Exception
    {
        initialise();

        Vector usernames = (Vector) confluence.execute("confluence1.getActiveUsers", makeParams(confluenceToken, Boolean.TRUE));
        for (int i = 0; i < usernames.size(); i++)
        {
            String username = (String) usernames.get(i);
            Hashtable confluenceUser = (Hashtable) confluence.execute("confluence1.getUser", makeParams(confluenceToken, username));

            RemoteUser jiraUser = jira.getUser(jiraToken, username);
            if (jiraUser == null)
            {
                LOG.info("Transferring user: " + username);
                String email = ((String) confluenceUser.get("email")).trim();
                if (email.equals(""))
                {
                    email = username + "@example.com";
                }
                String password = generatePassword();
                String fullname = ((String) confluenceUser.get("fullname")).trim();
                jiraUser = jira.createUser(jiraToken, username, password, fullname, email);
            }

            Vector groups = (Vector) confluence.execute("confluence1.getUserGroups", makeParams(confluenceToken, username));
            for (int j = 0; j < groups.size(); j++)
            {
                String groupname = (String) groups.elementAt(j);
                RemoteGroup jiraGroup = null;
                try
                {
                    jiraGroup = jira.getGroup(jiraToken, groupname);
                }
                catch (RemoteValidationException e)
                {
                    //noop.
                }

                if (jiraGroup == null)
                {
                    LOG.info("Transferring group: " + groupname);
                    jiraGroup = jira.createGroup(jiraToken, groupname, jiraUser);
                }

                jira.addUserToGroup(jiraToken, jiraGroup, jiraUser);
            }
        }
    }

    private static final char[] CHARACTERS = new char[]{'1', '2', '3', '4', '5', '6', '7', '8', '9', '0',
            'a', 'A', 'b', 'B', 'c', 'C', 'd', 'D', 'e', 'E', 'f', 'F', 'g', 'G', 'h', 'H', 'i', 'I', 'j', 'J',
            'k', 'K', 'l', 'L', 'm', 'M', 'n', 'N', 'o', 'O', 'p', 'P', 'q', 'Q', 'r', 'R', 's', 'S', 't', 'T',
            'u', 'U', 'v', 'V', 'w', 'W', 'x', 'X', 'y', 'Y', 'z', 'Z'
    };

    private static final Random RANDOM = new Random(System.currentTimeMillis());

    private String generatePassword()
    {
        StringBuffer password = new StringBuffer();
        for (int i = 0; i < 8; i++)
        {
            password.append(CHARACTERS[RANDOM.nextInt(CHARACTERS.length)]);
        }
        return password.toString();
    }

    /**
     * Initialise connections to confluence and jiras remote apis.
     */
    private boolean initialise()
    {
        Properties config;
        try
        {
            config = getConnectionConfig();
        }
        catch (Exception e)
        {
            LOG.fatal("Failed to load configuration properties.", e);
            return false;
        }

        // initialise and login to confluence and jira.
        try
        {
            confluence = new XmlRpcClient(config.getProperty("confluence.url"));
        }
        catch (Exception e)
        {
            LOG.fatal("Failed to connect to confluence xml rpc service.", e);
            return false;
        }

        try
        {
            confluenceToken = (String) confluence.execute("confluence1.login", makeParams("admin", "admin"));
        }
        catch (Exception e)
        {
            LOG.fatal("Failed to log in to confluence.", e);
            return false;
        }


        try
        {
            JiraSoapServiceService jiraSoapServiceGetter = new JiraSoapServiceServiceLocator();
            jira = jiraSoapServiceGetter.getJirasoapserviceV2(new URL(config.getProperty("jira.url")));
        }
        catch (Exception e)
        {
            LOG.fatal("Failed to connect to jira.", e);
            return false;
        }
        try
        {
            jiraToken = jira.login("admin", "admin");
        }
        catch (Exception e)
        {
            LOG.fatal("Failed to log in to jira.", e);
            return false;
        }

        return true;
    }

    private Properties getConnectionConfig() throws Exception
    {
        Properties config = new Properties();

        InputStream in = null;
        try
        {
            in = getClass().getResourceAsStream("/connection.properties");
            if (in == null)
            {
                in = new FileInputStream(new File("connection.properties"));
            }

            config.load(in);
            return config;
        }
        finally
        {
            if (in != null)
            {
                in.close();
            }
        }

    }

    // help for xml rpc.

    protected Vector makeParams(Object p1)
    {
        Vector params;
        params = new Vector();
        params.add(p1);
        return params;
    }

    protected Vector makeParams(Object p1, Object p2)
    {
        Vector params = makeParams(p1);
        params.add(p2);
        return params;
    }
}
