/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.search.common.jobs.checks;

import com.atlassian.bitbucket.internal.search.common.jobs.checks.HealthCheckRegistry;
import com.atlassian.bitbucket.internal.search.common.jobs.checks.HealthCheckResult;
import com.atlassian.bitbucket.internal.search.common.jobs.checks.SearchHealthCheck;
import com.atlassian.bitbucket.util.RetryBackoffUtils;
import com.atlassian.sal.api.lifecycle.LifecycleAware;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.RunMode;
import com.atlassian.scheduler.config.Schedule;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.time.Clock;
import java.time.Duration;
import java.util.Comparator;
import java.util.Date;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service(value="healthCheckRunner")
public class HealthCheckRunner
implements LifecycleAware,
HealthCheckRegistry,
JobRunner {
    private static final Duration INITIAL_SCHEDULE_DELAY = Duration.ofSeconds(20L);
    private static final String JOB_KEY = "SEARCH_HEALTH_CHECK";
    private static final Duration MAX_SCHEDULE_DELAY = Duration.ofHours(12L);
    private static final Duration MIN_SCHEDULE_DELAY = Duration.ofMinutes(5L);
    private static final Logger log = LoggerFactory.getLogger(HealthCheckRunner.class);
    private final Clock clock;
    private final Set<SearchHealthCheck> healthChecks;
    private final SchedulerService schedulerService;
    private JobConfig defaultJobConfig;
    private int successCount;

    @Autowired
    public HealthCheckRunner(SchedulerService schedulerService, Clock clock) {
        this.schedulerService = schedulerService;
        this.clock = clock;
        this.healthChecks = new ConcurrentSkipListSet<SearchHealthCheck>(new HealthCheckComparator());
        this.successCount = 0;
    }

    public void onStart() {
        JobRunnerKey jobKey = JobRunnerKey.of((String)JOB_KEY);
        this.schedulerService.getJobsByJobRunnerKey(jobKey).forEach(jobDetails -> this.schedulerService.unscheduleJob(jobDetails.getJobId()));
        this.schedulerService.registerJobRunner(jobKey, (JobRunner)this);
        this.defaultJobConfig = JobConfig.forJobRunnerKey((JobRunnerKey)jobKey).withRunMode(RunMode.RUN_ONCE_PER_CLUSTER);
        this.doSchedule(INITIAL_SCHEDULE_DELAY);
    }

    public void onStop() {
        this.schedulerService.unregisterJobRunner(JobRunnerKey.of((String)JOB_KEY));
    }

    @Override
    public void registerHealthCheck(@Nonnull SearchHealthCheck healthCheck) {
        this.checkCanRegister(Objects.requireNonNull(healthCheck, "healthCheck"));
        this.successCount = 0;
        this.healthChecks.add(healthCheck);
        log.info(String.format("New health check registered: %s", healthCheck.getClass().getSimpleName()));
    }

    @Nullable
    public JobRunnerResponse runJob(@Nonnull JobRunnerRequest request) {
        if (request.isCancellationRequested()) {
            log.info("HealthChecks job has been canceled.");
            return JobRunnerResponse.success();
        }
        log.debug(String.format("Running up to %d search health checks", this.healthChecks.size()));
        this.successCount = this.healthChecks.stream().filter(SearchHealthCheck::test).findFirst().map(failure -> {
            HealthCheckResult healthCheckResult = failure.manageFailure();
            log.debug("Recovery result for {} was {}", (Object)failure.getClass().getSimpleName(), (Object)healthCheckResult);
            return 0;
        }).orElse(this.successCount + 1);
        Duration backOffDelay = HealthCheckRunner.getBackOffDelay(this.successCount);
        this.doSchedule(backOffDelay);
        log.debug("Health checks will run again after " + String.valueOf(backOffDelay));
        return JobRunnerResponse.success();
    }

    @Override
    public void unregisterHealthCheck(@Nonnull SearchHealthCheck healthCheck) {
        this.healthChecks.remove(Objects.requireNonNull(healthCheck, "healthCheck"));
    }

    private static Duration getBackOffDelay(int successCount) {
        return RetryBackoffUtils.calculateDelay((int)successCount, (Duration)MIN_SCHEDULE_DELAY, (Duration)MAX_SCHEDULE_DELAY);
    }

    private void checkCanRegister(SearchHealthCheck check) {
        if (check.getWeight() > 1000 || check.getClass().getPackage().getName().startsWith("com.atlassian")) {
            return;
        }
        throw new IllegalArgumentException("Weights equal to or below 1000 are reserved for system use");
    }

    private void doSchedule(Duration delay) {
        Schedule schedule = Schedule.runOnce((Date)Date.from(this.clock.instant().plus(delay)));
        try {
            this.schedulerService.scheduleJobWithGeneratedId(this.defaultJobConfig.withSchedule(schedule));
        }
        catch (SchedulerServiceException e) {
            log.error("Unable to schedule health check job", (Throwable)e);
        }
    }

    private static class HealthCheckComparator
    implements Comparator<SearchHealthCheck> {
        private HealthCheckComparator() {
        }

        @Override
        public int compare(SearchHealthCheck o1, SearchHealthCheck o2) {
            return Long.compare(o1.getWeight(), o2.getWeight());
        }
    }
}

