/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.plugins.authentication.encryption;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.plugins.authentication.encryption.EncryptionConfigurationService;
import com.atlassian.plugins.authentication.encryption.db.SealedEntitiesDao;
import com.atlassian.plugins.authentication.encryption.exception.EncryptionException;
import com.atlassian.plugins.authentication.encryption.model.PlainSecret;
import com.atlassian.plugins.authentication.encryption.model.SealedSecret;
import com.atlassian.plugins.authentication.encryption.model.Secret;
import com.atlassian.secrets.DefaultSecretStoreProvider;
import com.atlassian.secrets.api.SecretStore;
import com.atlassian.secrets.store.algorithm.paramters.DecryptionParameters;
import com.atlassian.secrets.store.algorithm.paramters.EncryptionParameters;
import com.atlassian.secrets.store.algorithm.serialization.SerializationFileFactory;
import com.atlassian.secrets.store.algorithm.serialization.UniqueFilePathGenerator;
import com.google.gson.Gson;
import io.atlassian.fugue.Suppliers;
import jakarta.annotation.Nonnull;
import java.io.File;
import java.nio.file.Path;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class EncryptionService {
    private static final Logger log = LoggerFactory.getLogger(EncryptionService.class);
    private static final String ALGORITHM = "AES/CBC/PKCS5Padding";
    static final String CIPHER = "com.atlassian.db.config.password.ciphers.algorithm.AesOnlyAlgorithmCipher";
    final SealedEntitiesDao sealedEntitiesDao;
    final Gson gson = new Gson();
    private final AtomicReference<Supplier<SecretStore>> supplierReference;
    private final EncryptionConfigurationService encryptionConfigurationService;
    private final Clock clock;
    private final SerializationFileFactory serializationFileFactory;

    public EncryptionService(SealedEntitiesDao sealedEntitiesDao, EncryptionConfigurationService encryptionConfigurationService, Clock clock) {
        this(sealedEntitiesDao, new AtomicReference<Supplier<SecretStore>>(), encryptionConfigurationService, clock, new SerializationFileFactory());
    }

    @VisibleForTesting
    EncryptionService(SealedEntitiesDao sealedEntitiesDao, AtomicReference<Supplier<SecretStore>> supplierReference, EncryptionConfigurationService encryptionConfigurationService, Clock clock, SerializationFileFactory serializationFileFactory) {
        this.sealedEntitiesDao = sealedEntitiesDao;
        this.supplierReference = supplierReference;
        this.encryptionConfigurationService = encryptionConfigurationService;
        this.clock = clock;
        this.serializationFileFactory = serializationFileFactory;
    }

    @Nonnull
    public Secret seal(@Nonnull String plainTextToSecure) {
        Secret secret;
        try {
            Supplier<SecretStore> secretStoreSupplier = this.ensureSecretStoreInitialized();
            Optional<Path> encryptionKeyPath = this.encryptionConfigurationService.getEncryptionKeyPath();
            if (encryptionKeyPath.isEmpty()) {
                throw new IllegalStateException("Cannot encrypt values as encryption key is not configured");
            }
            Path keyDirectoryPath = this.encryptionConfigurationService.getKeyDirectory();
            EncryptionParameters encryptionParameters = new EncryptionParameters.Builder().setPlainTextPassword(plainTextToSecure).setAlgorithm(ALGORITHM).setAlgorithmKey("AES").setSaveSealedObjectToSeparateFile(Boolean.valueOf(false)).setSaveAlgorithmParametersToSeparateFile(Boolean.valueOf(false)).setOutputFilesBasePath(StringUtils.appendIfMissing((String)keyDirectoryPath.toString(), (CharSequence)File.separator, (CharSequence[])new CharSequence[0])).setKeyFilePath(this.absolutize(keyDirectoryPath, encryptionKeyPath.get().toString()).toString()).build();
            String sealedSecret = Optional.ofNullable(secretStoreSupplier.get()).map(secretStore -> secretStore.store(this.gson.toJson(encryptionParameters))).map(decryptionParametersJson -> (DecryptionParameters)this.gson.fromJson(decryptionParametersJson, DecryptionParameters.class)).map(decryptionParameters -> this.relativizeKeyPath((DecryptionParameters)decryptionParameters, keyDirectoryPath)).map(arg_0 -> ((Gson)this.gson).toJson(arg_0)).orElseGet(() -> {
                log.warn("Cannot instantiate secret store. The secret cannot be encrypted!");
                return plainTextToSecure;
            });
            secret = plainTextToSecure.equals(sealedSecret) ? Secret.unSealed(plainTextToSecure) : Secret.sealed(sealedSecret);
        }
        catch (Throwable e) {
            log.error("Error executing the plain text sealing: {}", (Object)e.getMessage());
            throw new EncryptionException("Unable to encrypt the secret", e);
        }
        return secret;
    }

    @Nonnull
    public Secret seal(@Nonnull String identifier, @Nonnull String plainTextToSecure) {
        Secret secret = this.seal(plainTextToSecure);
        this.sealedEntitiesDao.put(identifier, secret.serialize());
        return secret;
    }

    @Nonnull
    public Secret unseal(@Nonnull Secret secret) {
        return this.decrypt(secret);
    }

    @Nonnull
    public Secret unseal(@Nonnull String identifier) {
        Optional<String> sealedString = this.sealedEntitiesDao.get(identifier);
        if (sealedString.isEmpty()) {
            throw new EncryptionException("Cannot find sealed string for the provided identifier");
        }
        return this.decrypt(Secret.of(sealedString.get()));
    }

    @Nonnull
    public Optional<Path> generateKey() {
        try {
            Supplier<SecretStore> secretStoreSupplier = this.ensureSecretStoreInitialized();
            if (secretStoreSupplier.get() != null) {
                Path keyDirectoryPath = this.encryptionConfigurationService.getKeyDirectory();
                KeyGenerator keyGen = EncryptionService.getKeyGenerator();
                keyGen.init(128);
                SecretKeySpec key = (SecretKeySpec)keyGen.generateKey();
                String keyPath = keyDirectoryPath.resolve(new UniqueFilePathGenerator(SecretKeySpec.class.getName(), this.clock).generateName()).toString();
                this.serializationFileFactory.getSerializationFile(keyPath).createFileAndSave((Object)key);
                return Optional.of(this.relativize(keyDirectoryPath, keyPath));
            }
        }
        catch (Throwable e) {
            log.error("Error generating key: {}", (Object)e.getMessage());
            log.debug("Error generating key", e);
        }
        return Optional.empty();
    }

    @Nonnull
    private static KeyGenerator getKeyGenerator() throws NoSuchAlgorithmException, NoSuchProviderException {
        return Security.getProvider("BC") != null ? KeyGenerator.getInstance("AES", "BC") : KeyGenerator.getInstance("AES");
    }

    @Nonnull
    private DecryptionParameters relativizeKeyPath(@Nonnull DecryptionParameters decryptionParameters, @Nonnull Path keyDirectoryPath) {
        return new DecryptionParameters(decryptionParameters.getSealedObjectFilePath(), this.relativize(keyDirectoryPath, decryptionParameters.getKeyFilePath()).toString(), decryptionParameters.getSerializedSealedObject());
    }

    @Nonnull
    private Path relativize(@Nonnull Path keyDirectoryPath, @Nonnull String keyPath) {
        return keyDirectoryPath.relativize(Path.of(keyPath, new String[0]));
    }

    @Nonnull
    private Secret decrypt(@Nonnull Secret secret) {
        if (secret instanceof PlainSecret) {
            return secret;
        }
        try {
            Supplier<SecretStore> secretStoreSupplier = this.ensureSecretStoreInitialized();
            Secret secretWithAbsolutePath = this.absolutize(secret, this.encryptionConfigurationService.getKeyDirectory());
            return Optional.ofNullable(secretStoreSupplier.get()).map(secretStore -> secretStore.get(Secret.rawValue(secretWithAbsolutePath))).map(Secret::unSealed).orElseGet(() -> {
                log.warn("Cannot instantiate secret store");
                return secret;
            });
        }
        catch (Throwable e) {
            log.error("Cannot unseal the sealed string. Returning the sealed string \"as is\". Error: {}", (Object)e.getMessage());
            log.debug("Cannot unseal the sealed string.", e);
            return secret;
        }
    }

    @VisibleForTesting
    Supplier<SecretStore> ensureSecretStoreInitialized() {
        this.supplierReference.compareAndSet(null, Suppliers.memoize(() -> new DefaultSecretStoreProvider().getInstance(CIPHER).orElse(null)));
        return this.supplierReference.get();
    }

    @Nonnull
    private Secret absolutize(@Nonnull Secret secret, @Nonnull Path keyDirectoryPath) {
        if (!(secret instanceof SealedSecret)) {
            throw new IllegalArgumentException("Secret must be a SealedSecret in order to have its key path absolutized");
        }
        String decryptionParametersAsJson = Secret.rawValue(secret);
        DecryptionParameters decryptionParameters = (DecryptionParameters)this.gson.fromJson(decryptionParametersAsJson, DecryptionParameters.class);
        return Secret.sealed(this.absolutize(decryptionParameters, keyDirectoryPath));
    }

    @Nonnull
    private String absolutize(@Nonnull DecryptionParameters decryptionParameters, @Nonnull Path keyDirectoryPath) {
        DecryptionParameters decryptionParametersWithAbsolutePath = new DecryptionParameters(decryptionParameters.getSealedObjectFilePath(), this.absolutize(keyDirectoryPath, decryptionParameters.getKeyFilePath()).toString(), decryptionParameters.getSerializedSealedObject());
        return this.gson.toJson((Object)decryptionParametersWithAbsolutePath);
    }

    @Nonnull
    private Path absolutize(@Nonnull Path keyDirectoryPath, @Nonnull String keyFilePath) {
        return keyDirectoryPath.resolve(keyFilePath);
    }
}

