/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.oauth2.client.storage;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.oauth2.client.api.ClientConfiguration;
import com.atlassian.oauth2.client.api.ClientToken;
import com.atlassian.oauth2.client.api.ClientTokenMetadata;
import com.atlassian.oauth2.client.api.OAuth2Token;
import com.atlassian.oauth2.client.api.lib.token.TokenService;
import com.atlassian.oauth2.client.api.lib.token.TokenServiceException;
import com.atlassian.oauth2.client.api.storage.ClientCredentialsTokenHandler;
import com.atlassian.oauth2.client.api.storage.TokenHandler;
import com.atlassian.oauth2.client.api.storage.config.ClientConfigStorageService;
import com.atlassian.oauth2.client.api.storage.config.GrantType;
import com.atlassian.oauth2.client.api.storage.event.ClientTokenRecoverableEvent;
import com.atlassian.oauth2.client.api.storage.event.ClientTokenUnrecoverableEvent;
import com.atlassian.oauth2.client.api.storage.token.ClientTokenEntity;
import com.atlassian.oauth2.client.api.storage.token.exception.AccessTokenExpiredException;
import com.atlassian.oauth2.client.api.storage.token.exception.ConfigurationNotFoundException;
import com.atlassian.oauth2.client.api.storage.token.exception.RecoverableTokenException;
import com.atlassian.oauth2.client.api.storage.token.exception.RefreshTokenExpiredException;
import com.atlassian.oauth2.client.api.storage.token.exception.TokenNotFoundException;
import com.atlassian.oauth2.client.api.storage.token.exception.UnrecoverableTokenException;
import com.atlassian.oauth2.client.properties.SystemProperty;
import com.atlassian.oauth2.client.storage.token.InternalClientTokenStorageService;
import com.atlassian.oauth2.client.storage.token.OAuth2TokenImpl;
import com.atlassian.oauth2.common.concurrent.KeyedLocks;
import com.google.common.base.Throwables;
import jakarta.annotation.Nonnull;
import java.time.Clock;
import java.time.DateTimeException;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultTokenHandler
implements TokenHandler,
ClientCredentialsTokenHandler {
    private static final Logger logger = LoggerFactory.getLogger(DefaultTokenHandler.class);
    @VisibleForTesting
    static final int MAX_EXECUTE_ATTEMPTS = 2;
    private static final Duration LARGE_MARGIN = Duration.ofDays(3650L);
    private static final String NEW_TOKEN_LOCK_ID = "new_token";
    private final InternalClientTokenStorageService clientTokenStorageService;
    private final ClientConfigStorageService clientConfigStorageService;
    private final TokenService tokenService;
    private final Clock clock;
    private final KeyedLocks<String> tokenLocks = new KeyedLocks(SystemProperty.DEFAULT_MONITOR_STRIPE_COUNT.getValue());
    private final Duration minFailingPeriodForUnrecoverable;
    private final EventPublisher eventPublisher;

    public DefaultTokenHandler(InternalClientTokenStorageService clientTokenStorageService, ClientConfigStorageService clientConfigStorageService, TokenService tokenService, Clock clock, Duration minFailingPeriodForUnrecoverable, EventPublisher eventPublisher) {
        this.clientTokenStorageService = clientTokenStorageService;
        this.clientConfigStorageService = clientConfigStorageService;
        this.tokenService = tokenService;
        this.clock = clock;
        this.minFailingPeriodForUnrecoverable = minFailingPeriodForUnrecoverable;
        this.eventPublisher = eventPublisher;
    }

    public <T> T execute(String clientTokenId, TokenHandler.ClientTokenCallback<T> callback) throws UnrecoverableTokenException, RecoverableTokenException {
        return this.execute(clientTokenId, callback, LARGE_MARGIN);
    }

    public <T> T execute(String clientTokenId, TokenHandler.ClientTokenCallback<T> callback, Duration margin) throws UnrecoverableTokenException, RecoverableTokenException {
        ClientTokenEntity clientTokenEntity = null;
        TokenHandler.InvalidTokenException caught = null;
        for (int attempt = 1; attempt <= 2; ++attempt) {
            clientTokenEntity = this.getRefreshedToken(clientTokenId, margin, false);
            try {
                Object result = callback.apply((ClientToken)clientTokenEntity);
                this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.VALID));
                return (T)result;
            }
            catch (TokenHandler.InvalidTokenException e) {
                caught = e;
                logger.debug("Token with ID {} reported as invalid during attempt {} of {}", new Object[]{clientTokenId, attempt, 2});
                logger.trace("Exception caught: ", (Throwable)e);
                continue;
            }
        }
        if (this.getClientConfiguration(clientTokenEntity).getGrantType().requiresAuthorization()) {
            return this.handleAuthorizationCodeFailure(clientTokenEntity, (Exception)((Object)caught));
        }
        return this.handleClientCredentialsFailure(clientTokenEntity, (Exception)((Object)caught));
    }

    public OAuth2Token getRefreshedToken(@Nonnull String externalId, @Nonnull String configId, @Nonnull Duration margin) throws UnrecoverableTokenException, RecoverableTokenException {
        ClientTokenEntity clientTokenEntity = this.clientTokenStorageService.getByExternalIdAndConfigId(externalId, configId).orElseThrow(() -> new UnrecoverableTokenException("Token not found for external ID: " + externalId + " and config ID: " + configId));
        ClientTokenEntity refreshedToken = this.getRefreshedToken(clientTokenEntity.getId(), margin);
        return new OAuth2TokenImpl(refreshedToken.getAccessToken(), refreshedToken.getAccessTokenExpiration());
    }

    public OAuth2Token getClientCredentialsToken(@Nonnull String externalId, @Nonnull String configId, Duration margin) throws UnrecoverableTokenException, RecoverableTokenException {
        ClientTokenEntity refreshedToken;
        Optional<ClientTokenEntity> optionalClientTokenEntity = this.clientTokenStorageService.getByExternalIdAndConfigId(externalId, configId);
        ClientTokenEntity clientTokenEntity = optionalClientTokenEntity.orElseGet(() -> ClientTokenEntity.builder().accessToken("").configId(configId).externalId(externalId).accessTokenExpiration(Instant.EPOCH).status(ClientTokenMetadata.ClientTokenStatus.UNKNOWN).lastStatusUpdated(Instant.EPOCH).build());
        ClientConfiguration clientConfiguration = this.getClientConfiguration(clientTokenEntity);
        if (clientConfiguration.getGrantType() != GrantType.CLIENT_CREDENTIALS_GRANT) {
            throw new UnrecoverableTokenException("Can request only client_credentials token");
        }
        if (!this.isTokenLifetimeGreaterThan(clientTokenEntity, margin)) {
            try {
                String clientTokenEntityLock = optionalClientTokenEntity.map(ClientTokenEntity::getId).orElse(NEW_TOKEN_LOCK_ID + externalId + configId);
                refreshedToken = this.tokenLocks.executeWithLock(clientTokenEntityLock, () -> this.getNewClientCredentialsToken(clientTokenEntity, clientConfiguration));
            }
            catch (Exception e) {
                Throwables.propagateIfPossible((Throwable)e, UnrecoverableTokenException.class, RecoverableTokenException.class);
                throw new RuntimeException("Unexpected exception", e);
            }
        } else {
            refreshedToken = clientTokenEntity;
        }
        return new OAuth2TokenImpl(refreshedToken.getAccessToken(), refreshedToken.getAccessTokenExpiration());
    }

    public boolean deleteTokenByValue(@Nonnull String tokenValue) {
        return this.clientTokenStorageService.deleteTokenByValue(tokenValue);
    }

    public ClientTokenEntity getRefreshedToken(String clientTokenId) throws UnrecoverableTokenException, RecoverableTokenException {
        return this.getRefreshedToken(clientTokenId, LARGE_MARGIN);
    }

    public ClientTokenEntity getRefreshedToken(String clientTokenId, Duration margin) throws UnrecoverableTokenException, RecoverableTokenException {
        return this.getRefreshedToken(clientTokenId, margin, true);
    }

    private ClientTokenEntity getRefreshedToken(String clientTokenId, Duration margin, boolean allowRecovery) throws UnrecoverableTokenException, RecoverableTokenException {
        try {
            return this.tokenLocks.executeWithLock(clientTokenId, () -> this.refreshTokenIfNeeded(this.clientTokenStorageService.getByIdOrFail(clientTokenId), margin, allowRecovery));
        }
        catch (Exception e) {
            Throwables.propagateIfPossible((Throwable)e, UnrecoverableTokenException.class, RecoverableTokenException.class);
            throw new RuntimeException("Unexpected exception", e);
        }
    }

    private ClientTokenEntity refreshTokenIfNeeded(ClientTokenEntity clientTokenEntity, Duration margin, boolean allowRecovery) throws UnrecoverableTokenException, RecoverableTokenException {
        if (clientTokenEntity.getStatus() == ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE) {
            throw new UnrecoverableTokenException("Token already marked as invalid");
        }
        ClientConfiguration clientConfiguration = this.getClientConfiguration(clientTokenEntity);
        GrantType grantType = clientConfiguration.getGrantType();
        if (this.isTokenLifetimeGreaterThan(clientTokenEntity, margin)) {
            return clientTokenEntity;
        }
        return grantType.requiresAuthorization() ? this.refreshAuthorizationCodeToken(clientTokenEntity, allowRecovery, clientConfiguration, grantType) : this.getNewClientCredentialsToken(clientTokenEntity, clientConfiguration);
    }

    private ClientTokenEntity getNewClientCredentialsToken(ClientTokenEntity clientTokenEntity, ClientConfiguration clientConfiguration) throws UnrecoverableTokenException {
        try {
            ClientToken refreshedToken = this.tokenService.forceRefresh(clientConfiguration, (ClientToken)clientTokenEntity);
            return this.updateToken(clientTokenEntity, token -> token.updateFrom(refreshedToken).status(ClientTokenMetadata.ClientTokenStatus.UNKNOWN));
        }
        catch (TokenServiceException e) {
            return (ClientTokenEntity)this.handleClientCredentialsFailure(clientTokenEntity, (Exception)((Object)e));
        }
    }

    private <T> T handleClientCredentialsFailure(ClientTokenEntity clientTokenEntity, Exception e) throws UnrecoverableTokenException {
        this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE));
        throw new UnrecoverableTokenException("Token request failed.", (Throwable)e);
    }

    private ClientTokenEntity refreshAuthorizationCodeToken(ClientTokenEntity clientTokenEntity, boolean allowRecovery, ClientConfiguration clientConfiguration, GrantType grantType) throws UnrecoverableTokenException, RecoverableTokenException {
        if (clientTokenEntity.getRefreshToken() == null) {
            if (!this.isTokenLifetimeGreaterThan(clientTokenEntity, Duration.ZERO)) {
                this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE));
                throw new AccessTokenExpiredException("Cannot refresh expired access token for flow of type: " + String.valueOf(grantType) + " without a refresh token");
            }
            return clientTokenEntity;
        }
        try {
            ClientToken refreshedToken = this.tokenService.forceRefresh(clientConfiguration, (ClientToken)clientTokenEntity);
            boolean shouldRecover = allowRecovery && clientTokenEntity.getStatus() == ClientTokenMetadata.ClientTokenStatus.RECOVERABLE;
            return this.updateToken(clientTokenEntity, token -> token.updateFrom(refreshedToken).lastRefreshed(this.clock.instant()).status(shouldRecover ? ClientTokenMetadata.ClientTokenStatus.UNKNOWN : clientTokenEntity.getStatus()).incrementRefreshCount());
        }
        catch (TokenServiceException e) {
            return (ClientTokenEntity)this.handleAuthorizationCodeFailure(clientTokenEntity, (Exception)((Object)e));
        }
    }

    private <T> T handleAuthorizationCodeFailure(ClientTokenEntity clientTokenEntity, Exception e) throws UnrecoverableTokenException, RecoverableTokenException {
        Duration failingPeriod;
        Instant now = this.clock.instant().minus(SystemProperty.MAX_CLOCK_SKEW.getValue());
        if (clientTokenEntity.getRefreshTokenExpiration().isBefore(now)) {
            this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE));
            throw new RefreshTokenExpiredException("Cannot refresh the access token as the refresh token has expired");
        }
        if (clientTokenEntity.getStatus() == ClientTokenMetadata.ClientTokenStatus.RECOVERABLE && (failingPeriod = Duration.between(clientTokenEntity.getLastStatusUpdated(), this.clock.instant())).compareTo(this.minFailingPeriodForUnrecoverable) >= 0) {
            this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE));
            throw new UnrecoverableTokenException("Token refresh has been failing for longer than the configured duration: " + String.valueOf(this.minFailingPeriodForUnrecoverable), (Throwable)e);
        }
        ClientTokenEntity updatedClientTokenEntity = this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.RECOVERABLE));
        throw new RecoverableTokenException("An error has occurred while refreshing OAuth token", (Throwable)e, updatedClientTokenEntity.getLastStatusUpdated());
    }

    private ClientConfiguration getClientConfiguration(ClientTokenEntity clientTokenEntity) throws TokenNotFoundException, ConfigurationNotFoundException {
        Optional maybeClientConfiguration = this.clientConfigStorageService.getById(clientTokenEntity.getConfigId());
        if (maybeClientConfiguration.isEmpty()) {
            this.updateToken(clientTokenEntity, token -> token.status(ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE));
            throw new ConfigurationNotFoundException("Cannot refresh token as client configuration does not exist");
        }
        this.clientConfigStorageService.getById(clientTokenEntity.getConfigId()).orElseThrow(() -> new ConfigurationNotFoundException("Cannot refresh token as client configuration does not exist"));
        return (ClientConfiguration)maybeClientConfiguration.get();
    }

    private boolean isTokenLifetimeGreaterThan(ClientTokenEntity clientTokenEntity, Duration tokenLifetime) {
        Instant now = this.clock.instant().minus(SystemProperty.MAX_CLOCK_SKEW.getValue());
        try {
            return clientTokenEntity.getAccessTokenExpiration().isAfter(now.plus(tokenLifetime));
        }
        catch (ArithmeticException | DateTimeException e) {
            logger.debug("Margin exceeds limits of Instant. Refresh required", (Throwable)e);
            return true;
        }
    }

    private ClientTokenEntity updateToken(ClientTokenEntity clientTokenEntity, Consumer<ClientTokenEntity.Builder> update) throws TokenNotFoundException {
        ClientTokenEntity.Builder builder = ClientTokenEntity.builder((ClientTokenEntity)clientTokenEntity);
        update.accept(builder);
        if (!clientTokenEntity.getStatus().equals((Object)builder.getStatus()) || builder.getLastStatusUpdated() == null) {
            builder.lastStatusUpdated(this.clock.instant());
        }
        ClientTokenEntity updatedToken = builder.build();
        if (clientTokenEntity.getStatus() != updatedToken.getStatus()) {
            if (updatedToken.getStatus() == ClientTokenMetadata.ClientTokenStatus.RECOVERABLE) {
                this.eventPublisher.publish((Object)new ClientTokenRecoverableEvent(updatedToken.getId()));
            } else if (updatedToken.getStatus() == ClientTokenMetadata.ClientTokenStatus.UNRECOVERABLE) {
                this.eventPublisher.publish((Object)new ClientTokenUnrecoverableEvent(updatedToken.getId()));
            }
        }
        return Objects.equals(clientTokenEntity, updatedToken) ? updatedToken : this.clientTokenStorageService.save(updatedToken);
    }

    @VisibleForTesting
    int getTokensUnderRefreshCount() {
        return this.tokenLocks.size();
    }
}

