/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.scm.git.lfs.lock;

import com.atlassian.bitbucket.AuthorisationException;
import com.atlassian.bitbucket.NoSuchEntityException;
import com.atlassian.bitbucket.auth.AuthenticationContext;
import com.atlassian.bitbucket.event.repository.RepositoryDeletedEvent;
import com.atlassian.bitbucket.i18n.I18nService;
import com.atlassian.bitbucket.internal.scm.git.lfs.GitLfsDisabledException;
import com.atlassian.bitbucket.internal.scm.git.lfs.GitLfsUtils;
import com.atlassian.bitbucket.internal.scm.git.lfs.LfsService;
import com.atlassian.bitbucket.internal.scm.git.lfs.dao.AoLfsLock;
import com.atlassian.bitbucket.internal.scm.git.lfs.dao.LfsLockDao;
import com.atlassian.bitbucket.internal.scm.git.lfs.event.GitLfsCreateLockEvent;
import com.atlassian.bitbucket.internal.scm.git.lfs.event.GitLfsDeleteLockEvent;
import com.atlassian.bitbucket.internal.scm.git.lfs.event.GitLfsListLocksEvent;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLock;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockConflictException;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockCreationException;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockCreationRequest;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockDeletionRequest;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockListRequest;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockService;
import com.atlassian.bitbucket.internal.scm.git.lfs.lock.LfsLockUnsupportedException;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.permission.PermissionValidationService;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.server.ApplicationMode;
import com.atlassian.bitbucket.server.ApplicationPropertiesService;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.UserService;
import com.atlassian.bitbucket.util.MoreStreams;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;
import com.atlassian.bitbucket.util.PageUtils;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import jakarta.annotation.Nonnull;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.java.ao.ActiveObjectsException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultLfsLockService
implements LfsLockService {
    private static final Logger log = LoggerFactory.getLogger(DefaultLfsLockService.class);
    private final AuthenticationContext authContext;
    private final EventPublisher eventPublisher;
    private final I18nService i18nService;
    private final LfsService lfsService;
    private final LfsLockDao lockDao;
    private final ApplicationMode mode;
    private final TransactionTemplate transactionTemplate;
    private final UserService userService;
    private final PermissionValidationService validationService;

    public DefaultLfsLockService(@Nonnull AuthenticationContext authContext, @Nonnull EventPublisher eventPublisher, @Nonnull I18nService i18nService, @Nonnull LfsLockDao lockDao, @Nonnull LfsService lfsService, @Nonnull ApplicationPropertiesService propertiesService, @Nonnull TransactionTemplate transactionTemplate, @Nonnull UserService userService, @Nonnull PermissionValidationService validationService) {
        this.authContext = authContext;
        this.eventPublisher = eventPublisher;
        this.i18nService = i18nService;
        this.lfsService = lfsService;
        this.lockDao = lockDao;
        this.mode = propertiesService.getMode();
        this.transactionTemplate = transactionTemplate;
        this.userService = userService;
        this.validationService = validationService;
    }

    @Override
    @Nonnull
    public LfsLock createLock(@Nonnull LfsLockCreationRequest request) {
        Objects.requireNonNull(request, "request");
        Repository repository = request.getRepository();
        this.validationService.validateForRepository(repository, Permission.REPO_WRITE);
        this.requireLfsEnabledFor(repository);
        LfsLock created = (LfsLock)this.transactionTemplate.execute(() -> {
            Optional<AoLfsLock> existing = this.lockDao.getByRepositoryAndPath(repository, request.getPath());
            existing.ifPresent(aoLfsLock -> this.throwConflictException(repository, (AoLfsLock)aoLfsLock));
            ApplicationUser currentUser = this.authContext.getCurrentUser();
            try {
                AoLfsLock lock = this.lockDao.create(currentUser, request.getPath(), repository);
                lock.initialise(currentUser, repository);
                return lock;
            }
            catch (ActiveObjectsException ex) {
                log.error("{}: Failed to create lock: {}", new Object[]{repository, request.getPath(), ex});
                throw new LfsLockCreationException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.lock.creation.fail", new Object[0]));
            }
        });
        this.eventPublisher.publish((Object)new GitLfsCreateLockEvent(this, repository));
        return created;
    }

    @Override
    @Nonnull
    public LfsLock deleteLock(@Nonnull LfsLockDeletionRequest request) {
        Objects.requireNonNull(request, "request");
        Repository repository = request.getRepository();
        this.validationService.validateForRepository(repository, Permission.REPO_WRITE);
        this.requireLfsEnabledFor(repository);
        LfsLock deleted = (LfsLock)this.transactionTemplate.execute(() -> {
            Optional<AoLfsLock> existing = this.lockDao.getById(request.getId());
            if (existing.map(lock -> !this.isLockForRepo((AoLfsLock)lock, repository)).orElse(true).booleanValue()) {
                throw new NoSuchEntityException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.lock.no.such", new Object[]{String.valueOf(request.getId())}));
            }
            AoLfsLock lock2 = existing.get();
            ApplicationUser lockOwner = this.getLockOwner(lock2);
            lock2.initialise(lockOwner, repository);
            ApplicationUser currentUser = this.authContext.getCurrentUser();
            if (lockOwner.getId() != currentUser.getId()) {
                if (!request.isForce()) {
                    throw new AuthorisationException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.lock.deletion.permission.denied", new Object[]{GitLfsUtils.getLockOwnerName(lockOwner)}));
                }
                log.info("{}: Lock {} owned by {} was force unlocked by {}", new Object[]{repository, existing.get().getPath(), lockOwner.getSlug(), currentUser.getSlug()});
            }
            this.lockDao.delete(lock2);
            return lock2;
        });
        this.eventPublisher.publish((Object)new GitLfsDeleteLockEvent(this, repository, request.isForce()));
        return deleted;
    }

    @Override
    @Nonnull
    public Map<String, ApplicationUser> getEnforceableLocks(@Nonnull Repository repository) {
        Objects.requireNonNull(repository, "repository");
        ApplicationUser authenticatedUser = this.authContext.getCurrentUser();
        return PageUtils.toStream(pageRequest -> this.getLocks(null, "", repository, pageRequest), (int)100).filter(lock -> !lock.getOwner().equals((Object)authenticatedUser)).collect(Collectors.toMap(LfsLock::getPath, LfsLock::getOwner));
    }

    @Override
    @Nonnull
    public Page<LfsLock> listLocks(@Nonnull LfsLockListRequest request, @Nonnull PageRequest pageRequest) {
        Objects.requireNonNull(pageRequest, "pageRequest");
        Objects.requireNonNull(request, "request");
        Repository repository = request.getRepository();
        this.validationService.validateRepositoryAccessible(repository);
        if (this.isLfsDisabled(repository)) {
            return PageUtils.createEmptyPage((PageRequest)pageRequest);
        }
        Page<LfsLock> result = this.getLocks(request.getId(), request.getPath(), request.getRepository(), pageRequest);
        if (request.isSendingAnalytics()) {
            this.eventPublisher.publish((Object)new GitLfsListLocksEvent(this, repository, result.getSize()));
        }
        return result;
    }

    @Override
    @Nonnull
    public Page<LfsLock> listLocksByDirectory(@Nonnull Repository repository, @Nonnull String directory, @Nonnull PageRequest pageRequest) {
        Objects.requireNonNull(directory, "directory");
        Objects.requireNonNull(pageRequest, "pageRequest");
        Objects.requireNonNull(repository, "repository");
        this.validationService.validateRepositoryAccessible(repository);
        if (this.isLfsDisabled(repository)) {
            return PageUtils.createEmptyPage((PageRequest)pageRequest);
        }
        return (Page)this.transactionTemplate.execute(() -> {
            Page<AoLfsLock> lockPage = this.lockDao.findByRepositoryAndDirectory(repository, directory, pageRequest.buildRestrictedPageRequest(100));
            this.initialiseLocks(repository, lockPage);
            return PageUtils.asPageOf(LfsLock.class, lockPage);
        });
    }

    @EventListener
    public void onRepositoryDeleted(RepositoryDeletedEvent event) {
        this.transactionTemplate.execute(() -> {
            this.lockDao.deleteByRepository(event.getRepository());
            return null;
        });
    }

    private Page<AoLfsLock> findLocks(Integer id, String path, Repository repository, PageRequest pageRequest) {
        if (id == null && StringUtils.isEmpty((CharSequence)path)) {
            return this.lockDao.findByRepository(repository, pageRequest.buildRestrictedPageRequest(100));
        }
        if (id != null) {
            return PageUtils.createPage((Iterable)this.lockDao.getById(id).filter(lock -> lock.getRepositoryId().intValue() == repository.getId()).map(Collections::singleton).orElse(Collections.emptySet()), (PageRequest)pageRequest);
        }
        return PageUtils.createPage((Iterable)this.lockDao.getByRepositoryAndPath(repository, path).map(Collections::singleton).orElse(Collections.emptySet()), (PageRequest)pageRequest);
    }

    private ApplicationUser getLockOwner(AoLfsLock lock) {
        ApplicationUser lockOwner = this.userService.getUserById(lock.getOwnerId().intValue(), true);
        if (lockOwner == null) {
            log.warn("The owner of lock {} with user ID {} does not exist", (Object)lock.getId(), (Object)lock.getOwnerId());
            throw new NoSuchEntityException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.lock.owner.not.found", new Object[]{String.valueOf(lock.getId())}));
        }
        return lockOwner;
    }

    private Page<LfsLock> getLocks(Integer id, String path, Repository repository, PageRequest pageRequest) {
        return (Page)this.transactionTemplate.execute(() -> {
            Page<AoLfsLock> lockPage = this.findLocks(id, path, repository, pageRequest);
            this.initialiseLocks(repository, lockPage);
            return PageUtils.asPageOf(LfsLock.class, lockPage);
        });
    }

    private Map<Integer, ApplicationUser> getOwnersForLocks(Page<AoLfsLock> lockPage) {
        HashMap<Integer, ApplicationUser> ownersById = new HashMap<Integer, ApplicationUser>();
        Set ownerIds = MoreStreams.streamIterable((Iterable)lockPage.getValues()).map(AoLfsLock::getOwnerId).collect(Collectors.toSet());
        if (ownerIds.size() > 0) {
            ownersById.putAll(this.userService.getUsersById(ownerIds, true).stream().collect(Collectors.toMap(ApplicationUser::getId, Function.identity())));
        }
        return ownersById;
    }

    private void initialiseLocks(Repository repository, Page<AoLfsLock> lockPage) {
        Map<Integer, ApplicationUser> ownersById = this.getOwnersForLocks(lockPage);
        for (AoLfsLock lock : lockPage.getValues()) {
            ApplicationUser owner = ownersById.get(lock.getOwnerId());
            if (owner == null) continue;
            lock.initialise(owner, repository);
        }
    }

    private boolean isLfsDisabled(Repository repository) {
        return this.mode == ApplicationMode.MIRROR || !this.lfsService.isEnabled(repository);
    }

    private boolean isLockForRepo(AoLfsLock lock, Repository repository) {
        return lock.getRepositoryId().equals(repository.getId());
    }

    private void requireLfsEnabledFor(Repository repository) {
        if (this.mode == ApplicationMode.MIRROR) {
            log.warn("Git LFS locking commands are not supported on a mirror node - they should be proxied to the upstream.");
            throw new LfsLockUnsupportedException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.locking.unsupported", new Object[0]));
        }
        if (!this.lfsService.isEnabled(repository)) {
            throw new GitLfsDisabledException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.locking.unsupported", new Object[0]));
        }
    }

    private void throwConflictException(Repository repository, AoLfsLock existing) {
        ApplicationUser lockOwner = this.getLockOwner(existing);
        existing.initialise(lockOwner, repository);
        throw new LfsLockConflictException(this.i18nService.createKeyedMessage("bitbucket.scm.git.lfs.lock.creation.conflict", new Object[]{GitLfsUtils.getLockOwnerName(lockOwner)}), existing);
    }
}

