/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.hook.script;

import com.atlassian.bitbucket.hook.script.HookScriptOutOfDateException;
import com.atlassian.bitbucket.hook.script.HookScriptSizeExceededException;
import com.atlassian.bitbucket.hook.script.HookScriptStoreException;
import com.atlassian.bitbucket.hook.script.MinimalHookScript;
import com.atlassian.bitbucket.i18n.I18nService;
import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.server.StorageService;
import com.atlassian.bitbucket.util.MoreFiles;
import com.atlassian.bitbucket.util.NumberUtils;
import com.atlassian.stash.internal.hook.script.FileStagedHookScript;
import com.atlassian.stash.internal.hook.script.HookScriptStore;
import com.atlassian.stash.internal.hook.script.StagedHookScript;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.io.Closeables;
import io.atlassian.fugue.retry.ExceptionHandler;
import io.atlassian.fugue.retry.ExceptionHandlers;
import io.atlassian.fugue.retry.RetryFactory;
import jakarta.annotation.Nonnull;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component(value="hookScriptStore")
public class DiskHookScriptStore
implements HookScriptStore {
    private static final Logger log = LoggerFactory.getLogger(DiskHookScriptStore.class);
    private final I18nService i18nService;
    private final Path scriptDir;
    private long backOffTime;
    private int maxSize;

    @Autowired
    public DiskHookScriptStore(I18nService i18nService, StorageService storageService) {
        this.i18nService = i18nService;
        this.backOffTime = 1000L;
        this.maxSize = 0xA00000;
        this.scriptDir = MoreFiles.mkdir((Path)storageService.getConfigDir(), (String)"hook-scripts");
    }

    @Override
    public void commit(@Nonnull MinimalHookScript script, @Nonnull StagedHookScript staged) {
        long scriptId = Objects.requireNonNull(script, "script").getId();
        int scriptVersion = script.getVersion();
        Path tempFile = Objects.requireNonNull((FileStagedHookScript)staged, "staged").getTempFile();
        try {
            if (MoreFiles.isPosixSupported((Path)tempFile)) {
                Files.setPosixFilePermissions(tempFile, EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE));
            }
        }
        catch (IOException e) {
            log.error("{}: Version {} could not be set executable", new Object[]{scriptId, scriptVersion, e});
            throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.writefailed", new Object[0]), (Throwable)e);
        }
        Path scriptFile = this.buildPath(scriptId, scriptVersion);
        try {
            try {
                Files.move(tempFile, scriptFile, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (AtomicMoveNotSupportedException e) {
                Files.move(tempFile, scriptFile, new CopyOption[0]);
            }
        }
        catch (FileAlreadyExistsException e) {
            log.error("{}: A file already exists for version {}", new Object[]{scriptId, scriptVersion, e});
            throw new HookScriptOutOfDateException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.concurrentupdate", new Object[0]), (Throwable)e);
        }
        catch (IOException e) {
            log.error("{}: The temp file could not be renamed for version {}", new Object[]{scriptId, scriptVersion, e});
            throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.writefailed", new Object[0]), (Throwable)e);
        }
    }

    @Override
    public void delete(@Nonnull MinimalHookScript script) {
        long scriptId = Objects.requireNonNull(script, "script").getId();
        int deleted = 0;
        int total = 0;
        for (int i = script.getVersion() + 1; i > -1; --i) {
            try {
                if (this.tryDelete(this.buildPath(scriptId, i))) {
                    log.trace("{}: Deleted version {}", (Object)scriptId, (Object)i);
                    ++deleted;
                    ++total;
                    continue;
                }
                log.debug("{}: Script version {} does not exist", (Object)scriptId, (Object)i);
                continue;
            }
            catch (UncheckedIOException e) {
                log.warn("{}: Script version {} could not be deleted", new Object[]{scriptId, i, e});
                ++total;
            }
        }
        if (total > 0) {
            log.info("{}: Deleted {}/{} version(s) found", new Object[]{scriptId, deleted, total});
        }
    }

    @Override
    public void deleteUnreferencedScripts(@Nonnull Set<MinimalHookScript> existingScripts) throws IOException {
        Set expectedScripts = Objects.requireNonNull(existingScripts, "existingScripts").stream().map(script -> this.buildPath(script.getId(), script.getVersion())).collect(Collectors.toSet());
        try (Stream<Path> files = Files.list(this.scriptDir);){
            files.filter(path -> !expectedScripts.contains(path)).forEach(path -> {
                log.warn("Orphaned file found, deleting file: {}", (Object)path.getFileName());
                this.tryDelete((Path)path);
            });
        }
    }

    @Override
    public void deleteVersion(@Nonnull MinimalHookScript script) {
        long scriptId = Objects.requireNonNull(script, "script").getId();
        int version = script.getVersion();
        Path path = this.buildPath(scriptId, version);
        try {
            if (this.tryDelete(path)) {
                log.trace("{}: Deleted version {}", (Object)scriptId, (Object)version);
            } else {
                log.debug("{}: Script version {} does not exist", (Object)scriptId, (Object)version);
            }
        }
        catch (UncheckedIOException e) {
            log.error("{}: Version {} could not be deleted", new Object[]{scriptId, version, e});
        }
    }

    @Override
    public Path getExecutable(@Nonnull MinimalHookScript script) {
        Path path;
        block10: {
            long scriptId = Objects.requireNonNull(script, "script").getId();
            int scriptVersion = script.getVersion();
            Path scriptFile = this.buildPath(scriptId, scriptVersion);
            InputStream ignored = Files.newInputStream(scriptFile, new OpenOption[0]);
            try {
                path = scriptFile;
                if (ignored == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    if (e instanceof FileNotFoundException || e instanceof NoSuchFileException) {
                        log.error("{}: No file exists for version {}", new Object[]{scriptId, scriptVersion, e});
                    } else {
                        log.error("{}: Version {} could not be read", new Object[]{scriptId, scriptVersion, e});
                    }
                    throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.readfailed", new Object[]{scriptId}), (Throwable)e);
                }
            }
            ignored.close();
        }
        return path;
    }

    @Override
    public int getMaxSize() {
        return this.maxSize;
    }

    @Override
    @Nonnull
    public InputSupplier<InputStream> read(@Nonnull MinimalHookScript script) {
        Path scriptFile = this.getExecutable(script);
        return () -> Files.newInputStream(scriptFile, new OpenOption[0]);
    }

    @Value(value="${hookscripts.size.max:10485760}")
    public void setMaxSize(int maxSize) {
        this.maxSize = maxSize;
    }

    @Override
    @Nonnull
    public StagedHookScript stage(@Nonnull InputSupplier<InputStream> contents) {
        InputStream rawStream;
        try {
            rawStream = (InputStream)Objects.requireNonNull(contents, "contents").open();
        }
        catch (IOException e) {
            throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.supplierfailed", new Object[0]), (Throwable)e);
        }
        if (rawStream == null) {
            throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.nostream", new Object[0]));
        }
        Path tempFile = null;
        try {
            tempFile = Files.createTempFile(this.scriptDir, "unsaved", ".tmp", new FileAttribute[0]);
            MessageDigest digest = DigestUtils.getSha256Digest();
            int size = this.writeAndDigest(tempFile, rawStream, digest);
            if (size > this.maxSize) {
                throw new HookScriptSizeExceededException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.oversized", new Object[]{NumberUtils.formatSize((double)size), NumberUtils.formatSize((double)this.maxSize)}), this.maxSize, size);
            }
            String hash = Hex.encodeHexString((byte[])digest.digest());
            FileStagedHookScript staged = new FileStagedHookScript(tempFile, hash, size);
            tempFile = null;
            FileStagedHookScript fileStagedHookScript = staged;
            return fileStagedHookScript;
        }
        catch (IOException e) {
            log.error("Failed to stage contents", (Throwable)e);
            throw new HookScriptStoreException(this.i18nService.createKeyedMessage("bitbucket.service.hook.script.store.writefailed", new Object[0]), (Throwable)e);
        }
        finally {
            Closeables.closeQuietly((InputStream)rawStream);
            if (tempFile != null) {
                MoreFiles.deleteQuietly((Path)tempFile);
            }
        }
    }

    @VisibleForTesting
    Path buildPath(long id, int version) {
        return this.scriptDir.resolve(id + "-" + version);
    }

    @VisibleForTesting
    void setBackOffTime(long backOffTime) {
        this.backOffTime = backOffTime;
    }

    private boolean tryDelete(Path path) {
        Objects.requireNonNull(path, "path");
        Supplier<Boolean> deleter = () -> {
            try {
                Files.delete(path);
                return true;
            }
            catch (FileNotFoundException | NoSuchFileException e) {
                return false;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        };
        try {
            if (deleter.get().booleanValue()) {
                return true;
            }
        }
        catch (UncheckedIOException uncheckedIOException) {
            // empty catch block
        }
        return (Boolean)RetryFactory.create(deleter, (int)5, (ExceptionHandler)ExceptionHandlers.ignoreExceptionHandler(), (long)this.backOffTime).get();
    }

    private int writeAndDigest(Path tempFile, InputStream rawStream, MessageDigest digest) throws IOException {
        int size = 0;
        try (DigestInputStream inputStream = new DigestInputStream(rawStream, digest);
             OutputStream outputStream = Files.newOutputStream(tempFile, new OpenOption[0]);){
            int read;
            byte[] buffer = new byte[8192];
            while ((read = ((InputStream)inputStream).read(buffer, 0, buffer.length)) != -1) {
                if ((size += read) > this.maxSize) continue;
                outputStream.write(buffer, 0, read);
            }
        }
        return size;
    }
}

