/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.migration;

import com.atlassian.bitbucket.io.IoConsumer;
import com.atlassian.bitbucket.migration.SequentialArchive;
import com.atlassian.bitbucket.util.IoUtils;
import com.atlassian.stash.internal.migration.FilePermissionUtils;
import com.atlassian.stash.internal.migration.FileWalkerUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.CountingOutputStream;
import com.google.common.primitives.Ints;
import jakarta.annotation.Nonnull;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.PosixFileAttributeView;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.utils.CharsetNames;
import org.apache.commons.io.TaggedIOException;
import org.apache.commons.io.output.CloseShieldOutputStream;
import org.apache.commons.io.output.TaggedOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TarArchive
implements SequentialArchive {
    private static final int BASE_FILE_MODE = 32768;
    private static final int COPY_BUFFER_SIZE = 32768;
    private static final FileSystem FS = FileSystems.getDefault();
    private static final long MAX_FULLY_BUFFERED_COPY_SIZE = 524288L;
    private static final String PATH_SEP = File.separatorChar == '\\' ? "\\\\" : File.separator;
    private static final PathMatcher NFS_SILLY_RENAME_MATCHER = FS.getPathMatcher("glob:{**/,}.nfs??????*".replace("/", PATH_SEP));
    private static final Logger log = LoggerFactory.getLogger(TarArchive.class);
    private final TarArchiveOutputStream outputStream;
    private final TaggedOutputStream underlyingOutputStream;

    TarArchive(OutputStream outputStream) {
        this.underlyingOutputStream = new TaggedOutputStream((OutputStream)new CloseShieldOutputStream(outputStream));
        this.outputStream = new TarArchiveOutputStream((OutputStream)this.underlyingOutputStream, CharsetNames.UTF_8);
        this.outputStream.setLongFileMode(3);
        this.outputStream.setBigNumberMode(2);
        this.outputStream.setAddPaxHeadersForNonAsciiNames(true);
    }

    public void addEntry(@Nonnull Path entryName, @Nonnull IoConsumer<OutputStream> writer) throws IOException {
        Objects.requireNonNull(entryName, "entryName");
        Objects.requireNonNull(writer, "writer");
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        writer.accept((Object)buffer);
        TarArchiveEntry entry = new TarArchiveEntry(entryName.toString());
        entry.setSize((long)buffer.size());
        this.outputStream.putArchiveEntry(entry);
        this.outputStream.write(buffer.toByteArray());
        this.outputStream.closeArchiveEntry();
    }

    public void addPathFromDisk(@Nonnull Path destinationInBundle, @Nonnull Path sourceOnDisk, @Nonnull PathMatcher filter, int attempts) throws IOException {
        Objects.requireNonNull(sourceOnDisk, "sourceOnDisk");
        Objects.requireNonNull(destinationInBundle, "destinationInBundle");
        Objects.requireNonNull(filter, "filter");
        if (!filter.matches(sourceOnDisk) || NFS_SILLY_RENAME_MATCHER.matches(sourceOnDisk)) {
            log.trace("addPathFromDisk('{}'): File or directory won't be copied, because it doesn't pass the filter.", (Object)sourceOnDisk);
            return;
        }
        if (Files.isDirectory(sourceOnDisk, new LinkOption[0])) {
            log.trace("addPathFromDisk('{}'): Adding path as directory recursively.", (Object)sourceOnDisk);
            this.addDirectoryPathFromDisk(destinationInBundle, sourceOnDisk, filter, attempts);
        } else {
            log.trace("addPathFromDisk('{}'): Adding path as file.", (Object)sourceOnDisk);
            this.addFile(destinationInBundle.resolve(sourceOnDisk.getFileName()), sourceOnDisk);
        }
    }

    public void close() throws IOException {
        this.outputStream.close();
    }

    @VisibleForTesting
    void writeEntry(@Nonnull SeekableByteChannel inChannel, @Nonnull Path entryName, @Nonnull IoConsumer<TarArchiveEntry> entryProcessor) throws IOException {
        InputStream inputStream = Channels.newInputStream(inChannel);
        long size = inChannel.size();
        TarArchiveEntry entry = new TarArchiveEntry(entryName.toString());
        entry.setSize(size);
        entryProcessor.accept((Object)entry);
        CountingOutputStream countingOutputStream = new CountingOutputStream((OutputStream)this.outputStream);
        try {
            this.outputStream.putArchiveEntry(entry);
        }
        catch (IOException e) {
            throw new FatalIOException(TarArchive.unwrapTaggedIOException(e));
        }
        try {
            int bufferSize;
            int n = bufferSize = size > 524288L ? 32768 : Ints.saturatedCast((long)size);
            if (bufferSize > 0) {
                IoUtils.copy((InputStream)inputStream, (OutputStream)countingOutputStream, (int)bufferSize);
            }
            this.outputStream.closeArchiveEntry();
        }
        catch (IOException exception) {
            IOException cause = TarArchive.unwrapTaggedIOException(exception);
            log.debug("[{}] Exception while writing file to archive after {} bytes", new Object[]{entryName, countingOutputStream.getCount(), log.isTraceEnabled() ? cause : null});
            if (this.underlyingOutputStream.isCauseOf((Exception)exception)) {
                throw new FatalIOException(cause);
            }
            try {
                TarArchive.padExistingEntryAndMarkAsDeleted(this.outputStream, entryName, size - countingOutputStream.getCount());
            }
            catch (IOException suppressed) {
                IOException nestedCause = TarArchive.unwrapTaggedIOException(suppressed);
                FatalIOException nestedException = new FatalIOException(cause);
                nestedException.addSuppressed(nestedCause);
                throw nestedException;
            }
            throw cause;
        }
    }

    private static void padExistingEntryAndMarkAsDeleted(@Nonnull TarArchiveOutputStream tarOutputStream, @Nonnull Path entryName, long remainingBytes) throws IOException {
        IoUtils.copy((InputStream)new ZeroFillingInputStream(remainingBytes), (OutputStream)tarOutputStream, (int)32768);
        tarOutputStream.closeArchiveEntry();
        TarArchiveEntry deletedMarker = new TarArchiveEntry(entryName.toString() + ".atl.deleted");
        deletedMarker.setSize(0L);
        tarOutputStream.putArchiveEntry(deletedMarker);
        tarOutputStream.closeArchiveEntry();
    }

    private static IOException unwrapTaggedIOException(IOException exception) {
        return exception instanceof TaggedIOException ? (IOException)exception.getCause() : exception;
    }

    private void addDirectoryPathFromDisk(final @Nonnull Path destinationInBundle, final @Nonnull Path sourcePath, final @Nonnull PathMatcher filter, int attempts) throws IOException {
        FileWalkerUtils.walkFileTreeWithRetry((Path)sourcePath, (Set)ImmutableSet.of((Object)((Object)FileVisitOption.FOLLOW_LINKS)), (int)Integer.MAX_VALUE, (FileVisitor)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (!filter.matches(dir)) {
                    log.trace("addDirectoryPathFromDisk('{}'): Skipping directory '{}' because it doesn't pass the filter.", (Object)sourcePath, (Object)sourcePath.relativize(dir));
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return super.preVisitDirectory(dir, attrs);
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (filter.matches(file) && !NFS_SILLY_RENAME_MATCHER.matches(file)) {
                    if (log.isTraceEnabled()) {
                        log.trace("addDirectoryPathFromDisk('{}'): Adding file '{}'.", (Object)sourcePath, (Object)sourcePath.relativize(file));
                    }
                    TarArchive.this.addFile(destinationInBundle.resolve(sourcePath.relativize(file)), file);
                } else if (log.isTraceEnabled()) {
                    log.trace("addDirectoryPathFromDisk('{}'): Skipping file '{}' because it doesn't pass the filter.", (Object)sourcePath, (Object)sourcePath.relativize(file));
                }
                return FileVisitResult.CONTINUE;
            }
        }, (int)attempts, (long)2000L, (long)TimeUnit.SECONDS.toMillis(512L), e -> {
            if (e instanceof FatalIOException) {
                throw e;
            }
        });
    }

    private void addFile(@Nonnull Path entryName, @Nonnull Path sourcePath) throws IOException {
        try (FileChannel open = FileChannel.open(sourcePath, StandardOpenOption.READ);){
            this.writeEntry(open, entryName, (IoConsumer<TarArchiveEntry>)((IoConsumer)entry -> {
                PosixFileAttributeView fileAttributeView = Files.getFileAttributeView(sourcePath, PosixFileAttributeView.class, new LinkOption[0]);
                if (fileAttributeView != null) {
                    entry.setMode(FilePermissionUtils.toIntMode(fileAttributeView.readAttributes().permissions(), (int)32768));
                }
            }));
        }
    }

    static class FatalIOException
    extends IOException {
        private static final long serialVersionUID = 4762280231270037748L;

        FatalIOException(Throwable cause) {
            super(cause);
        }
    }

    static class ZeroFillingInputStream
    extends InputStream {
        private final long bytesToGenerate;
        private long currentBytes;

        ZeroFillingInputStream(long bytesToGenerate) {
            this.bytesToGenerate = bytesToGenerate;
        }

        @Override
        public int read() {
            if (this.currentBytes >= this.bytesToGenerate) {
                return -1;
            }
            ++this.currentBytes;
            return 0;
        }

        @Override
        public int read(@Nonnull byte[] b, int off, int len) {
            int actualLen = (int)Math.min(this.bytesToGenerate - this.currentBytes, (long)len);
            if (actualLen == 0) {
                return -1;
            }
            Arrays.fill(b, off, off + actualLen, (byte)0);
            this.currentBytes += (long)actualLen;
            return actualLen;
        }
    }
}

