/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.throttle;

import com.atlassian.stash.internal.ApplicationConstants;
import com.atlassian.stash.internal.config.Clock;
import com.atlassian.stash.internal.server.OperatingSystemService;
import com.atlassian.stash.internal.throttle.ResourceThrottleStrategy;
import com.atlassian.stash.internal.throttle.SemaphoreTicketBucket;
import com.atlassian.stash.internal.throttle.ThrottlingOutputUtils;
import com.atlassian.stash.internal.throttle.TicketSummary;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import jakarta.annotation.Nonnull;
import java.text.DecimalFormat;
import java.util.OptionalDouble;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AdaptiveResourceThrottleStrategy
implements ResourceThrottleStrategy {
    private static final Logger log = LoggerFactory.getLogger(AdaptiveResourceThrottleStrategy.class);
    private static final long FALLBACK_WARN_INTERVAL = TimeUnit.HOURS.toMillis(1L);
    private final int acquireTimeoutSeconds;
    private final Clock clock;
    private final double cpuHistoricalWeighting;
    private final int intervalSeconds;
    private final int maxTickets;
    private final int minTickets;
    private final int fallbackTickets;
    private final OperatingSystemService operatingSystemService;
    private final String resourceName;
    private final double targetCpuLoad;
    private final int fallbackThresholdSeconds;
    private final Logger profileLogger;
    private final double typicalCpuPerTicket;
    private volatile State state;

    AdaptiveResourceThrottleStrategy(String resourceName, Clock clock, OperatingSystemService operatingSystemService, int intervalSeconds, int minTickets, int maxTickets, int fallbackTickets, double targetCpuLoad, double typicalCpuPerTicket, double cpuHistoricalWeighting, int acquireTimeoutSeconds, int fallbackThresholdSeconds) {
        this.typicalCpuPerTicket = typicalCpuPerTicket;
        this.acquireTimeoutSeconds = acquireTimeoutSeconds;
        this.clock = clock;
        this.cpuHistoricalWeighting = cpuHistoricalWeighting;
        this.intervalSeconds = intervalSeconds;
        this.maxTickets = maxTickets;
        this.minTickets = minTickets;
        this.fallbackTickets = fallbackTickets;
        this.operatingSystemService = operatingSystemService;
        this.resourceName = resourceName;
        this.targetCpuLoad = targetCpuLoad;
        this.fallbackThresholdSeconds = fallbackThresholdSeconds;
        this.profileLogger = LoggerFactory.getLogger((String)("throttle.profile." + resourceName));
        this.state = new StartingState(fallbackTickets, clock.utcMillis());
    }

    @Override
    @Nonnull
    public SemaphoreTicketBucket create() {
        return new SemaphoreTicketBucket(this.fallbackTickets, this.acquireTimeoutSeconds, TimeUnit.SECONDS);
    }

    @Override
    @Nonnull
    public OptionalLong getUpdateInterval(@Nonnull TimeUnit timeUnit) {
        return OptionalLong.of(TimeUnit.SECONDS.convert(this.intervalSeconds, timeUnit));
    }

    public String toString() {
        return "[state=" + String.valueOf(this.state) + ", config=[ticket-range=[" + this.minTickets + ".." + this.maxTickets + "], target-cpu=" + this.targetCpuLoad + ", fallback-tickets=" + this.fallbackTickets + ", fallback-timeout=" + this.fallbackThresholdSeconds + "s, acquire-timeout=" + this.acquireTimeoutSeconds + "s]]";
    }

    @Override
    public void update(@Nonnull SemaphoreTicketBucket bucket) {
        long now = this.clock.utcMillis();
        State nextState = null;
        try {
            TicketSummary ticketSummary = bucket.summarize(this.resourceName);
            OptionalDouble optCpuLoad = this.operatingSystemService.getCpuLoad();
            int bucketTickets = ticketSummary.getTotal();
            int previousTickets = (int)Math.round(this.state.tickets);
            if (this.updateTicketsIfDifferent(bucket, bucketTickets, previousTickets)) {
                log.warn("[{}] Throttling ticket limit ({}) and the tickets on the bucket ({}) have diverged since the last interval", new Object[]{this.resourceName, previousTickets, bucketTickets});
            }
            nextState = this.state.transition(now, optCpuLoad, ticketSummary);
        }
        catch (RuntimeException e) {
            log.error("[{}] Error encountered during adaptive throttling update. Setting tickets to {} as a fallback measure", (Object)this.resourceName, (Object)e);
        }
        if (nextState == null) {
            this.updateTicketsIfDifferent(bucket, this.fallbackTickets, (int)Math.round(this.state.tickets));
            return;
        }
        this.updateTicketsIfDifferent(bucket, (int)Math.round(nextState.tickets), (int)Math.round(this.state.tickets));
        this.state = nextState;
    }

    private boolean updateTicketsIfDifferent(@Nonnull SemaphoreTicketBucket bucket, int newValue, int oldValue) {
        if (oldValue == newValue) {
            return false;
        }
        bucket.setPermits(newValue);
        return true;
    }

    @VisibleForTesting
    int getAcquireTimeoutSeconds() {
        return this.acquireTimeoutSeconds;
    }

    @VisibleForTesting
    double getCpuHistoricalWeighting() {
        return this.cpuHistoricalWeighting;
    }

    @VisibleForTesting
    int getMaxTickets() {
        return this.maxTickets;
    }

    @VisibleForTesting
    int getMinTickets() {
        return this.minTickets;
    }

    @VisibleForTesting
    int getFallbackTickets() {
        return this.fallbackTickets;
    }

    @VisibleForTesting
    int getFallbackThresholdSeconds() {
        return this.fallbackThresholdSeconds;
    }

    @VisibleForTesting
    double getTargetCpuLoad() {
        return this.targetCpuLoad;
    }

    @VisibleForTesting
    int getTotalTickets() {
        return (int)Math.round(this.state.tickets);
    }

    @VisibleForTesting
    double getTypicalCpuPerTicket() {
        return this.typicalCpuPerTicket;
    }

    @VisibleForTesting
    boolean isAdapting() {
        return this.state instanceof AdaptiveState || this.state instanceof StartingState;
    }

    @VisibleForTesting
    boolean isDisabled() {
        return this.state instanceof FixedState;
    }

    @VisibleForTesting
    boolean isPaused() {
        return this.state instanceof PausedState;
    }

    private void logCpuAvailable() {
        this.profileLogger.trace("[{}] The system CPU load is available again", (Object)this.resourceName);
        log.info("[{}] The system CPU load is available again. Adaptive throttling will now resume", (Object)this.resourceName);
    }

    private void logMissingCpuLoad(long duration) {
        this.profileLogger.trace("[{}] The system CPU load is still not available. Setting a fixed ticket limit of {}", (Object)this.resourceName, (Object)this.fallbackTickets);
        log.warn("[{}] The system CPU load has not been available for {}. A fixed ticket limit of {} is in place until the reading is next available", new Object[]{this.resourceName, ThrottlingOutputUtils.toHumanDuration(duration), this.fallbackTickets});
        if (SystemUtils.IS_OS_WINDOWS) {
            log.warn("[{}] Please ensure the Windows user running {} is a member of the local Windows group 'Performance Monitor Users' so that it is permitted to read the system CPU load", (Object)this.resourceName, (Object)ApplicationConstants.PRODUCT_NAME);
        }
    }

    private void logUpdate(AdaptiveState nextState, double cpuLoad, double targetTickets, TicketSummary ticketSummary) {
        if (this.profileLogger.isTraceEnabled()) {
            DecimalFormat briefly = ThrottlingOutputUtils.newBriefNumberFormat();
            long sinceLastUpdate = this.state.sinceUpdate(nextState.timestamp);
            int usedTickets = ticketSummary.getUsed();
            int totalTickets = ticketSummary.getTotal();
            int ticketQueueLength = ticketSummary.getQueuedRequests();
            OptionalDouble previousSmoothedCpu = this.state.smoothedCpuLoad;
            OptionalDouble nextSmoothedCpu = nextState.smoothedCpuLoad;
            this.profileLogger.trace("[{}] Ticket: {}->{}, Smoothed CPU: {}->{}, Inputs: [cpu={}, used-tickets={}/{}, ticket-queue-length={}, target-tickets={}, since-last-update={}]", new Object[]{this.resourceName, this.state.tickets, nextState.tickets, previousSmoothedCpu.isPresent() ? briefly.format(previousSmoothedCpu.getAsDouble()) : "unknown", briefly.format(nextSmoothedCpu.getAsDouble()), cpuLoad, usedTickets, totalTickets, ticketQueueLength, briefly.format(targetTickets), ThrottlingOutputUtils.toHumanDuration(sinceLastUpdate)});
        }
    }

    class StartingState
    extends State {
        public StartingState(double tickets, long timestamp) {
            super(OptionalDouble.empty(), tickets, timestamp);
        }

        @Override
        public State transition(long now, OptionalDouble cpuLoad, TicketSummary ticketSummary) {
            if (!cpuLoad.isPresent()) {
                return new PausedState(now + TimeUnit.SECONDS.toMillis(AdaptiveResourceThrottleStrategy.this.fallbackThresholdSeconds), this.tickets, now);
            }
            return this.transitionToAdaptive(now, cpuLoad.getAsDouble(), ticketSummary);
        }
    }

    abstract class State {
        final double tickets;
        final long timestamp;
        final OptionalDouble smoothedCpuLoad;

        public State(OptionalDouble smoothedCpuLoad, double tickets, long timestamp) {
            this.smoothedCpuLoad = smoothedCpuLoad;
            this.tickets = tickets;
            this.timestamp = timestamp;
        }

        public abstract State transition(long var1, OptionalDouble var3, TicketSummary var4);

        public long sinceUpdate(long when) {
            return when - this.timestamp;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("tickets", this.tickets).add("timestamp", this.timestamp).add("smoothedCpuLoad", this.smoothedCpuLoad.isPresent() ? Double.valueOf(this.smoothedCpuLoad.getAsDouble()) : "unknown").toString();
        }

        OptionalDouble smooth(double cpuLoad) {
            if (!this.smoothedCpuLoad.isPresent()) {
                return OptionalDouble.of(cpuLoad);
            }
            return OptionalDouble.of((1.0 - AdaptiveResourceThrottleStrategy.this.cpuHistoricalWeighting) * cpuLoad + AdaptiveResourceThrottleStrategy.this.cpuHistoricalWeighting * this.smoothedCpuLoad.getAsDouble());
        }

        State transitionToAdaptive(long now, double cpuLoad, TicketSummary ticketSummary) {
            int curUsedTickets = ticketSummary.getUsed();
            double smoothedCpuLoad = this.smooth(cpuLoad).getAsDouble();
            double targetTickets = (double)curUsedTickets + (AdaptiveResourceThrottleStrategy.this.targetCpuLoad - smoothedCpuLoad) / AdaptiveResourceThrottleStrategy.this.typicalCpuPerTicket;
            int tickets = Math.min(AdaptiveResourceThrottleStrategy.this.maxTickets, Math.max(AdaptiveResourceThrottleStrategy.this.minTickets, (int)Math.round(targetTickets)));
            AdaptiveState update = new AdaptiveState(smoothedCpuLoad, (double)tickets, now);
            AdaptiveResourceThrottleStrategy.this.logUpdate(update, cpuLoad, targetTickets, ticketSummary);
            return update;
        }

        State transitionToFixedIfAfter(long now, long deadline, long timestamp) {
            if (now < deadline) {
                return this;
            }
            AdaptiveResourceThrottleStrategy.this.logMissingCpuLoad(this.sinceUpdate(now));
            return new FixedState(now + FALLBACK_WARN_INTERVAL, (double)AdaptiveResourceThrottleStrategy.this.fallbackTickets, timestamp);
        }
    }

    class AdaptiveState
    extends State {
        private AdaptiveState(double smoothedCpuLoad, double tickets, long timestamp) {
            super(OptionalDouble.of(smoothedCpuLoad), tickets, timestamp);
        }

        @Override
        public State transition(long now, OptionalDouble cpuLoad, TicketSummary summary) {
            if (!cpuLoad.isPresent()) {
                return new PausedState(now + TimeUnit.SECONDS.toMillis(AdaptiveResourceThrottleStrategy.this.fallbackThresholdSeconds), this.tickets, now);
            }
            return this.transitionToAdaptive(now, cpuLoad.getAsDouble(), summary);
        }
    }

    class FixedState
    extends State {
        private final long nextWarning;

        public FixedState(long nextWarning, double tickets, long timestamp) {
            super(OptionalDouble.empty(), tickets, timestamp);
            this.nextWarning = nextWarning;
        }

        @Override
        public State transition(long now, OptionalDouble cpuLoad, TicketSummary ticketSummary) {
            if (!cpuLoad.isPresent()) {
                return this.transitionToFixedIfAfter(now, this.nextWarning, this.timestamp);
            }
            AdaptiveResourceThrottleStrategy.this.logCpuAvailable();
            return this.transitionToAdaptive(now, cpuLoad.getAsDouble(), ticketSummary);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("nextWarning", this.nextWarning).add("tickets", this.tickets).add("timestamp", this.timestamp).add("smoothedCpuLoad", this.smoothedCpuLoad.isPresent() ? Double.valueOf(this.smoothedCpuLoad.getAsDouble()) : "unknown").toString();
        }
    }

    class PausedState
    extends State {
        private final long fallbackAfter;

        public PausedState(long fallbackAfter, double tickets, long timestamp) {
            super(OptionalDouble.empty(), tickets, timestamp);
            this.fallbackAfter = fallbackAfter;
        }

        @Override
        public State transition(long now, OptionalDouble cpuLoad, TicketSummary ticketSummary) {
            if (!cpuLoad.isPresent()) {
                return this.transitionToFixedIfAfter(now, this.fallbackAfter, now);
            }
            AdaptiveResourceThrottleStrategy.this.profileLogger.trace("[{}] The system CPU load is available again", (Object)AdaptiveResourceThrottleStrategy.this.resourceName);
            log.info("[{}] The system CPU load is available again. Adaptive throttling will now resume", (Object)AdaptiveResourceThrottleStrategy.this.resourceName);
            return this.transitionToAdaptive(now, cpuLoad.getAsDouble(), ticketSummary);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("fallbackAfter", this.fallbackAfter).add("tickets", this.tickets).add("timestamp", this.timestamp).add("smoothedCpuLoad", this.smoothedCpuLoad.isPresent() ? Double.valueOf(this.smoothedCpuLoad.getAsDouble()) : "unknown").toString();
        }
    }
}

