/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.process.nu;

import com.atlassian.bitbucket.dmz.process.NioProcess;
import com.atlassian.bitbucket.dmz.process.NioStdioHandler;
import com.atlassian.bitbucket.i18n.I18nService;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.internal.process.NioProcessParameters;
import com.atlassian.bitbucket.internal.process.nu.NuNioProcess;
import com.atlassian.bitbucket.scm.CommandExitHandler;
import com.atlassian.bitbucket.scm.CommandFailedException;
import com.atlassian.bitbucket.scm.CommandResult;
import com.atlassian.bitbucket.scm.CommandSummary;
import com.atlassian.bitbucket.scm.CommandSummaryHandler;
import com.atlassian.bitbucket.scm.CommandTimeoutException;
import com.atlassian.bitbucket.scm.ProcessFailedException;
import com.google.common.base.Throwables;
import com.google.common.math.LongMath;
import com.zaxxer.nuprocess.NuProcess;
import com.zaxxer.nuprocess.NuProcessHandler;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BooleanSupplier;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NioNuProcessHandler<T>
implements NuProcessHandler {
    private static final Logger log = LoggerFactory.getLogger(NioNuProcessHandler.class);
    private static final Logger processLog = LoggerFactory.getLogger(NioProcess.class);
    private final String commandLine;
    private final long executionTimeout;
    private final CommandExitHandler exitHandler;
    private final CompletableFuture<T> future;
    private final I18nService i18nService;
    private final Duration idleInterval;
    private final AtomicReference<HandlerState> state;
    private final NioStdioHandler<T> stdioHandler;
    private final boolean throwOnNonZeroExit;
    private final ScheduledExecutorService timeoutExecutor;
    private final Path workDir;
    private NuNioProcess process;
    private long startTimestamp;
    private StringBuilder stderrBuffer;
    private long stderrRead;
    private long stdinWritten;
    private long stdoutRead;
    private Throwable thrown;
    private volatile long idleTimeout;
    private volatile ScheduledFuture<?> timeoutFuture;

    NioNuProcessHandler(I18nService i18nService, NioProcessParameters<T> parameters, ScheduledExecutorService timeoutExecutor) {
        this.i18nService = Objects.requireNonNull(i18nService, "i18nService");
        this.timeoutExecutor = Objects.requireNonNull(timeoutExecutor, "timeoutExecutor");
        this.commandLine = Objects.requireNonNull(parameters, "parameters").toString();
        this.exitHandler = parameters.getExitHandler();
        this.future = new CompletableFuture();
        this.state = new AtomicReference<HandlerState>(HandlerState.CREATED);
        this.stdioHandler = parameters.getStdioHandler();
        this.throwOnNonZeroExit = parameters.isThrowOnNonZeroExit();
        this.workDir = parameters.getWorkDir();
        Duration execution = parameters.getExecutionInterval();
        this.executionTimeout = NioNuProcessHandler.isUnlimited(execution) ? Long.MAX_VALUE : LongMath.saturatedAdd((long)System.currentTimeMillis(), (long)execution.toMillis());
        Duration idle = parameters.getIdleInterval();
        this.idleInterval = NioNuProcessHandler.isUnlimited(idle) ? Duration.ofSeconds(60L) : idle;
        this.idleTimeout = LongMath.saturatedAdd((long)System.currentTimeMillis(), (long)this.idleInterval.toMillis());
    }

    public void onExit(int exitCode) {
        ScheduledFuture<?> timeoutFuture;
        if (processLog.isDebugEnabled()) {
            processLog.debug("{}: [{}] exited {} in {}ms (stdin: {}, stdout: {}, stderr: {})", new Object[]{this.process.getPid(), this.commandLine, exitCode, System.currentTimeMillis() - this.startTimestamp, this.stdinWritten, this.stdoutRead, this.stderrRead});
        }
        if ((timeoutFuture = this.timeoutFuture) != null && !timeoutFuture.isDone()) {
            timeoutFuture.cancel(false);
        }
        try {
            this.stdioHandler.onExit(exitCode);
        }
        catch (Throwable t) {
            if (this.thrown != null) {
                t.addSuppressed(this.thrown);
            }
            this.thrown = t;
        }
        this.finish(exitCode);
    }

    public void onPreStart(NuProcess nuProcess) {
        this.process = new NuNioProcess(nuProcess, this.commandLine);
        this.invokeCallback(() -> this.stdioHandler.onPreStart((NioProcess)this.process));
    }

    public void onStart(NuProcess nuProcess) {
        this.maybeMarkStarted();
    }

    public void onStderr(@Nonnull ByteBuffer buffer, boolean closed) {
        if (buffer.hasRemaining()) {
            this.maybeMarkStarted();
        }
        int initialPosition = buffer.position();
        int initialRemaining = buffer.remaining();
        this.invokeCallback(() -> {
            if (!this.maybeDestroy()) {
                this.stdioHandler.onStderr(buffer, closed);
                this.resetIdleTimeout();
            }
        });
        if (buffer.hasRemaining() && buffer.position() == initialPosition) {
            if (this.stderrBuffer == null) {
                this.stderrBuffer = new StringBuilder();
            }
            this.stderrBuffer.append(StandardCharsets.UTF_8.decode(buffer));
        }
        this.stderrRead += (long)(initialRemaining - buffer.remaining());
    }

    public boolean onStdinReady(ByteBuffer buffer) {
        this.maybeMarkStarted();
        return this.invokeCallback(() -> {
            if (this.maybeDestroy()) {
                return false;
            }
            boolean result = this.stdioHandler.onStdinReady(buffer);
            this.resetIdleTimeout();
            this.stdinWritten += (long)buffer.remaining();
            return result;
        });
    }

    public void onStdout(@Nonnull ByteBuffer buffer, boolean closed) {
        if (buffer.hasRemaining()) {
            this.maybeMarkStarted();
        }
        BooleanSupplier callback = () -> {
            if (this.maybeDestroy()) {
                return false;
            }
            this.stdioHandler.onStdout(buffer, closed);
            this.resetIdleTimeout();
            if (!closed && buffer.remaining() == buffer.capacity()) {
                log.warn("{}: {}.onStdout has not consumed any data. Aborting [{}]", new Object[]{this.process.getPid(), this.stdioHandler.getClass().getSimpleName(), this.commandLine});
                throw new IllegalStateException("Process aborted due to unconsumed output");
            }
            return true;
        };
        int initialRemaining = buffer.remaining();
        if (this.invokeCallback(callback)) {
            this.stdoutRead += (long)(initialRemaining - buffer.remaining());
        } else if (buffer.hasRemaining()) {
            log.debug("{}: Discarding {} bytes of stdout", (Object)this.process.getPid(), (Object)buffer.remaining());
            buffer.position(buffer.position() + buffer.remaining());
        }
    }

    @Nonnull
    Future<T> asFuture() {
        if (this.state.get() == HandlerState.CREATED && !this.future.isDone()) {
            this.finish(-1001);
        }
        this.future.whenComplete((result, exception) -> this.process.terminateIfRunning());
        return this.future;
    }

    @Nullable
    T asResult() {
        if (!this.future.isDone()) {
            this.finish(-1002);
        }
        try {
            return this.future.getNow(null);
        }
        catch (CompletionException e) {
            throw (RuntimeException)e.getCause();
        }
    }

    private static boolean isUnlimited(Duration duration) {
        return duration == null || duration.isNegative() || duration.isZero();
    }

    private void callExitHandler(int exitCode) {
        String stderr;
        boolean canceled = this.isCanceled();
        String string = stderr = this.stderrBuffer == null ? null : StringUtils.chomp((String)this.stderrBuffer.toString());
        if (this.thrown == null) {
            if (this.isTimedOut()) {
                this.thrown = new CommandTimeoutException(this.i18nService.createKeyedMessage("bitbucket.scm.command.timeout", new Object[]{this.commandLine, exitCode}));
            } else if (!canceled && exitCode != 0 && this.throwOnNonZeroExit) {
                KeyedMessage message = StringUtils.isBlank((CharSequence)stderr) ? this.i18nService.createKeyedMessage("bitbucket.scm.command.failed", new Object[]{this.commandLine, exitCode}) : this.i18nService.createKeyedMessage("bitbucket.scm.command.failed.saying", new Object[]{this.commandLine, exitCode, stderr});
                this.thrown = new ProcessFailedException(message, exitCode);
            }
        }
        if (canceled) {
            this.exitHandler.onCancel(this.commandLine, exitCode, stderr, this.thrown);
        } else {
            this.exitHandler.onExit(this.commandLine, exitCode, stderr, this.thrown);
        }
    }

    private void ensureStarted() {
        if (this.state.get() == HandlerState.CREATED) {
            CommandFailedException e = new CommandFailedException(this.i18nService.createKeyedMessage("bitbucket.scm.command.not-started", new Object[]{this.commandLine}));
            if (this.thrown == null) {
                this.thrown = e;
            } else {
                this.thrown.addSuppressed((Throwable)e);
            }
        }
    }

    private void finish(int exitCode) {
        if (this.state.get() == HandlerState.FINISHED) {
            return;
        }
        this.ensureStarted();
        this.maybeSummarize(exitCode);
        try {
            this.callExitHandler(exitCode);
            this.future.complete(this.stdioHandler.getOutput());
        }
        catch (RuntimeException e) {
            if (this.thrown != null && !this.thrown.equals(e) && !Throwables.getCausalChain((Throwable)e).contains(this.thrown)) {
                e.addSuppressed(this.thrown);
            }
            this.thrown = e;
            this.future.completeExceptionally(this.thrown);
        }
        finally {
            this.state.set(HandlerState.FINISHED);
        }
    }

    private long getTimeoutDelay() {
        return Math.min(this.executionTimeout, this.idleTimeout) - System.currentTimeMillis();
    }

    private void invokeCallback(Runnable callback) {
        this.invokeCallback(() -> {
            callback.run();
            return true;
        });
    }

    private boolean invokeCallback(BooleanSupplier callback) {
        try {
            return callback.getAsBoolean();
        }
        catch (Throwable t) {
            if (this.thrown == null) {
                this.thrown = t;
            } else {
                this.thrown.addSuppressed(t);
            }
            if (log.isDebugEnabled()) {
                log.debug("{}: Destroying [{}] after handler failure", new Object[]{this.process.getPid(), this.commandLine, t});
            }
            this.process.destroyIfRunning();
            return false;
        }
    }

    private boolean isCanceled() {
        return this.future.isCancelled() || this.process.isCanceled();
    }

    private boolean isTimedOut() {
        long now = System.currentTimeMillis();
        return now > this.executionTimeout || now > this.idleTimeout;
    }

    private boolean maybeDestroy() {
        boolean canceled = this.isCanceled();
        boolean timedOut = this.isTimedOut();
        if (this.thrown != null || canceled || timedOut) {
            if (log.isDebugEnabled()) {
                log.debug("{}: Dropping [{}] callback invocation (Canceled: {}; Timed out: {})", new Object[]{this.process.getPid(), this.commandLine, canceled, timedOut, this.thrown});
            }
            this.process.destroyIfRunning();
            return true;
        }
        return false;
    }

    private synchronized void maybeMarkStarted() {
        if (this.state.compareAndSet(HandlerState.CREATED, HandlerState.STARTED)) {
            if (processLog.isTraceEnabled()) {
                processLog.trace("{}: [{}] started (cwd: {})", new Object[]{this.process.getPid(), this.commandLine, this.workDir});
            }
            this.startTimestamp = System.currentTimeMillis();
            this.invokeCallback(() -> this.stdioHandler.onStart((NioProcess)this.process));
            this.maybeScheduleTimeout();
        }
    }

    private void maybeScheduleTimeout() {
        if (this.thrown == null && !this.isCanceled()) {
            long delay = Math.max(this.getTimeoutDelay(), 5000L);
            this.timeoutFuture = this.timeoutExecutor.schedule(new TimeoutTask(), delay, TimeUnit.MILLISECONDS);
        }
    }

    private void maybeSummarize(int exitCode) {
        if (this.stdioHandler instanceof CommandSummaryHandler) {
            CommandResult result = this.isCanceled() ? CommandResult.CANCELED : (exitCode == 0 && this.thrown == null && !this.isTimedOut() ? CommandResult.SUCCEEDED : CommandResult.FAILED);
            CommandSummary summary = new CommandSummary.Builder(result).build();
            try {
                ((CommandSummaryHandler)this.stdioHandler).onComplete(summary);
            }
            catch (Exception e) {
                if (this.thrown == null) {
                    this.thrown = e;
                }
                this.thrown.addSuppressed(e);
            }
        }
    }

    private void resetIdleTimeout() {
        this.idleTimeout = System.currentTimeMillis() + this.idleInterval.toMillis();
    }

    private static enum HandlerState {
        CREATED,
        FINISHED,
        STARTED;

    }

    private class TimeoutTask
    implements Runnable {
        private TimeoutTask() {
        }

        @Override
        public void run() {
            long nextDelay = NioNuProcessHandler.this.getTimeoutDelay();
            if (nextDelay > 0L) {
                NioNuProcessHandler.this.timeoutFuture = NioNuProcessHandler.this.timeoutExecutor.schedule(this, nextDelay, TimeUnit.MILLISECONDS);
            } else {
                NioNuProcessHandler.this.timeoutFuture = null;
                NioNuProcessHandler.this.process.destroyIfRunning();
            }
        }
    }
}

