/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.gpg;

import com.atlassian.bitbucket.dmz.signature.verification.SignatureState;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerificationPublicKey;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerificationRequest;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerificationResult;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerifier;
import com.atlassian.bitbucket.internal.gpg.GpgPublicKey;
import com.atlassian.bitbucket.internal.gpg.GpgSystemSigningKeyProvider;
import com.atlassian.bitbucket.internal.gpg.PublicKeySupplier;
import com.atlassian.bitbucket.internal.gpg.bc.SignatureUtil;
import com.atlassian.bitbucket.server.ApplicationPropertiesService;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.UserService;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import jakarta.annotation.Nonnull;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.regex.Pattern;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

@Component
public class GpgSignatureVerifier
implements SignatureVerifier {
    @VisibleForTesting
    static final String IS_SYSTEM_SIGNED = "is_system_signed";
    private static final int DEFAULT_KEY_CACHE_SIZE = 250;
    private static final Pattern PGP_HEADER = Pattern.compile("-----BEGIN PGP ([^\\s\\d]+)-----[\\s\\S]*");
    private static final String PROP_KEY_CACHE_SIZE = "plugin.gpg.key.cache.max";
    private static final String SIGNATURE_TYPE = "GPG";
    private static final Logger log = LoggerFactory.getLogger(GpgSignatureVerifier.class);
    private final GpgSystemSigningKeyProvider gpgSystemSigningKeyProvider;
    private final LoadingCache<Long, Map<PGPPublicKey, ApplicationUser>> keysCache;
    private final UserService userService;
    private final PGPContentVerifierBuilderProvider verifierBuilderProvider;

    public GpgSignatureVerifier(GpgSystemSigningKeyProvider gpgSystemSigningKeyProvider, ApplicationPropertiesService propertiesService, final PublicKeySupplier publicKeySupplier, UserService userService, @Qualifier(value="jcaPGPContentVerifierBuilderProvider") PGPContentVerifierBuilderProvider verifierBuilderProvider) {
        this.gpgSystemSigningKeyProvider = gpgSystemSigningKeyProvider;
        this.userService = userService;
        this.verifierBuilderProvider = verifierBuilderProvider;
        this.keysCache = CacheBuilder.newBuilder().maximumSize((long)propertiesService.getPluginProperty(PROP_KEY_CACHE_SIZE, 250)).build((CacheLoader)new CacheLoader<Long, Map<PGPPublicKey, ApplicationUser>>(this){

            public Map<PGPPublicKey, ApplicationUser> load(@Nonnull Long key) {
                return publicKeySupplier.getPublicKey(key);
            }
        });
    }

    @Nonnull
    public Optional<SignatureVerificationResult> verifySignature(@Nonnull SignatureVerificationRequest request) {
        Objects.requireNonNull(request, "request");
        String signature = request.getSignature();
        if (!this.isGpgSignature(signature)) {
            return Optional.empty();
        }
        Optional<PGPSignature> maybeSignature = SignatureUtil.parseSignature(signature.getBytes(StandardCharsets.UTF_8));
        if (!maybeSignature.isPresent()) {
            return Optional.of(new SignatureVerificationResult.Builder(SignatureState.ERROR).build());
        }
        PGPSignature pgpSignature = maybeSignature.get();
        long keyId = pgpSignature.getKeyID();
        SignatureState result = SignatureState.PUBLIC_KEY_NOT_FOUND;
        String signedContent = request.getSignedContent();
        PGPPublicKey systemPublicKey = this.gpgSystemSigningKeyProvider.getPublicKey().orElse(null);
        if (systemPublicKey != null && systemPublicKey.getKeyID() == keyId) {
            result = this.verifySignature(signedContent.getBytes(StandardCharsets.UTF_8), pgpSignature, systemPublicKey);
            return Optional.of(new SignatureVerificationResult.Builder(result).owner((ApplicationUser)this.userService.getSystemServiceUser()).signatureType(SIGNATURE_TYPE).verificationPublicKey((SignatureVerificationPublicKey)new GpgPublicKey(systemPublicKey)).signatureMetadata(Collections.singletonMap(IS_SYSTEM_SIGNED, "true")).build());
        }
        Map<PGPPublicKey, ApplicationUser> publicKeysAndUser = this.getPublicKeysAndUser(keyId);
        if (publicKeysAndUser.isEmpty()) {
            return Optional.of(new SignatureVerificationResult.Builder(SignatureState.PUBLIC_KEY_NOT_FOUND).signatureType(SIGNATURE_TYPE).build());
        }
        for (PGPPublicKey pgpPublicKey : publicKeysAndUser.keySet()) {
            result = this.verifySignature(signedContent.getBytes(StandardCharsets.UTF_8), pgpSignature, pgpPublicKey);
            if (!result.isVerified()) continue;
            return Optional.of(new SignatureVerificationResult.Builder(result).owner(publicKeysAndUser.get(pgpPublicKey)).signatureType(SIGNATURE_TYPE).verificationPublicKey((SignatureVerificationPublicKey)new GpgPublicKey(pgpPublicKey)).build());
        }
        return Optional.of(new SignatureVerificationResult.Builder(result).signatureType(SIGNATURE_TYPE).build());
    }

    private Date getExpireDate(PGPPublicKey key) {
        long expirySeconds = key.getValidSeconds();
        if (expirySeconds == 0L) {
            return null;
        }
        return new Date(key.getCreationTime().getTime() + 1000L * expirySeconds);
    }

    private Map<PGPPublicKey, ApplicationUser> getPublicKeysAndUser(long keyId) {
        try {
            return (Map)this.keysCache.get((Object)keyId);
        }
        catch (ExecutionException e) {
            log.warn("Could not load GPG keys for key {}", (Object)Long.toHexString(keyId), (Object)e);
            return Collections.emptyMap();
        }
    }

    private boolean isGpgSignature(String signature) {
        return PGP_HEADER.matcher(signature).matches();
    }

    private SignatureState verifySignature(byte[] signedContent, PGPSignature pgpSignature, PGPPublicKey pgpPublicKey) {
        try {
            pgpSignature.init(this.verifierBuilderProvider, pgpPublicKey);
        }
        catch (PGPException e) {
            return SignatureState.ERROR;
        }
        pgpSignature.update(signedContent);
        try {
            if (pgpSignature.verify()) {
                if (pgpPublicKey.hasRevocation()) {
                    return SignatureState.GOOD_BUT_REVOKED;
                }
                Date keyExpiry = this.getExpireDate(pgpPublicKey);
                if (keyExpiry != null && keyExpiry.getTime() < System.currentTimeMillis()) {
                    return pgpSignature.getCreationTime().before(keyExpiry) ? SignatureState.GOOD_BUT_OUT_OF_DATE : SignatureState.GOOD_BUT_MADE_AFTER_EXPIRY;
                }
                return SignatureState.GOOD;
            }
            return SignatureState.BAD;
        }
        catch (PGPException pGPException) {
            return SignatureState.ERROR;
        }
    }
}

