/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.gpg.bc;

import com.atlassian.bitbucket.gpg.GpgKey;
import com.atlassian.bitbucket.internal.gpg.SimpleGpgKey;
import com.atlassian.bitbucket.util.MoreCollectors;
import com.atlassian.bitbucket.util.MoreStreams;
import com.google.common.collect.ImmutableList;
import jakarta.annotation.Nonnull;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.jcajce.JcaPGPObjectFactory;
import org.bouncycastle.openpgp.jcajce.JcaPGPSecretKeyRingCollection;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SignatureUtil {
    private static final Logger log = LoggerFactory.getLogger(SignatureUtil.class);
    private static final Pattern EMAIL_PATTERN = Pattern.compile(".*<([^<>]+)>");

    @Nonnull
    public static Optional<PGPPublicKey> parsePublicKey(@Nonnull String publicKeyText, long keyId) {
        return SignatureUtil.streamPublicKeys(publicKeyText).filter(key -> key.getKeyID() == keyId).findFirst();
    }

    @Nonnull
    public static List<GpgKey> parseGpgKeyRing(@Nonnull String keyText) throws IOException {
        return (List)SignatureUtil.streamPublicKeyRings(keyText).flatMap(keyRing -> SignatureUtil.toGpgKeys(keyText, keyRing).stream()).collect(MoreCollectors.toImmutableList());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    public static Optional<PGPPublicKey> parsePublicKeyFromSecretKey(@Nonnull String secretKeyText) {
        Objects.requireNonNull(secretKeyText, "secretKeyText");
        ByteArrayInputStream secretKeyStream = new ByteArrayInputStream(Objects.requireNonNull(secretKeyText, "secretKeyText").getBytes(StandardCharsets.UTF_8));
        try (InputStream decoderStream = PGPUtil.getDecoderStream(secretKeyStream);){
            PGPSecretKey key;
            JcaPGPSecretKeyRingCollection pgpSec = new JcaPGPSecretKeyRingCollection(decoderStream);
            Iterator<PGPSecretKeyRing> keyRingIter = pgpSec.getKeyRings();
            do {
                if (!keyRingIter.hasNext()) return Optional.empty();
                PGPSecretKeyRing keyRing = keyRingIter.next();
                key = null;
                Iterator<PGPSecretKey> keyIter = keyRing.getSecretKeys();
                while (key == null && keyIter.hasNext()) {
                    PGPSecretKey keyCandidate = keyIter.next();
                    if (!keyCandidate.isSigningKey() || !keyCandidate.isMasterKey()) continue;
                    key = keyCandidate;
                }
            } while (key == null);
            Optional<PGPPublicKey> optional = Optional.of(key.getPublicKey());
            return optional;
        }
        catch (IOException | PGPException e) {
            log.error("Unable to parse secret key", (Throwable)e);
        }
        return Optional.empty();
    }

    @Nonnull
    public static Optional<String> parsePublicKeyToString(@Nonnull PGPPublicKey publicKey) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try (ArmoredOutputStream armoredOutputStream = new ArmoredOutputStream(outputStream);){
            armoredOutputStream.setHeader("Version", null);
            publicKey.encode(armoredOutputStream);
        }
        catch (IOException e) {
            log.error("Error parsing public key to string", (Throwable)e);
        }
        try {
            return Optional.ofNullable(outputStream.toString(StandardCharsets.UTF_8.name()));
        }
        catch (UnsupportedEncodingException e) {
            log.error("Unsupported encoding while parsing public key to string", (Throwable)e);
            return Optional.empty();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    public static Optional<PGPSignature> parseSignature(@Nonnull byte[] signature) {
        ByteArrayInputStream signatureStream = new ByteArrayInputStream(Objects.requireNonNull(signature, "signature"));
        try (InputStream decoderStream = PGPUtil.getDecoderStream(signatureStream);){
            JcaPGPObjectFactory gpgFactory = new JcaPGPObjectFactory(decoderStream);
            PGPSignatureList pgpSignatures = (PGPSignatureList)gpgFactory.nextObject();
            if (pgpSignatures == null || pgpSignatures.size() == 0) {
                Optional<PGPSignature> optional2 = Optional.empty();
                return optional2;
            }
            Optional<PGPSignature> optional = Optional.of(pgpSignatures.get(0));
            return optional;
        }
        catch (IOException | RuntimeException e) {
            log.error("Error while parsing signature", (Throwable)e);
            return Optional.empty();
        }
    }

    @Nonnull
    public static Stream<PGPPublicKeyRing> streamPublicKeyRings(@Nonnull String publicKeyText) {
        try {
            ByteArrayInputStream publicKeyStream = new ByteArrayInputStream(Objects.requireNonNull(publicKeyText, "publicKeyText").getBytes(StandardCharsets.UTF_8));
            InputStream decoderStream = PGPUtil.getDecoderStream(publicKeyStream);
            JcaPGPObjectFactory gpgFactory = new JcaPGPObjectFactory(decoderStream);
            return MoreStreams.streamIterable((Iterable)((Object)gpgFactory)).filter(PGPPublicKeyRing.class::isInstance).map(PGPPublicKeyRing.class::cast);
        }
        catch (IOException e) {
            log.error("Unable to parse stored public key", (Throwable)e);
            return Stream.empty();
        }
    }

    @Nonnull
    public static Stream<PGPPublicKey> streamPublicKeys(@Nonnull String publicKeyText) {
        return SignatureUtil.streamPublicKeyRings(publicKeyText).flatMap(MoreStreams::streamIterable);
    }

    private static List<GpgKey> toGpgKeys(String keyText, PGPPublicKeyRing keyRing) {
        ImmutableList.Builder resultBuilder = ImmutableList.builder();
        SimpleGpgKey.Builder keyBuilder = null;
        Iterator<PGPPublicKey> iterator = keyRing.iterator();
        while (iterator.hasNext()) {
            PGPPublicKey publicKey = iterator.next();
            if (publicKey.isMasterKey()) {
                if (keyBuilder != null) {
                    resultBuilder.add((Object)keyBuilder.build());
                }
                keyBuilder = new SimpleGpgKey.Builder().emailAddress(SignatureUtil.getEmail(publicKey)).expiryDate(SignatureUtil.getExpiryDate(publicKey)).fingerprint(Hex.toHexString((byte[])publicKey.getFingerprint())).text(keyText);
                continue;
            }
            if (keyBuilder != null) {
                keyBuilder.subKey(Hex.toHexString((byte[])publicKey.getFingerprint()), SignatureUtil.getExpiryDate(publicKey));
                continue;
            }
            log.warn("Found a GPG sub-key without a master key");
        }
        if (keyBuilder != null) {
            resultBuilder.add((Object)keyBuilder.build());
        }
        return resultBuilder.build();
    }

    private static String getEmail(PGPPublicKey key) {
        Iterator<String> it = key.getUserIDs();
        while (it.hasNext()) {
            Matcher matcher;
            String userId = it.next();
            if (userId == null || !(matcher = EMAIL_PATTERN.matcher(userId)).matches()) continue;
            return matcher.group(1);
        }
        return null;
    }

    private static Date getExpiryDate(PGPPublicKey key) {
        return key.getValidSeconds() == 0L ? null : new Date(key.getCreationTime().getTime() + TimeUnit.SECONDS.toMillis(key.getValidSeconds()));
    }
}

