/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.platform.security.file;

import com.atlassian.platform.security.file.DetectableFileType;
import com.atlassian.platform.security.file.FileContentProvider;
import com.atlassian.platform.security.file.FileMetadata;
import com.atlassian.platform.security.file.FileValidationError;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.tika.config.TikaConfig;
import org.apache.tika.exception.TikaException;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.Metadata;

public class FileSafetyChecker {
    private static final Set<Character> ILLEGAL_WINDOWS_FILE_NAMING_CHARS = Set.of(Character.valueOf('?'), Character.valueOf('\"'), Character.valueOf('/'), Character.valueOf('<'), Character.valueOf('>'), Character.valueOf('*'), Character.valueOf('|'), Character.valueOf(':'), Character.valueOf('\\'));
    private static final Set<Character> ILLEGAL_UNIX_FILE_NAMING_CHARS = Set.of(Character.valueOf('/'), Character.valueOf('%'));
    private static final Set<Character> NOT_RECOMMENDED_NAMING_CHARS = Set.of(Character.valueOf('+'), Character.valueOf('&'), Character.valueOf('['), Character.valueOf(']'), Character.valueOf('^'), Character.valueOf('`'), Character.valueOf('{'), Character.valueOf('}'), Character.valueOf('('), Character.valueOf(')'));
    private static final Set<String> ILLEGAL_WINDOWS_FILE_NAMES = Set.of("CON", "PRN", "AUX", "NUL", "COM0", "COM1", "COM2", "COM3", "COM4", "COM5", "COM6", "COM7", "COM8", "COM9", "COM\u00b9", "COM\u00b2", "COM\u00b3", "LPT0", "LPT1", "LPT2", "LPT3", "LPT4", "LPT5", "LPT6", "LPT7", "LPT8", "LPT9", "LPT\u00b9", "LPT\u00b2", "LPT\u00b3");
    private static final TikaConfig TIKA_CONFIG;

    public static Collection<FileValidationError> validateBeforeUpload(FileMetadata fileMetadata, long maxFileSize, boolean strictFilenameCheck, Collection<DetectableFileType> expectedTypes) {
        ArrayList<FileValidationError> errors = new ArrayList<FileValidationError>();
        errors.addAll(FileSafetyChecker.validateFileSize(fileMetadata, maxFileSize));
        errors.addAll(FileSafetyChecker.validateFileName(fileMetadata, strictFilenameCheck, expectedTypes));
        errors.addAll(FileSafetyChecker.validateProvidedContentType(fileMetadata, expectedTypes));
        return errors;
    }

    public static Collection<FileValidationError> validateAfterUpload(FileContentProvider contentProvider, Collection<DetectableFileType> expectedTypes) throws IOException {
        ArrayList<FileValidationError> errors = new ArrayList<FileValidationError>();
        errors.addAll(FileSafetyChecker.validateDetectedContentType(contentProvider, expectedTypes));
        return errors;
    }

    private static Collection<FileValidationError> validateFileSize(FileMetadata fileMetadata, long maxFileSize) {
        if (maxFileSize > 0L && fileMetadata.fileSize() > maxFileSize) {
            return List.of(new FileValidationError(FileValidationError.Type.TOO_LARGE, String.valueOf(maxFileSize)));
        }
        return List.of();
    }

    private static Collection<FileValidationError> validateFileName(FileMetadata fileMetadata, boolean strictFilenameCheck, Collection<DetectableFileType> expectedTypes) {
        String fileName = fileMetadata.fileName();
        if (fileName == null || fileName.isEmpty()) {
            return List.of(new FileValidationError(FileValidationError.Type.NAME_EMPTY, null));
        }
        int maxFileNameLength = 150;
        if (fileName.length() > 150) {
            return List.of(new FileValidationError(FileValidationError.Type.NAME_TOO_LONG, String.valueOf(150)));
        }
        if (fileName.endsWith(" ") || fileName.endsWith(".")) {
            return List.of(new FileValidationError(FileValidationError.Type.NAME_ILLEGAL_SUFFIX, String.valueOf(fileName.charAt(fileName.length() - 1))));
        }
        if (ILLEGAL_WINDOWS_FILE_NAMES.stream().anyMatch(fileName::equalsIgnoreCase)) {
            return List.of(new FileValidationError(FileValidationError.Type.NAME_RESERVED, null));
        }
        List<FileValidationError> charErrorList = FileSafetyChecker.validateChars(fileName, strictFilenameCheck);
        if (!charErrorList.isEmpty()) {
            return charErrorList;
        }
        List<FileValidationError> extensionTypeErrorList = FileSafetyChecker.validateExtension(fileMetadata, expectedTypes);
        if (!extensionTypeErrorList.isEmpty()) {
            return extensionTypeErrorList;
        }
        return List.of();
    }

    private static List<FileValidationError> validateChars(String fileName, boolean strictFilenameCheck) {
        return fileName.codePoints().mapToObj(codePoint -> {
            int type = Character.getType(codePoint);
            if (type == 16 || type == 18 || type == 0) {
                return new FileValidationError(FileValidationError.Type.NAME_ILLEGAL_CHARACTER, String.valueOf(Character.toChars(codePoint)));
            }
            if (Character.isISOControl(codePoint)) {
                return new FileValidationError(FileValidationError.Type.NAME_CONTROL_CHARACTER, String.valueOf(Character.toChars(codePoint)));
            }
            if (!Character.isBmpCodePoint(codePoint)) {
                return null;
            }
            char symbol = (char)codePoint;
            if (ILLEGAL_WINDOWS_FILE_NAMING_CHARS.contains(Character.valueOf(symbol)) || ILLEGAL_UNIX_FILE_NAMING_CHARS.contains(Character.valueOf(symbol)) || strictFilenameCheck && NOT_RECOMMENDED_NAMING_CHARS.contains(Character.valueOf(symbol))) {
                return new FileValidationError(FileValidationError.Type.NAME_ILLEGAL_CHARACTER, String.valueOf(symbol));
            }
            return null;
        }).filter(Objects::nonNull).toList();
    }

    private static List<FileValidationError> validateExtension(FileMetadata fileMetadata, Collection<DetectableFileType> expectedTypes) {
        if (!expectedTypes.isEmpty()) {
            int dotIndex = fileMetadata.fileName().lastIndexOf(46);
            String extension = dotIndex == -1 ? "" : fileMetadata.fileName().substring(dotIndex + 1);
            boolean extensionMatches = expectedTypes.stream().flatMap(fileType -> fileType.getFileExtensions().stream()).anyMatch(extension::equalsIgnoreCase);
            if (!extensionMatches) {
                return List.of(new FileValidationError(FileValidationError.Type.NAME_EXTENSION_MISMATCH, extension));
            }
        }
        return List.of();
    }

    private static Collection<FileValidationError> validateProvidedContentType(FileMetadata fileMetadata, Collection<DetectableFileType> expectedTypes) {
        if (!expectedTypes.isEmpty()) {
            boolean contentTypeMatches = expectedTypes.stream().flatMap(fileType -> fileType.getContentTypes().stream()).anyMatch(fileMetadata.contentType()::equalsIgnoreCase);
            if (!contentTypeMatches) {
                return List.of(new FileValidationError(FileValidationError.Type.PROVIDED_CONTENT_TYPE_MISMATCH, null));
            }
        }
        return List.of();
    }

    private static Collection<FileValidationError> validateDetectedContentType(FileContentProvider contentProvider, Collection<DetectableFileType> expectedTypes) throws IOException {
        if (expectedTypes.isEmpty()) {
            return List.of();
        }
        String detectedContentType = contentProvider.processFile(inputStream -> TIKA_CONFIG.getDetector().detect((InputStream)TikaInputStream.get((InputStream)inputStream), new Metadata()).toString());
        boolean contentTypeMatches = expectedTypes.stream().flatMap(fileType -> fileType.getContentTypes().stream()).anyMatch(detectedContentType::equalsIgnoreCase);
        if (!contentTypeMatches) {
            return List.of(new FileValidationError(FileValidationError.Type.DETECTED_CONTENT_TYPE_MISMATCH, detectedContentType));
        }
        return List.of();
    }

    static {
        try {
            TIKA_CONFIG = new TikaConfig();
        }
        catch (IOException | TikaException e) {
            throw new RuntimeException(String.format("Couldn't instantiate %s - problem creating %s", FileSafetyChecker.class.getName(), TikaConfig.class.getName()), e);
        }
    }
}

