/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.hikari;

import com.atlassian.bitbucket.util.Drainable;
import com.atlassian.bitbucket.util.ForcedDrainable;
import com.atlassian.stash.internal.db.DbType;
import com.atlassian.stash.internal.jdbc.ConnectionTracker;
import com.atlassian.stash.internal.util.StackException;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import jakarta.annotation.Nonnull;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.SQLTransientConnectionException;
import java.sql.Statement;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExtendedHikariDataSource
extends HikariDataSource
implements Drainable,
ForcedDrainable {
    private static final long DEFAULT_DRAIN_POLL_INTERVAL = TimeUnit.SECONDS.toMillis(2L);
    private static final Logger log = LoggerFactory.getLogger(ExtendedHikariDataSource.class);
    private long drainPollInterval;
    private ConnectionTracker tracker;

    public ExtendedHikariDataSource(HikariConfig config, ConnectionTracker tracker) {
        super(config);
        this.tracker = tracker;
        this.drainPollInterval = DEFAULT_DRAIN_POLL_INTERVAL;
    }

    public void close() {
        super.close();
        String driverClassName = this.getDriverClassName();
        if (driverClassName != null && DbType.isInternal((String)driverClassName)) {
            this.shutdownDatabase();
        }
    }

    public boolean drain(long timeout, @Nonnull TimeUnit unit) {
        Preconditions.checkArgument((timeout >= 0L ? 1 : 0) != 0, (Object)"timeout must be non-negative");
        Objects.requireNonNull(unit, "unit");
        return this.drainInterruptibly(timeout, unit) == DrainResult.DRAINED || this.isDrained();
    }

    public boolean forceDrain(long timeout, @Nonnull TimeUnit unit) {
        Preconditions.checkArgument((timeout >= 0L ? 1 : 0) != 0, (Object)"timeout must be non-negative");
        Objects.requireNonNull(unit, "unit");
        if (this.isDrained()) {
            return true;
        }
        log.info("Forcibly draining the database pool");
        log.debug("{} connections still leased. Owning threads will be interrupted with a {} {} delay", new Object[]{this.getConnectionCount(), timeout, unit});
        this.interruptThreadsWithConnections();
        switch (this.drainInterruptibly(timeout, unit).ordinal()) {
            case 1: {
                return this.isDrained();
            }
            case 0: {
                return true;
            }
        }
        log.debug("{} connections still leased. Connections will be rolled back and closed", (Object)this.getConnectionCount());
        this.forciblyCloseConnections();
        int totalLeased = this.getConnectionCount();
        log.info("{} connections still leased. Forced draining has {}", (Object)totalLeased, (Object)(totalLeased == 0 ? "succeeded" : "failed"));
        return totalLeased == 0;
    }

    public Connection getConnection() throws SQLException {
        try {
            return this.tracker.register(super.getConnection());
        }
        catch (SQLTransientConnectionException e) {
            this.tracker.onConnectionRejected();
            throw e;
        }
    }

    @VisibleForTesting
    void setDrainPollInterval(long drainPollInterval) {
        this.drainPollInterval = drainPollInterval;
    }

    private static String threadName(Thread thread) {
        return thread == null ? "<unknown>" : thread.getName();
    }

    private DrainResult drainInterruptibly(long timeout, @Nonnull TimeUnit unit) {
        long start = System.currentTimeMillis();
        long end = start + unit.toMillis(timeout);
        log.debug("Draining the database pool");
        int leased = this.getConnectionCount();
        while (leased > 0) {
            long tilEnd = end - System.currentTimeMillis();
            long interval = Math.min(this.drainPollInterval, tilEnd);
            if (tilEnd <= 0L) {
                log.debug("The database pool did not drain in {} {}; {} connections are still leased", new Object[]{timeout, unit, leased});
                return DrainResult.TIMED_OUT;
            }
            log.debug("{} connections still leased; waiting {}ms", (Object)leased, (Object)interval);
            try {
                Thread.sleep(interval);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.debug("Interrupted while waiting for the database pool to drain");
                return DrainResult.INTERRUPTED;
            }
            leased = this.getConnectionCount();
        }
        log.debug("The database pool has drained in {}ms", (Object)(System.currentTimeMillis() - start));
        return DrainResult.DRAINED;
    }

    private void forciblyCloseConnections() {
        this.tracker.forEach((connection, thread) -> {
            String threadName = ExtendedHikariDataSource.threadName(thread);
            try {
                log.info("Forcibly closing database connection on thread [{}]", (Object)threadName);
                connection.close();
            }
            catch (Exception e) {
                log.info("Failed to close database connection on thread [{}]", (Object)threadName, (Object)e);
            }
        });
    }

    private int getConnectionCount() {
        return this.tracker.getConnectionCount();
    }

    private void interruptThreadsWithConnections() {
        this.tracker.forEach((connection, thread) -> {
            log.debug("Interrupting thread [{}] to facilitate draining the database pool", (Object)ExtendedHikariDataSource.threadName(thread), (Object)new StackException(thread));
            thread.interrupt();
        });
    }

    private boolean isDrained() {
        return this.getConnectionCount() == 0;
    }

    private void shutdownDatabase() {
        try (Connection connection = DriverManager.getConnection(this.getJdbcUrl(), this.getUsername(), this.getPassword());
             Statement statement = connection.createStatement();){
            statement.execute("SHUTDOWN");
        }
        catch (SQLException e) {
            log.warn("Internal database could not be shutdown", (Throwable)e);
        }
    }

    private static enum DrainResult {
        DRAINED,
        INTERRUPTED,
        TIMED_OUT;

    }
}

