/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.key.ssh.verify;

import com.atlassian.bitbucket.dmz.signature.verification.SignatureState;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerificationRequest;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerificationResult;
import com.atlassian.bitbucket.dmz.signature.verification.SignatureVerifier;
import com.atlassian.bitbucket.internal.key.ssh.verify.DecodedSshSignature;
import com.atlassian.bitbucket.internal.key.ssh.verify.SshSignatureVerificationException;
import com.atlassian.bitbucket.internal.ssh.utils.KeyUtils;
import com.atlassian.bitbucket.server.ApplicationPropertiesService;
import com.atlassian.bitbucket.server.Feature;
import com.atlassian.bitbucket.server.FeatureManager;
import com.atlassian.bitbucket.server.StandardFeature;
import com.atlassian.bitbucket.ssh.KeyType;
import com.atlassian.bitbucket.ssh.SshKey;
import com.atlassian.bitbucket.ssh.SshKeyService;
import com.atlassian.bitbucket.ssh.StandardDigests;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import jakarta.annotation.Nonnull;
import java.io.IOException;
import java.security.PublicKey;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import org.apache.sshd.common.config.keys.u2f.SkED25519PublicKey;
import org.apache.sshd.common.config.keys.u2f.SkEcdsaPublicKey;
import org.apache.sshd.common.signature.BuiltinSignatures;
import org.apache.sshd.common.signature.Signature;
import org.apache.sshd.common.signature.SignatureFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class SshSignatureVerifier
implements SignatureVerifier {
    @VisibleForTesting
    static final int DEFAULT_KEY_CACHE_SIZE = 250;
    @VisibleForTesting
    static final String PROP_KEY_CACHE_SIZE = "plugin.ssh.key.cache.max";
    private static final String FOOTER = "-----END SSH SIGNATURE-----\n";
    private static final String HEADER = "-----BEGIN SSH SIGNATURE-----";
    private static final Logger log = LoggerFactory.getLogger(SshSignatureVerifier.class);
    private static final String SIGNATURE_TYPE = "SSH";
    private static final Optional<SignatureVerificationResult> ERROR_RESULT = Optional.of(new SignatureVerificationResult.Builder(SignatureState.ERROR).signatureType("SSH").build());
    private static final Optional<SignatureVerificationResult> PUBLIC_KEY_NOT_FOUND_RESULT = Optional.of(new SignatureVerificationResult.Builder(SignatureState.PUBLIC_KEY_NOT_FOUND).signatureType("SSH").build());
    private final FeatureManager featureManager;
    private final LoadingCache<PublicKey, Optional<SshKey>> keysCache;

    public SshSignatureVerifier(@Nonnull FeatureManager featureManager, @Nonnull ApplicationPropertiesService propertiesService, final @Nonnull SshKeyService sshKeyService) {
        this.featureManager = Objects.requireNonNull(featureManager, "featureManager");
        Objects.requireNonNull(propertiesService, "propertiesService");
        Objects.requireNonNull(sshKeyService, "sshKeyService");
        this.keysCache = CacheBuilder.newBuilder().maximumSize((long)propertiesService.getPluginProperty(PROP_KEY_CACHE_SIZE, 250)).build((CacheLoader)new CacheLoader<PublicKey, Optional<SshKey>>(this){

            public Optional<SshKey> load(@Nonnull PublicKey publicKey) {
                return Optional.ofNullable(sshKeyService.getByPublicKey(publicKey));
            }
        });
    }

    @Nonnull
    public Optional<SignatureVerificationResult> verifySignature(@Nonnull SignatureVerificationRequest request) {
        Objects.requireNonNull(request, "request");
        if (!request.getSignature().startsWith(HEADER) || !request.getSignature().endsWith(FOOTER)) {
            log.trace("The SSH signature verifier is unable to verify the signed object as it's not signed with an SSH key");
            return Optional.empty();
        }
        if (this.featureManager.isDisabled((Feature)StandardFeature.SSH_KEYS_FOR_CODE_SIGNING)) {
            log.trace("The SSH signature cannot be verified because the SSH keys for code signing feature is not enabled");
            return Optional.empty();
        }
        try {
            DecodedSshSignature decodedSignature = DecodedSshSignature.fromPem(request.getSignature());
            PublicKey publicKey = decodedSignature.getPublicKey();
            Optional sshKeyCacheEntry = (Optional)this.keysCache.get((Object)publicKey);
            if (!sshKeyCacheEntry.isPresent()) {
                return PUBLIC_KEY_NOT_FOUND_RESULT;
            }
            SshKey sshKey = (SshKey)sshKeyCacheEntry.get();
            if (sshKey.getType() != KeyType.USER_KEY || sshKey.getUser() == null || !sshKey.getUser().isActive()) {
                if (sshKey.getType() != KeyType.USER_KEY) {
                    log.debug("The SSH signature could not be verified because the SSH key is not a personal SSH key.");
                    return ERROR_RESULT;
                }
                if (sshKey.getUser() == null) {
                    log.debug("The SSH signature could not be verified because the SSH key's user is null.");
                    return ERROR_RESULT;
                }
                if (!sshKey.getUser().isActive()) {
                    log.debug("The SSH signature could not be verified because the SSH key's user is inactive.");
                    return ERROR_RESULT;
                }
            }
            if (!SshSignatureVerifier.isValidSignature(decodedSignature, request.getSignedContent())) {
                return Optional.of(new SignatureVerificationResult.Builder(SignatureState.BAD).signatureType(SIGNATURE_TYPE).build());
            }
            return Optional.of(new SignatureVerificationResult.Builder(SignatureState.GOOD).owner(sshKey.getUser()).signatureType(SIGNATURE_TYPE).verificationPublicKey(() -> KeyUtils.calculateFingerprint(publicKey, StandardDigests.SHA256)).build());
        }
        catch (SshSignatureVerificationException | IOException | IllegalArgumentException | ExecutionException e) {
            log.debug("An error occurred while verifying the SSH signature", (Throwable)e);
            return ERROR_RESULT;
        }
    }

    private static boolean isValidSignature(DecodedSshSignature decodedSignature, String signedContent) throws SshSignatureVerificationException {
        try {
            PublicKey publicKey = decodedSignature.getPublicKey();
            Signature sign = publicKey instanceof SkED25519PublicKey ? (Signature)BuiltinSignatures.sk_ssh_ed25519.create() : (publicKey instanceof SkEcdsaPublicKey ? (Signature)BuiltinSignatures.sk_ecdsa_sha2_nistp256.create() : (Signature)BuiltinSignatures.fromFactory(SignatureFactory.resolveSignatureFactoryByPublicKey(decodedSignature.getPublicKey(), decodedSignature.getSignatureAlgorithm())).create());
            sign.initVerifier(null, decodedSignature.getPublicKey());
            sign.update(null, decodedSignature.createSignedData(signedContent));
            return sign.verify(null, decodedSignature.getSignature());
        }
        catch (Exception e) {
            throw new SshSignatureVerificationException(e);
        }
    }
}

