/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.ssh.utils;

import com.atlassian.bitbucket.ssh.Digest;
import com.atlassian.bitbucket.ssh.SshKey;
import com.atlassian.bitbucket.ssh.StandardDigests;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteStreams;
import jakarta.annotation.Nonnull;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InaccessibleObjectException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.UndeclaredThrowableException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import net.i2p.crypto.eddsa.EdDSAPublicKey;
import org.apache.commons.lang3.StringUtils;
import org.apache.sshd.common.config.keys.PublicKeyEntryDecoder;
import org.apache.sshd.common.config.keys.u2f.SecurityKeyPublicKey;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.DecoderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyUtils {
    private static final Logger log = LoggerFactory.getLogger(KeyUtils.class);
    private static final String ED25519_ALGORITHM = "EDDSA";
    private static final String ED25519_ALGORITHM_BC = "ED25519";
    private static final String KEY_PREFIX = "AAAA";
    private static final String X509_FORMAT = "X.509";

    private KeyUtils() {
        throw new UnsupportedOperationException(this.getClass().getName() + " is a utility class and should not be instantiated");
    }

    public static String calculateFingerprint(@Nonnull PublicKey key) {
        return KeyUtils.calculateFingerprint(key, StandardDigests.MD5);
    }

    public static String calculateFingerprint(@Nonnull PublicKey key, @Nonnull Digest digest) {
        return digest.apply(KeyUtils.encodePublicKey(key));
    }

    public static int calculateKeyLength(@Nonnull PublicKey publicKey) {
        int keyLength;
        PublicKey actualKey = publicKey;
        if (publicKey instanceof SecurityKeyPublicKey) {
            actualKey = ((SecurityKeyPublicKey)publicKey).getDelegatePublicKey();
        }
        int n = keyLength = KeyUtils.isEdDSAKey(actualKey) ? 256 : org.apache.sshd.common.config.keys.KeyUtils.getKeySize(actualKey);
        if (keyLength < 0) {
            throw new IllegalArgumentException("Unable to determine key length. Unsupported algorithm: " + publicKey.getAlgorithm());
        }
        return keyLength;
    }

    public static String getKeyAlgorithm(PublicKey publicKey) {
        if (KeyUtils.isEdDSAKey(publicKey)) {
            return ED25519_ALGORITHM_BC;
        }
        if (publicKey.getAlgorithm().equals("EC")) {
            return "ECDSA";
        }
        return publicKey.getAlgorithm().toUpperCase(Locale.US);
    }

    public static String getKeyComment(String keyText) {
        if (keyText != null) {
            try {
                return KeyUtils.decodeKeyAndLabel(keyText).getLabel();
            }
            catch (IllegalArgumentException | IllegalStateException e) {
                log.debug("Unable to get comment from key", (Throwable)e);
            }
        }
        return "";
    }

    public static String getKeyText(PublicKey publicKey) {
        byte[] keyBytes = KeyUtils.encodePublicKey(publicKey);
        return KeyUtils.readType(keyBytes) + " " + Base64.toBase64String((byte[])keyBytes);
    }

    public static PublicKey getPublicKey(@Nonnull byte[] keyData) {
        Objects.requireNonNull(keyData, "keyData");
        String keyType = KeyUtils.readType(keyData);
        try {
            return KeyUtils.getPublicKeyEntryDecoder(keyType).decodePublicKey(null, keyType, keyData, (Map<String, String>)ImmutableMap.of());
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalArgumentException("Unable to decode public key", e);
        }
    }

    public static PublicKey getPublicKey(String keyText) {
        return KeyUtils.decodeKeyAndLabel(keyText).getPublicKey();
    }

    public static PublicKey getX509PublicKey(String keyText, String algorithm) {
        try {
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decode((String)keyText));
            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
            return keyFactory.generatePublic(keySpec);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    public static boolean isEdDSAKey(@Nonnull PublicKey publicKey) {
        String keyAlgorithm = Objects.requireNonNull(publicKey, "publicKey").getAlgorithm().toUpperCase();
        return ED25519_ALGORITHM.equals(keyAlgorithm) || ED25519_ALGORITHM_BC.equals(keyAlgorithm);
    }

    public static boolean isRSAKeyFactorSmall(@Nonnull RSAPublicKey key) {
        try {
            Method hasAnySmallFactors = RSAKeyParameters.class.getDeclaredMethod("hasAnySmallFactors", BigInteger.class);
            hasAnySmallFactors.setAccessible(true);
            return (Boolean)hasAnySmallFactors.invoke(null, key.getModulus());
        }
        catch (ClassCastException | IllegalAccessException | NoSuchMethodException | SecurityException | InaccessibleObjectException | InvocationTargetException e) {
            log.debug("Unable to determine if the RSA key modulus has small factors", (Throwable)e);
            return false;
        }
    }

    public static boolean isUndecodable(SshKey sshKey) {
        try {
            sshKey.toPublicKey();
            return false;
        }
        catch (IllegalArgumentException | UndeclaredThrowableException e) {
            log.debug("Invalid SSH key detected for ID: {}", (Object)sshKey.getId(), (Object)e);
            return true;
        }
    }

    private static PublicKeyAndLabel decodeKeyAndLabel(String keyText) {
        IllegalArgumentException lastException = null;
        for (KeyBytesAndLabel keyAndLabel : KeyUtils.getKeyPossibilities(keyText)) {
            try {
                return new PublicKeyAndLabel(KeyUtils.getPublicKey(keyAndLabel.getDecodedBytes()), keyAndLabel.getLabel());
            }
            catch (Exception e) {
                lastException = new IllegalArgumentException("Failed to decode public key: " + keyText, e);
            }
        }
        if (lastException == null) {
            throw new IllegalArgumentException("Invalid key");
        }
        throw lastException;
    }

    private static byte[] encodePublicKey(@Nonnull PublicKey key) {
        byte[] byArray;
        Objects.requireNonNull(key, "key");
        ByteArrayOutputStream baos = new ByteArrayOutputStream(8192);
        try {
            PublicKey sshdKey = KeyUtils.maybeConvertPublicKey(key);
            KeyUtils.getPublicKeyEntryDecoder(sshdKey).encodePublicKey(baos, sshdKey);
            byArray = baos.toByteArray();
        }
        catch (Throwable throwable) {
            try {
                try {
                    baos.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }
        baos.close();
        return byArray;
    }

    private static List<KeyBytesAndLabel> getKeyPossibilities(String keyText) {
        Object[] keyParts = keyText.split("\\s+");
        ArrayList<KeyBytesAndLabel> keyAndLabels = new ArrayList<KeyBytesAndLabel>();
        for (int i = 0; i < keyParts.length; ++i) {
            if (!keyParts[i].startsWith(KEY_PREFIX)) continue;
            for (int j = i + 1; j <= keyParts.length; ++j) {
                String encodedText = StringUtils.join((Object[])keyParts, (String)"", (int)i, (int)j);
                String labelText = StringUtils.join((Object[])keyParts, (String)" ", (int)j, (int)keyParts.length);
                try {
                    keyAndLabels.add(new KeyBytesAndLabel(Base64.decode((String)encodedText), labelText));
                    continue;
                }
                catch (DecoderException e) {
                    log.trace("Could not decode possible key text: {}.", (Object)encodedText);
                }
            }
            break;
        }
        return keyAndLabels;
    }

    private static <T extends PublicKey> PublicKeyEntryDecoder<T, ?> getPublicKeyEntryDecoder(Key key) {
        PublicKeyEntryDecoder<?, ?> decoder = org.apache.sshd.common.config.keys.KeyUtils.getPublicKeyEntryDecoder(key);
        if (decoder == null) {
            throw new IllegalArgumentException(String.format("Unable to decode key. Unsupported algorithm of type [%s] encoded in format [%s]", key.getAlgorithm(), key.getFormat()));
        }
        return decoder;
    }

    private static <T extends PublicKey> PublicKeyEntryDecoder<T, ?> getPublicKeyEntryDecoder(String keyType) {
        PublicKeyEntryDecoder<?, ?> decoder = org.apache.sshd.common.config.keys.KeyUtils.getPublicKeyEntryDecoder(keyType);
        if (decoder == null) {
            throw new IllegalArgumentException(String.format("Unknown public key type: " + keyType, new Object[0]));
        }
        return decoder;
    }

    private static PublicKey maybeConvertPublicKey(PublicKey key) {
        if (KeyUtils.isEdDSAKey(key) && !(key instanceof EdDSAPublicKey)) {
            if (!X509_FORMAT.equals(key.getFormat())) {
                throw new IllegalArgumentException("Unable to convert Ed25519 public key. Unsupported key format: " + key.getFormat());
            }
            try {
                return new EdDSAPublicKey(new X509EncodedKeySpec(key.getEncoded()));
            }
            catch (InvalidKeySpecException e) {
                throw new IllegalArgumentException("Unable to convert Ed25519 public key: Invalid X.509 encoding", e);
            }
        }
        return key;
    }

    private static String readType(byte[] bytes) {
        String string;
        DataInputStream data = new DataInputStream(new ByteArrayInputStream(bytes));
        try {
            byte[] b = new byte[data.readInt()];
            ByteStreams.readFully((InputStream)data, (byte[])b);
            string = new String(b, StandardCharsets.US_ASCII);
        }
        catch (Throwable throwable) {
            try {
                try {
                    data.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new IllegalArgumentException(e);
            }
        }
        data.close();
        return string;
    }

    private static class PublicKeyAndLabel {
        private final String label;
        private final PublicKey publicKey;

        public PublicKeyAndLabel(PublicKey publicKey, String label) {
            this.publicKey = publicKey;
            this.label = label;
        }

        public String getLabel() {
            return this.label;
        }

        public PublicKey getPublicKey() {
            return this.publicKey;
        }
    }

    private static class KeyBytesAndLabel {
        private final byte[] decodedBytes;
        private final String label;

        public KeyBytesAndLabel(byte[] decodedBytes, String label) {
            this.decodedBytes = decodedBytes;
            this.label = label;
        }

        public byte[] getDecodedBytes() {
            return this.decodedBytes;
        }

        public String getLabel() {
            return this.label;
        }
    }
}

