/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.mesh.transaction;

import com.atlassian.bitbucket.mesh.io.MoreFiles;
import com.atlassian.bitbucket.mesh.io.UnderLock;
import com.atlassian.bitbucket.mesh.rpc.v1.RpcTransaction;
import com.atlassian.bitbucket.mesh.transaction.TransactionLog;
import com.atlassian.bitbucket.mesh.transaction.TransactionLogException;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import jakarta.annotation.Nonnull;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Queue;
import java.util.UUID;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileTransactionLog
implements TransactionLog {
    private static final Logger log = LoggerFactory.getLogger(FileTransactionLog.class);
    private static final Duration DEFAULT_LOCK_TIMEOUT = Duration.ofSeconds(5L);
    private static final int DEFAULT_TRANSACTIONS_PER_SEGMENT = 1024;
    private static final long MAX_SIZE_INLINE = 1024L;
    private static final int VERSION = 1;
    private final Path file;
    private final int maxTransactionsPerSegment;

    public FileTransactionLog(Path file) {
        this(file, 1024);
    }

    public FileTransactionLog(Path file, int maxTransactionsPerSegment) {
        this.file = file;
        this.maxTransactionsPerSegment = maxTransactionsPerSegment;
    }

    @Override
    public void append(@Nonnull RpcTransaction transaction) {
        this.internalAppend(Objects.requireNonNull(transaction, "transaction"));
    }

    public void init() {
        this.internalAppend(null);
    }

    @Override
    public void process(@Nonnull TransactionLog.Processor processor, int preferredBatchSize) {
        Objects.requireNonNull(processor, "processor");
        Preconditions.checkArgument((preferredBatchSize > 0 ? 1 : 0) != 0, (Object)"preferredBatchSize must be non-negative");
        for (Path segment : this.getSegments()) {
            TransactionLog.Result result = this.processSegment(segment, processor, preferredBatchSize);
            if (result.getUnprocessed() == 0) {
                MoreFiles.deleteQuietly((Path)segment);
            }
            result.getException().ifPresent(exception -> {
                Throwables.throwIfUnchecked((Throwable)exception);
                throw new TransactionLogException("Error while processing the transaction log", (Throwable)exception);
            });
            if (!result.isAbort()) continue;
            break;
        }
    }

    @Nonnull
    @VisibleForTesting
    public Stream<RpcTransaction> stream() {
        return this.streamSegment(this.file);
    }

    private static void checkVersionSupported(int version) {
        if (version > 1) {
            throw new IllegalStateException("Cannot read transaction logs of version " + version);
        }
    }

    private static String getRepositoryPrefix(String repositoryId) {
        int index = repositoryId.lastIndexOf(47);
        String prefix = index == -1 ? repositoryId : repositoryId.substring(index + 1);
        return MoreFiles.pathSafe((String)prefix);
    }

    private static RpcTransaction loadTransactionFromFile(Path file) {
        RpcTransaction rpcTransaction;
        block9: {
            InputStream inputStream = Files.newInputStream(file, new OpenOption[0]);
            try {
                CodedInputStream cis = CodedInputStream.newInstance((InputStream)inputStream, (int)1);
                FileTransactionLog.checkVersionSupported(cis.readRawVarint32());
                rpcTransaction = RpcTransaction.parseDelimitedFrom((InputStream)inputStream);
                if (inputStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (FileNotFoundException | NoSuchFileException e) {
                    return null;
                }
                catch (IOException e) {
                    log.error("Failed to read transaction details from {}", (Object)file, (Object)e);
                    throw new TransactionLogException("Error reading transaction details from " + String.valueOf(file.getFileName()));
                }
            }
            inputStream.close();
        }
        return rpcTransaction;
    }

    private static List<RpcTransaction> readTransactions(InputStream input, int numberToRead) throws IOException {
        RpcTransaction transaction;
        ImmutableList.Builder builder = ImmutableList.builder();
        for (int size = 0; size < numberToRead && (transaction = RpcTransaction.parseDelimitedFrom((InputStream)input)) != null; ++size) {
            builder.add((Object)transaction);
        }
        return builder.build();
    }

    private static void writeTransactionDetails(Path file, RpcTransaction transaction) {
        MoreFiles.mkdir((Path)file.getParent());
        try (OutputStream outputStream = Files.newOutputStream(file, new OpenOption[0]);){
            CodedOutputStream cos = CodedOutputStream.newInstance((OutputStream)outputStream);
            cos.writeInt32NoTag(1);
            cos.flush();
            transaction.writeDelimitedTo(outputStream);
        }
        catch (IOException e) {
            log.error("Failed to write transaction details to {}", (Object)file, (Object)e);
            throw new TransactionLogException("Error writing transaction details to " + String.valueOf(file.getFileName()));
        }
    }

    private static void writeTransactions(OutputStream output, List<RpcTransaction> transactions) throws IOException {
        for (RpcTransaction transaction : transactions) {
            transaction.writeDelimitedTo(output);
        }
    }

    private void deleteTransactionDetails(List<RpcTransaction> transactions) {
        for (RpcTransaction transaction : transactions) {
            Path detailsFile = this.getTransactionPath(transaction.getId());
            try {
                Files.delete(detailsFile);
            }
            catch (FileNotFoundException | NoSuchFileException iOException) {
            }
            catch (IOException e) {
                log.warn("Failed to delete transaction details file {}", (Object)detailsFile);
            }
        }
    }

    private Path getSegmentPath(long segmentId) {
        return this.file.resolveSibling(String.valueOf(this.file.getFileName()) + "." + segmentId);
    }

    private List<Path> getSegments() {
        final ArrayList<Path> segments = new ArrayList<Path>();
        final String prefix = this.file.getFileName().toString();
        try {
            Files.walkFileTree(this.file.getParent(), Collections.emptySet(), 1, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    if (attrs.isRegularFile() && file.getFileName().toString().startsWith(prefix)) {
                        segments.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (FileNotFoundException | NoSuchFileException iOException) {
        }
        catch (IOException e) {
            log.warn("Error while searching for segments of {}", (Object)this.file, (Object)e);
            throw new TransactionLogException("Error while searching for segments of " + String.valueOf(this.file.getFileName()), e);
        }
        segments.sort((path1, path2) -> {
            if (path1.equals(path2)) {
                return 0;
            }
            if (path1.equals(this.file)) {
                return 1;
            }
            if (path2.equals(this.file)) {
                return -1;
            }
            long segmentId1 = Long.parseLong(path1.getFileName().toString().substring(prefix.length() + 1));
            long segmentId2 = Long.parseLong(path2.getFileName().toString().substring(prefix.length() + 1));
            return Long.compare(segmentId1, segmentId2);
        });
        return segments;
    }

    private Path getTransactionPath(String transactionId) {
        return this.file.resolveSibling(transactionId + ".dat");
    }

    private void internalAppend(RpcTransaction transaction) {
        boolean appended = false;
        try {
            MoreFiles.mkdir((Path)this.file.getParent());
            UnderLock.forFile((Path)this.file, (Duration)DEFAULT_LOCK_TIMEOUT).editRaw((input, output) -> {
                long segmentId;
                long prevSegmentId = 0L;
                LinkedList<RpcTransaction> remaining = new LinkedList<RpcTransaction>();
                if (input == null) {
                    segmentId = Instant.now().toEpochMilli();
                } else {
                    RpcTransaction existing;
                    CodedInputStream cis = CodedInputStream.newInstance((InputStream)input, (int)1);
                    FileTransactionLog.checkVersionSupported(cis.readUInt32());
                    prevSegmentId = cis.readUInt64();
                    segmentId = cis.readUInt64();
                    input = new BufferedInputStream((InputStream)input, 1024);
                    while ((existing = RpcTransaction.parseDelimitedFrom((InputStream)input)) != null) {
                        remaining.add(existing);
                    }
                }
                if (transaction != null) {
                    remaining.add(transaction);
                }
                if (remaining.size() > this.maxTransactionsPerSegment) {
                    this.writeSegment(remaining, prevSegmentId, segmentId);
                    prevSegmentId = segmentId++;
                }
                this.writeSegment((OutputStream)output, (Queue<RpcTransaction>)remaining, prevSegmentId, segmentId);
            }, false);
            appended = true;
        }
        catch (IOException e) {
            log.error("Failed to write to transaction log {}", (Object)this.file, (Object)e);
            throw new TransactionLogException("Failed to write to transaction log " + String.valueOf(this.file.getFileName()));
        }
        finally {
            if (!appended) {
                log.error("Failed to record transaction in {}. Details: {}", (Object)this.file, (Object)transaction);
            }
        }
    }

    private RpcTransaction maybeLoadDetails(RpcTransaction transaction) {
        Path detailPath;
        RpcTransaction fullTransaction;
        if (transaction.getDetailsOneofCase() == RpcTransaction.DetailsOneofCase.DETAILSONEOF_NOT_SET && (fullTransaction = FileTransactionLog.loadTransactionFromFile(detailPath = this.getTransactionPath(transaction.getId()))) != null) {
            return fullTransaction;
        }
        return transaction;
    }

    private List<RpcTransaction> maybeLoadDetails(List<RpcTransaction> transactions) {
        return (List)transactions.stream().map(this::maybeLoadDetails).collect(ImmutableList.toImmutableList());
    }

    private TransactionLog.Result processSegment(Path path, TransactionLog.Processor processor, int preferredBatchSize) {
        try {
            MutableObject retVal = new MutableObject((Object)TransactionLog.Result.proceed());
            UnderLock.forFile((Path)path, (Duration)DEFAULT_LOCK_TIMEOUT).editRaw((input, output) -> retVal.setValue((Object)this.processSegment((InputStream)input, (OutputStream)output, processor, preferredBatchSize)), false);
            return (TransactionLog.Result)retVal.getValue();
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            return TransactionLog.Result.proceed();
        }
        catch (IOException e) {
            log.error("Error processing transaction log segment {}", (Object)path, (Object)e);
            throw new TransactionLogException("Error processing transaction log segment " + String.valueOf(path.getFileName()), e);
        }
    }

    private TransactionLog.Result processSegment(InputStream input, OutputStream output, TransactionLog.Processor processor, int preferredBatchSize) throws IOException {
        List<RpcTransaction> transactions;
        if (input == null) {
            return TransactionLog.Result.proceed();
        }
        CodedInputStream cis = CodedInputStream.newInstance((InputStream)input, (int)1);
        FileTransactionLog.checkVersionSupported(cis.readUInt32());
        long prevSegmentId = cis.readUInt64();
        long segmentId = cis.readUInt64();
        CodedOutputStream cos = CodedOutputStream.newInstance((OutputStream)output, (int)12);
        cos.writeUInt32NoTag(1);
        cos.writeUInt64NoTag(prevSegmentId);
        cos.writeUInt64NoTag(segmentId);
        cos.flush();
        TransactionLog.Result result = TransactionLog.Result.proceed();
        while (!result.isAbort()) {
            transactions = FileTransactionLog.readTransactions(input, preferredBatchSize);
            try {
                result = processor.process(this.maybeLoadDetails(transactions));
            }
            catch (RuntimeException e) {
                result = TransactionLog.Result.error(transactions.size(), e);
            }
            int unprocessed = Math.min(Math.max(0, result.getUnprocessed()), transactions.size());
            int indexFirstUnprocessed = transactions.size() - unprocessed;
            this.deleteTransactionDetails(transactions.subList(0, indexFirstUnprocessed));
            FileTransactionLog.writeTransactions(output, transactions.subList(indexFirstUnprocessed, transactions.size()));
            if (transactions.size() >= preferredBatchSize) continue;
            return result;
        }
        if (result.isAbort()) {
            do {
                if (!(transactions = FileTransactionLog.readTransactions(input, 1000)).isEmpty()) {
                    result = new TransactionLog.Result(result.isAbort(), Math.max(1, result.getUnprocessed()), result.getException().orElse(null));
                }
                FileTransactionLog.writeTransactions(output, transactions);
            } while (transactions.size() == 1000);
        }
        return result;
    }

    @Nonnull
    private Stream<RpcTransaction> streamSegment(Path path) {
        Stream<RpcTransaction> stream;
        block10: {
            InputStream input = Files.newInputStream(path, new OpenOption[0]);
            try {
                RpcTransaction transaction;
                CodedInputStream cis = CodedInputStream.newInstance((InputStream)input, (int)1);
                int version = cis.readUInt32();
                FileTransactionLog.checkVersionSupported(version);
                long prevSegmentId = cis.readUInt64();
                cis.readUInt64();
                ImmutableList.Builder builder = ImmutableList.builder();
                while ((transaction = RpcTransaction.parseDelimitedFrom((InputStream)input)) != null) {
                    builder.add((Object)transaction);
                }
                Stream<RpcTransaction> result = builder.build().stream().map(this::maybeLoadDetails);
                Stream<RpcTransaction> stream2 = stream = prevSegmentId == 0L ? result : Stream.concat(this.streamSegment(this.getSegmentPath(prevSegmentId)), result);
                if (input == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (input != null) {
                        try {
                            input.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (FileNotFoundException | NoSuchFileException e) {
                    return Stream.empty();
                }
                catch (IOException e) {
                    log.error("Error streaming transaction log segment {}", (Object)path, (Object)e);
                    throw new TransactionLogException("Error streaming transaction log segment " + String.valueOf(path.getFileName()), e);
                }
            }
            input.close();
        }
        return stream;
    }

    private void writeSegment(Queue<RpcTransaction> transactions, long prevSegmentId, long segmentId) {
        Path path = this.getSegmentPath(segmentId);
        try (OutputStream output = Files.newOutputStream(path, StandardOpenOption.CREATE_NEW);){
            this.writeSegment(output, transactions, prevSegmentId, segmentId);
        }
        catch (IOException e) {
            log.error("Error writing transaction log segment {}", (Object)path, (Object)e);
            throw new TransactionLogException("Error writing transaction log segment " + String.valueOf(path.getFileName()), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeSegment(OutputStream output, Queue<RpcTransaction> transactions, long prevSegmentId, long segmentId) throws IOException {
        BufferedOutputStream bufferedOut = new BufferedOutputStream(output);
        try {
            CodedOutputStream cos = CodedOutputStream.newInstance((OutputStream)bufferedOut, (int)1);
            cos.writeUInt32NoTag(1);
            cos.writeUInt64NoTag(prevSegmentId);
            cos.writeUInt64NoTag(segmentId);
            cos.flush();
            for (int i = 0; i < this.maxTransactionsPerSegment; ++i) {
                RpcTransaction transaction = transactions.poll();
                if (transaction == null) {
                    break;
                }
                String prefix = FileTransactionLog.getRepositoryPrefix(transaction.getRepository());
                Object transactionId = transaction.getId();
                if (StringUtils.isBlank((CharSequence)transactionId)) {
                    transactionId = prefix + "-" + segmentId + "-" + String.valueOf(UUID.randomUUID());
                    transaction = transaction.toBuilder().setId((String)transactionId).build();
                }
                if ((long)transaction.getSerializedSize() <= 1024L) {
                    transaction.writeDelimitedTo((OutputStream)bufferedOut);
                    continue;
                }
                FileTransactionLog.writeTransactionDetails(this.getTransactionPath((String)transactionId), transaction);
                transaction.toBuilder().clearDetailsOneof().setId((String)transactionId).build().writeDelimitedTo((OutputStream)bufferedOut);
            }
        }
        finally {
            bufferedOut.flush();
        }
    }
}

