/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.internal.tag.idx;

import com.atlassian.bitbucket.internal.tag.idx.TagIndexReader;
import com.atlassian.bitbucket.internal.tag.idx.TagIndexWriter;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.repository.RefOrder;
import com.atlassian.bitbucket.repository.RefService;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositorySupplier;
import com.atlassian.bitbucket.repository.RepositoryTagsRequest;
import com.atlassian.bitbucket.repository.Tag;
import com.atlassian.bitbucket.repository.TagCallback;
import com.atlassian.bitbucket.server.ApplicationPropertiesService;
import com.atlassian.bitbucket.server.StorageService;
import com.atlassian.bitbucket.user.SecurityService;
import com.atlassian.bitbucket.util.MoreFiles;
import com.atlassian.bitbucket.util.Timer;
import com.atlassian.bitbucket.util.TimerUtils;
import com.atlassian.bitbucket.util.concurrent.ExecutorUtils;
import com.atlassian.util.contentcache.BackgroundThreadStreamPumper;
import com.atlassian.util.contentcache.ContentCache;
import com.atlassian.util.contentcache.ContentProvider;
import com.atlassian.util.contentcache.FileContentCacheManager;
import com.atlassian.util.contentcache.StreamPumper;
import io.atlassian.util.concurrent.ThreadFactories;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Date;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TagIndexer {
    static final String CACHE_PATH = "tag-index";
    private static final String CACHE_KEY = "tag-index";
    private static final String CACHE_PUMP_THREAD_NAME = "tag-index-io-pump";
    private static final String PROP_INDEX_QUEUE_SIZE = "plugin.tag.index.background.index.queue.size";
    private static final String PROP_MIN_SPACE_IN_BYTES = "plugin.tag.index.cache.min.free.space";
    private static final Logger log = LoggerFactory.getLogger(TagIndexer.class);
    private final ExecutorService backgroundIndexExecutor;
    private final ContentCache cache;
    private final BackgroundThreadStreamPumper cachePump;
    private final RefService refService;
    private final RepositorySupplier repositorySupplier;
    private final SecurityService securityService;
    private final Path tagCacheDirectory;
    private volatile boolean active;

    public TagIndexer(ApplicationPropertiesService propertiesService, RefService refService, RepositorySupplier repositorySupplier, SecurityService securityService, StorageService storageService) {
        this.refService = refService;
        this.repositorySupplier = repositorySupplier;
        this.securityService = securityService;
        int queueSize = Math.max(1, propertiesService.getPluginProperty(PROP_INDEX_QUEUE_SIZE, 500));
        this.backgroundIndexExecutor = new ThreadPoolExecutor(1, 1, Long.MAX_VALUE, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(queueSize), ThreadFactories.namedThreadFactory((String)"tag-indexer", (ThreadFactories.Type)ThreadFactories.Type.DAEMON));
        Path cacheDir = storageService.getCacheDir();
        this.tagCacheDirectory = cacheDir.resolve("tag-index");
        this.cachePump = new BackgroundThreadStreamPumper(CACHE_PUMP_THREAD_NAME);
        this.cache = new FileContentCacheManager.Builder(cacheDir).minFreeSpaceBytes((long)propertiesService.getPluginProperty(PROP_MIN_SPACE_IN_BYTES, 0x100000)).streamPumper((StreamPumper)this.cachePump).build().getCache("tag-index");
    }

    public void clearAll() {
        log.debug("Clearing tag cache");
        this.cache.clear();
    }

    public void destroy() {
        log.trace("Shutting down");
        this.active = false;
        ExecutorUtils.shutdown((ExecutorService)this.backgroundIndexExecutor);
        this.clearCacheDirectory();
        this.cachePump.shutdown();
    }

    public void init() {
        log.trace("Initializing");
        this.clearCacheDirectory();
        this.active = true;
    }

    public Iterable<Tag> readIndex(final Repository repository, int max, Set<String> commitIds) {
        if (!this.active) {
            log.debug("{}: The indexer is no longer active", (Object)repository);
            return Collections.emptyList();
        }
        log.debug("{}: Streaming cached tags", (Object)repository);
        try (TagIndexReader reader = new TagIndexReader(commitIds, max);
             Timer ignored = TimerUtils.start((String)("tags: read index " + repository.getId()));){
            this.cache.stream(TagIndexer.getCacheKey(repository.getId()), (OutputStream)reader, new ContentProvider(){

                public void apply(OutputStream out) {
                    long start = System.currentTimeMillis();
                    try (Timer ignored = TimerUtils.start((String)("tags: write index " + repository.getId()));){
                        log.debug("{}: Writing tag cache", (Object)repository);
                        RepositoryTagsRequest request = ((RepositoryTagsRequest.Builder)new RepositoryTagsRequest.Builder(repository).order(RefOrder.ALPHABETICAL)).build();
                        TagIndexer.this.refService.streamTags(request, (TagCallback)new TagIndexWriter(out));
                        log.debug("{}: Tag cache written in {}ms", (Object)repository, (Object)(System.currentTimeMillis() - start));
                    }
                }

                public Date getExpiry() {
                    return null;
                }
            });
        }
        return reader.getMatches();
    }

    public void scheduleReindex(int repositoryId) {
        log.debug("{}: Clearing cached tags for reindexing", (Object)repositoryId);
        this.cache.remove(TagIndexer.getCacheKey(repositoryId));
        try {
            this.backgroundIndexExecutor.submit(() -> {
                try {
                    this.securityService.withPermission(Permission.REPO_READ, "Reindexing tags").call(() -> {
                        Repository repository = this.repositorySupplier.getById(repositoryId);
                        if (repository == null) {
                            log.debug("{}: Skipping tag indexing; the repository was deleted", (Object)repositoryId);
                        } else {
                            this.readIndex(repository, 0, Collections.emptySet());
                        }
                        return null;
                    });
                }
                catch (Exception e) {
                    log.warn("{}: Background tag indexing failed: {}", new Object[]{repositoryId, e.getMessage(), log.isDebugEnabled() ? e : null});
                }
            });
        }
        catch (RejectedExecutionException e) {
            log.warn("{}: Could not schedule background tag indexing; the task queue is full", (Object)repositoryId);
        }
    }

    private static String getCacheKey(int repositoryId) {
        return Integer.toString(repositoryId);
    }

    private void clearCacheDirectory() {
        try {
            MoreFiles.cleanDirectory((Path)this.tagCacheDirectory);
        }
        catch (IOException e) {
            log.warn("Could not clear cache directory {}: {}", new Object[]{this.tagCacheDirectory, e.getMessage(), log.isDebugEnabled() ? e : null});
        }
    }
}

