/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.mesh.util;

import com.atlassian.bitbucket.mesh.util.KeyFilePermissionsException;
import com.google.common.collect.ImmutableSet;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.EnumSet;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyUtils {
    private static final String ALGORITHM_EC = "EC";
    private static final String ALGORITHM_ECDSA = "ECDSA";
    private static final String ALGORITHM_RSA = "RSA";
    private static final JcaPEMKeyConverter CONVERTER;
    private static final String EC_CURVE_P256 = "nistp256";
    private static final String EC_CURVE_P384 = "nistp384";
    private static final String EC_CURVE_P521 = "nistp521";
    private static final ECParameterSpec EC_SPEC_P256;
    private static final ECParameterSpec EC_SPEC_P384;
    private static final ECParameterSpec EC_SPEC_P521;
    private static final String EC_TYPE_P256 = "ecdsa-sha2-nistp256";
    private static final String EC_TYPE_P384 = "ecdsa-sha2-nistp384";
    private static final String EC_TYPE_P512 = "ecdsa-sha2-nistp521";
    private static final BaseEncoding ENCODING;
    private static final HashFunction HASHER;
    private static final String RSA_TYPE = "ssh-rsa";
    private static final Set<PosixFilePermission> TOO_OPEN;
    private static final Set<PosixFilePermission> OWNER_RW;
    private static final Logger log;

    private KeyUtils() {
        throw new UnsupportedOperationException(String.valueOf(this.getClass()) + " is not intended to be instantiated");
    }

    @Nonnull
    public static String getFingerprint(@Nonnull PublicKey publicKey) {
        Objects.requireNonNull(publicKey, "publicKey");
        return ENCODING.encode(HASHER.hashBytes(switch (publicKey.getAlgorithm().toUpperCase(Locale.ROOT)) {
            case ALGORITHM_EC, ALGORITHM_ECDSA -> KeyUtils.encodeEc(publicKey);
            case ALGORITHM_RSA -> KeyUtils.encodeRsa(publicKey);
            default -> throw new UnsupportedOperationException("Fingerprints are not supported for " + publicKey.getAlgorithm());
        }).asBytes());
    }

    public static boolean hasTooLoosePermissions(Path path) throws IOException {
        return Files.getPosixFilePermissions(path, new LinkOption[0]).stream().anyMatch(TOO_OPEN::contains);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    public static KeyPair parseKeyPairFromPkcs8(@Nonnull String pkcs8) {
        try (StringReader reader = new StringReader(Objects.requireNonNull(pkcs8, "pkcs8"));
             PEMParser parser = new PEMParser((Reader)reader);){
            Object keyPair = parser.readObject();
            if (keyPair instanceof PEMKeyPair) {
                KeyPair keyPair2 = CONVERTER.getKeyPair((PEMKeyPair)keyPair);
                return keyPair2;
            }
            if (keyPair != null) throw new IllegalArgumentException("The provided PKCS8 data contained a " + String.valueOf(keyPair.getClass()));
            throw new IllegalArgumentException("The provided PKCS8 data did not contain an object");
        }
        catch (IOException e) {
            throw new IllegalArgumentException("The provided PKCS8 data could not be parsed", e);
        }
    }

    @Nonnull
    public static PrivateKey parsePrivateKeyFromDer(@Nonnull String derBase64) {
        return KeyUtils.parsePrivateKeyFromDer(ENCODING.decode((CharSequence)Objects.requireNonNull(derBase64, "derBase64")));
    }

    @Nonnull
    public static PrivateKey parsePrivateKeyFromDer(@Nonnull byte[] der) {
        PrivateKeyInfo info = PrivateKeyInfo.getInstance((Object)Objects.requireNonNull(der, "der"));
        try {
            return CONVERTER.getPrivateKey(info);
        }
        catch (IOException e) {
            throw new IllegalArgumentException("The provided DER data could not be parsed");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    public static PrivateKey parsePrivateKeyFromPkcs8(@Nonnull String pkcs8) {
        try (StringReader reader = new StringReader(Objects.requireNonNull(pkcs8, "pkcs8"));
             PEMParser parser = new PEMParser((Reader)reader);){
            Object privateKey = parser.readObject();
            if (privateKey instanceof PEMKeyPair) {
                PrivateKey privateKey2 = CONVERTER.getPrivateKey(((PEMKeyPair)privateKey).getPrivateKeyInfo());
                return privateKey2;
            }
            if (privateKey instanceof PrivateKeyInfo) {
                PrivateKey privateKey3 = CONVERTER.getPrivateKey((PrivateKeyInfo)privateKey);
                return privateKey3;
            }
            if (privateKey != null) throw new IllegalArgumentException("The provided PKCS8 data contained a " + String.valueOf(privateKey.getClass()));
            throw new IllegalArgumentException("The provided PKCS8 data did not contain an object");
        }
        catch (IOException e) {
            throw new IllegalArgumentException("The provided PKCS8 data could not be parsed");
        }
    }

    @Nonnull
    public static PublicKey parsePublicKeyFromDer(@Nonnull String derBase64) {
        return KeyUtils.parsePublicKeyFromDer(ENCODING.decode((CharSequence)Objects.requireNonNull(derBase64, "derBase64")));
    }

    @Nonnull
    public static PublicKey parsePublicKeyFromDer(@Nonnull byte[] der) {
        SubjectPublicKeyInfo info = SubjectPublicKeyInfo.getInstance((Object)Objects.requireNonNull(der, "der"));
        try {
            return CONVERTER.getPublicKey(info);
        }
        catch (IOException e) {
            throw new IllegalArgumentException("The provided DER data could not be parsed");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    public static PublicKey parsePublicKeyFromPkcs8(@Nonnull String pkcs8) {
        try (StringReader reader = new StringReader(Objects.requireNonNull(pkcs8, "pkcs8"));
             PEMParser parser = new PEMParser((Reader)reader);){
            Object publicKey = parser.readObject();
            if (publicKey instanceof PEMKeyPair) {
                PublicKey publicKey2 = CONVERTER.getPublicKey(((PEMKeyPair)publicKey).getPublicKeyInfo());
                return publicKey2;
            }
            if (publicKey instanceof SubjectPublicKeyInfo) {
                PublicKey publicKey3 = CONVERTER.getPublicKey((SubjectPublicKeyInfo)publicKey);
                return publicKey3;
            }
            if (publicKey != null) throw new IllegalArgumentException("The provided PKCS8 data contained a " + String.valueOf(publicKey.getClass()));
            throw new IllegalArgumentException("The provided PKCS8 data did not contain an object");
        }
        catch (IOException e) {
            throw new IllegalArgumentException("The provided PKCS8 data could not be parsed");
        }
    }

    @Nullable
    public static KeyPair readKeyPair(@Nonnull Path path) throws IOException {
        Object keyPair = KeyUtils.read(path);
        if (keyPair instanceof PEMKeyPair) {
            return CONVERTER.getKeyPair((PEMKeyPair)keyPair);
        }
        if (keyPair == null) {
            return null;
        }
        throw new IllegalStateException(String.valueOf(path.getFileName()) + " contains unexpected data: " + String.valueOf(keyPair.getClass()));
    }

    @Nullable
    public static PublicKey readPublicKey(@Nonnull Path path) throws IOException {
        Object publicKey = KeyUtils.read(path);
        if (publicKey instanceof PEMKeyPair) {
            return CONVERTER.getKeyPair((PEMKeyPair)publicKey).getPublic();
        }
        if (publicKey instanceof SubjectPublicKeyInfo) {
            return CONVERTER.getPublicKey((SubjectPublicKeyInfo)publicKey);
        }
        if (publicKey == null) {
            return null;
        }
        throw new IllegalStateException(String.valueOf(path.getFileName()) + " contains unexpected data: " + String.valueOf(publicKey.getClass()));
    }

    @Nonnull
    public static String toPkcs8(@Nonnull KeyPair keyPair) throws IOException {
        return KeyUtils.toPkcs8(Objects.requireNonNull(keyPair, "keyPair").getPrivate());
    }

    @Nonnull
    public static String toPkcs8(@Nonnull PublicKey publicKey) throws IOException {
        return KeyUtils.toPkcs8((Object)Objects.requireNonNull(publicKey, "publicKey"));
    }

    public static void writeKeyPair(@Nonnull Path path, @Nonnull KeyPair keyPair) throws IOException {
        KeyUtils.write(path, (Object)Objects.requireNonNull(keyPair, "keyPair").getPrivate());
    }

    public static void writePublicKey(@Nonnull Path path, @Nonnull PublicKey publicKey) throws IOException {
        KeyUtils.write(path, (Object)Objects.requireNonNull(publicKey, "publicKey"));
    }

    private static byte[] encodeEc(PublicKey publicKey) {
        String type;
        String curve;
        BCECPublicKey bcEcKey = publicKey instanceof BCECPublicKey ? (BCECPublicKey)publicKey : new BCECPublicKey((ECPublicKey)publicKey, BouncyCastleProvider.CONFIGURATION);
        ECParameterSpec ecParameterSpec = bcEcKey.getParameters();
        if (ecParameterSpec.equals((Object)EC_SPEC_P256)) {
            curve = EC_CURVE_P256;
            type = EC_TYPE_P256;
        } else if (ecParameterSpec.equals((Object)EC_SPEC_P384)) {
            curve = EC_CURVE_P384;
            type = EC_TYPE_P384;
        } else if (ecParameterSpec.equals((Object)EC_SPEC_P521)) {
            curve = EC_CURVE_P521;
            type = EC_TYPE_P512;
        } else {
            throw new IllegalArgumentException("Unable to encode public key: Unsupported elliptic curve");
        }
        byte[] t = type.getBytes(StandardCharsets.US_ASCII);
        byte[] c = curve.getBytes(StandardCharsets.US_ASCII);
        byte[] q = bcEcKey.getQ().getEncoded(false);
        ByteBuffer buffer = ByteBuffer.allocate(t.length + c.length + q.length + 12);
        buffer.putInt(t.length);
        buffer.put(t);
        buffer.putInt(c.length);
        buffer.put(c);
        buffer.putInt(q.length);
        buffer.put(q);
        return buffer.array();
    }

    private static byte[] encodeRsa(PublicKey publicKey) {
        RSAPublicKey rsaKey = (RSAPublicKey)publicKey;
        byte[] t = RSA_TYPE.getBytes(StandardCharsets.US_ASCII);
        byte[] exp = rsaKey.getPublicExponent().toByteArray();
        byte[] mod = rsaKey.getModulus().toByteArray();
        boolean padExp = (exp[0] & 0x80) != 0;
        boolean padMod = (mod[0] & 0x80) != 0;
        int size = t.length + exp.length + mod.length + 12;
        if (padExp) {
            ++size;
        }
        if (padMod) {
            ++size;
        }
        ByteBuffer buffer = ByteBuffer.allocate(size);
        buffer.putInt(t.length);
        buffer.put(t);
        if (padExp) {
            buffer.putInt(exp.length + 1);
            buffer.put((byte)0);
        } else {
            buffer.putInt(exp.length);
        }
        buffer.put(exp);
        if (padMod) {
            buffer.putInt(mod.length + 1);
            buffer.put((byte)0);
        } else {
            buffer.putInt(mod.length);
        }
        buffer.put(mod);
        return buffer.array();
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static Object read(Path path) throws IOException {
        Objects.requireNonNull(path, "path");
        try {
            Set<PosixFilePermission> permissions = Files.getPosixFilePermissions(path, new LinkOption[0]);
            if (permissions.stream().anyMatch(TOO_OPEN::contains)) {
                throw new KeyFilePermissionsException(String.format("%s will not be read; its permissions are too loose (%s). Set permissions to (%s) and restart.", path, PosixFilePermissions.toString(permissions), PosixFilePermissions.toString(OWNER_RW)));
            }
            try (BufferedReader reader = Files.newBufferedReader(path, StandardCharsets.US_ASCII);){
                Object object;
                try (PEMParser pemParser = new PEMParser((Reader)reader);){
                    object = pemParser.readObject();
                }
                return object;
            }
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            return null;
        }
    }

    private static String toPkcs8(Object data) throws IOException {
        StringWriter writer = new StringWriter();
        KeyUtils.write(writer, data);
        return writer.toString();
    }

    private static void write(Path path, Object data) throws IOException {
        Objects.requireNonNull(path, "path");
        try (BufferedWriter writer = Files.newBufferedWriter(path, StandardCharsets.US_ASCII, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
            KeyUtils.write(writer, data);
        }
        Files.setPosixFilePermissions(path, OWNER_RW);
    }

    private static void write(Writer writer, Object data) throws IOException {
        try (JcaPEMWriter pemWriter = new JcaPEMWriter(writer);){
            pemWriter.writeObject(data, null);
        }
    }

    static {
        EC_SPEC_P256 = ECNamedCurveTable.getParameterSpec((String)"secp256r1");
        EC_SPEC_P384 = ECNamedCurveTable.getParameterSpec((String)"secp384r1");
        EC_SPEC_P521 = ECNamedCurveTable.getParameterSpec((String)"secp521r1");
        ENCODING = BaseEncoding.base64().omitPadding();
        HASHER = Hashing.sha256();
        log = LoggerFactory.getLogger(KeyUtils.class);
        BouncyCastleProvider provider = new BouncyCastleProvider();
        Security.addProvider((Provider)provider);
        CONVERTER = new JcaPEMKeyConverter().setProvider((Provider)provider);
        EnumSet<PosixFilePermission> rw = EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE);
        OWNER_RW = ImmutableSet.copyOf(rw);
        EnumSet<PosixFilePermission> other = EnumSet.complementOf(rw);
        TOO_OPEN = ImmutableSet.copyOf(other);
    }
}

