import React, { Fragment, useEffect } from 'react';
import Spinner from '@atlaskit/spinner';
import { I18n } from '@atlassian/wrm-react-i18n';
import SectionMessage from '@atlaskit/section-message';
import {
    AtlassianOauth2UiDialog,
    AtlassianOauth2UiDialogFormState,
} from '../AtlassianOauth2UiDialog';
import { ApplicationUrlField } from './ApplicationUrlField';
import { RedirectUriField } from './RedirectUriField';
import {
    performGetCreateIncomingLinkIntermediateDetails,
    performGetUpdateIncomingLinkIntermediateDetails,
    performPostCreateIncomingLink,
    performPutUpdateIncomingLink,
} from './createIncomingLinkDialogUtils';
import { IncomingScopeField } from './IncomingScopeField';
import wrmData from 'wrm/data';
import { getScopeFromForm } from './permissions/utils/scope-helpers';
import { LoadingDetailsErrorMessage } from '../LoadingDetailsErrorMessage';
import { dispatchApplinkDialogClosedEvent } from '../atlassianOauth2UiEvents';
import { isOauth2NotSupportedError } from './errorHelpers';

const DATA_PROVIDER_KEY =
    'com.atlassian.applinks.applinks-plugin:entrypoint-atlassian-oauth2-ui-page-init.permissions-data';

const permissionsData = wrmData.claim(DATA_PROVIDER_KEY);

export type CreateIncomingLinkDialogState = {
    isOpen?: boolean;
    applinkId?: string;
    incomingApplicationLinkDetails?: {
        rpcUrl?: string;
        readonlyRedirectUri?: boolean;
        incomingRedirectUri?: string;
        availableIncomingScopes?: string[];
        incomingScopes?: string[];
        incomingScopeDescription?: string;
        incomingScopeLabel?: string;
        incomingScopePlaceholder?: string;
    };
    needToLoadIntermediateDetails?: boolean | null;
    hasErrorLoadingIntermediateDetails?: boolean | null;
    isPostLoading?: boolean;
    hasErrorPosting?: boolean | null;
    error?: Error | null;
    correctedUrl?: string;
    enteredRedirectUriValue?: string;
    enteredIncomingScope?: boolean;
};

type CreateIncomingLinkDialogReducerAction = {
    type: string;
} & CustomEvent<
    {
        creationState?: {
            getCorrectedUrl: () => string;
        };
    } & Partial<CreateIncomingLinkDialogState>
> &
    Partial<CreateIncomingLinkDialogState>;

export const createIncomingLinkDialogInitialState: CreateIncomingLinkDialogState = {
    isOpen: false,
    incomingApplicationLinkDetails: undefined,
    needToLoadIntermediateDetails: null,
    hasErrorLoadingIntermediateDetails: null,
    isPostLoading: undefined,
    hasErrorPosting: null,
    error: null,
};

export function createIncomingLinkDialogReducer(
    substate: Partial<CreateIncomingLinkDialogState>,
    action: CreateIncomingLinkDialogReducerAction
): CreateIncomingLinkDialogState {
    switch (action.type) {
        case 'cil:open': {
            const { detail } = action;
            const { creationState, applinkId } = detail;
            return {
                ...substate,
                isOpen: true,
                needToLoadIntermediateDetails: true,
                applinkId,
                ...(creationState
                    ? {
                          correctedUrl: creationState.getCorrectedUrl(),
                      }
                    : {}),
            };
        }
        case 'cil:close': {
            dispatchApplinkDialogClosedEvent();
            return {
                ...substate,
                isOpen: false,
            };
        }
        case 'cil:intermediateGetSuccessful': {
            return {
                ...substate,
                needToLoadIntermediateDetails: false,
                incomingApplicationLinkDetails: action.incomingApplicationLinkDetails,
            };
        }
        case 'cil:intermediateGetFailed': {
            return {
                ...substate,
                needToLoadIntermediateDetails: false,
                hasErrorLoadingIntermediateDetails: true,
                error: action.error,
            };
        }
        case 'cil:postStarted': {
            return {
                ...substate,
                isPostLoading: true,
                enteredRedirectUriValue: action.enteredRedirectUriValue,
                enteredIncomingScope: action.enteredIncomingScope,
            };
        }
        case 'cil:postSuccessful': {
            return {
                ...substate,
                isPostLoading: false,
                hasErrorPosting: false,
                isOpen: false,
            };
        }
        case 'cil:postFailed': {
            return {
                ...substate,
                isPostLoading: false,
                hasErrorPosting: true,
            };
        }
    }
    return substate;
}

interface CreateIncomingLinkDialogProps {
    substate: CreateIncomingLinkDialogState;
    dispatch: (action: CreateIncomingLinkDialogReducerAction) => void;
    onClose: () => void;
}

export function CreateIncomingLinkDialog(props: CreateIncomingLinkDialogProps) {
    const { substate, dispatch } = props;

    function handleSubmit(formState: AtlassianOauth2UiDialogFormState) {
        const redirectUri = incomingApplicationLinkDetails?.readonlyRedirectUri
            ? incomingApplicationLinkDetails.incomingRedirectUri
            : formState.redirectUri;
        const selectedIncomingScopes = getScopeFromForm(permissionsData.product, formState);

        if (substate.applinkId) {
            performPutUpdateIncomingLink(
                substate.applinkId,
                redirectUri,
                selectedIncomingScopes,
                dispatch
            );
        } else {
            performPostCreateIncomingLink(substate, redirectUri, selectedIncomingScopes, dispatch);
        }
    }

    function closeModal() {
        props.onClose();
    }

    const {
        correctedUrl,
        incomingApplicationLinkDetails,
        needToLoadIntermediateDetails,
        hasErrorLoadingIntermediateDetails,
        isPostLoading,
        hasErrorPosting,
        error,
    } = substate;

    useEffect(() => {
        if (substate.needToLoadIntermediateDetails) {
            if (substate.applinkId) {
                performGetUpdateIncomingLinkIntermediateDetails(substate.applinkId, dispatch);
            } else {
                performGetCreateIncomingLinkIntermediateDetails(substate.correctedUrl, dispatch);
            }
        }
    }, [
        substate.needToLoadIntermediateDetails,
        substate.applinkId,
        substate.correctedUrl,
        dispatch,
    ]);

    const getApplicationUrl = () => {
        if (incomingApplicationLinkDetails && incomingApplicationLinkDetails.rpcUrl)
            return incomingApplicationLinkDetails.rpcUrl;
        return correctedUrl ?? '';
    };

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_CreateIncomingLinkDialog"
            onClose={closeModal}
            onSubmit={handleSubmit}
            titleText={I18n.getText('applinks-atlassian-oauth2-ui.create-incoming-link')}
            primaryButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.continue')}
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.cancel')}
            primaryButtonIsLoading={isPostLoading}
        >
            <ApplicationUrlField applicationUrl={getApplicationUrl()}></ApplicationUrlField>
            {needToLoadIntermediateDetails ? (
                <p>
                    <Spinner />
                    &nbsp;
                    {I18n.getText('applinks-atlassian-oauth2-ui.loading-link-data')}
                </p>
            ) : hasErrorLoadingIntermediateDetails ? (
                <LoadingDetailsErrorMessage
                    isOauth2NotSupportedError={isOauth2NotSupportedError(error)}
                />
            ) : (
                <Fragment>
                    {!incomingApplicationLinkDetails?.readonlyRedirectUri && (
                        <RedirectUriField substate={substate} />
                    )}
                    <IncomingScopeField substate={substate} providerData={permissionsData} />
                </Fragment>
            )}
            {hasErrorPosting ? (
                <SectionMessage
                    testId="default-incoming-applink-error"
                    appearance="error"
                    title={I18n.getText(
                        'applinks-atlassian-oauth2-ui.incoming-link-creation-failed.title'
                    )}
                >
                    <p>
                        {I18n.getText(
                            'applinks-atlassian-oauth2-ui.incoming-link-creation-failed.description'
                        )}
                    </p>
                </SectionMessage>
            ) : null}
        </AtlassianOauth2UiDialog>
    );
}
