import React, { useContext, Fragment, FC, PropsWithChildren } from 'react';
import { css } from '@compiled/react';
import CheckCircleIcon from '@atlaskit/icon/glyph/check-circle';
import WarningIcon from '@atlaskit/icon/glyph/warning';
import { isEmpty } from 'lodash';

import { DESCRIPTION_TITLE } from './bitbucket/BitbucketPermissions.i18n';
import { getSortPermissionsFunction } from './ProductScopeDescriptions';
import { PermissionsContext, ScopeDescriptions } from './PermissionsContext';
import { BasicPermission } from './permissions';
import { JSM_KB_IMPERSONATION_WARNING_I18N } from './JSM/JsmKbWarning.i18n';

const titleStyles = css({
    marginTop: '35px',
    marginBottom: '11.67px',
});
const descriptionListStyles = css({
    display: 'contents',
});
const descriptionStyles = css({
    marginLeft: '10px',
    marginBottom: '2.5px',
});
const descriptionRowStyles = css({
    display: 'flex',
    alignItems: 'flex-end',
    marginLeft: '10px',
    marginBottom: '13px',
});
const DescriptionRow: FC<PropsWithChildren> = ({ children, ...props }) => (
    <div css={descriptionRowStyles} {...props}>
        {children}
    </div>
);

const Description: FC<PropsWithChildren> = ({ children }) => (
    <span css={descriptionStyles}>{children}</span>
);

const DescriptionList: FC<PropsWithChildren> = ({ children }) => (
    <ul css={descriptionListStyles}>{children}</ul>
);

const Title: FC<PropsWithChildren> = ({ children }) => <p css={titleStyles}>{children}</p>;

function filterBasicScopeDescriptions(
    scopeDescriptions: ScopeDescriptions,
    providedScopes: string[],
    getPermissionRank: (permission: string) => number
): string[] {
    const sortFunction = getSortPermissionsFunction(getPermissionRank);
    return Object.entries(scopeDescriptions)
        .filter((entry) => providedScopes.includes(entry[0]))
        .sort((permissionA, permissionB) => sortFunction(permissionA[0], permissionB[0]))
        .reduce<string[]>((arr, current) => {
            return [...arr, current[1].title];
        }, []);
}

interface DropdownPermissionDescriptionsProps {
    scope: string | null;
    getPermissionRank: (permission: string) => number;
    handleInheritedScopes: (scopes: string[]) => string[];
}

export default function DropdownPermissionDescriptions({
    scope,
    getPermissionRank,
    handleInheritedScopes,
}: DropdownPermissionDescriptionsProps) {
    const { descriptions } = useContext(PermissionsContext);

    let flattenedDescriptions: string[];

    if (isEmpty(scope)) {
        flattenedDescriptions = [];
    } else {
        const scopes = handleInheritedScopes([scope as string]);

        flattenedDescriptions = filterBasicScopeDescriptions(
            descriptions.consent ?? {},
            scopes,
            getPermissionRank
        );
    }

    const showJsmImpersonationWarning = scope === BasicPermission.JSM_KB;

    return (
        <Fragment>
            {flattenedDescriptions.length !== 0 && <Title>{DESCRIPTION_TITLE}</Title>}
            <DescriptionList>
                {flattenedDescriptions.length > 0 &&
                    flattenedDescriptions.map((description, index) => {
                        return (
                            <DescriptionRow key={description + index}>
                                <CheckCircleIcon
                                    label={'Check-circle-icon'}
                                    primaryColor="var(--ds-icon-success)"
                                />
                                <Description>{description}</Description>
                            </DescriptionRow>
                        );
                    })}
                {showJsmImpersonationWarning && (
                    <DescriptionRow key={'jsm-impersonation-warning'}>
                        <WarningIcon label={'warning-icon'} primaryColor="var(--ds-icon-warning)" />
                        <Description>{JSM_KB_IMPERSONATION_WARNING_I18N}</Description>
                    </DescriptionRow>
                )}
            </DescriptionList>
        </Fragment>
    );
}
